# -*- coding: utf-8 -*-

import logging

import six.moves

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry


class YappyClient(object):

    LOG_PREFIX = '<YAPPY CLIENT>'

    SUCCESSFUL_RELEASE_STATUSES = (200, 204)
    BASE_YAPPY_URL = 'https://yappy.z.yandex-team.ru'
    METHOD_URLS = {
        'create_release': '/api/yappy.stand.SandboxReleaseService/create'
    }

    MAX_RETRIES = 10
    BACKOFF_FACTOR = 0.2

    def __init__(self):
        self.session = requests.Session()
        adapter = HTTPAdapter(max_retries=Retry(total=self.MAX_RETRIES, backoff_factor=self.BACKOFF_FACTOR))
        self.session.mount('https://', adapter)

    @classmethod
    def _make_url(cls, api_method):
        """
        Makes yappy api url for given method
        :param api_method: api method alias as given at METHOD_URLS property
        :type api_method: str
        :return: formatted url
        :rtype: str
        """
        return six.moves.urllib_parse.urljoin(cls.BASE_YAPPY_URL, cls.METHOD_URLS.get(api_method))

    def _post(self, api_method, data):
        """
        Do a post request to yappy api
        :param api_method: api method alias as given at METHOD_URLS property
        :type api_method: str
        :param data: request payload (to be converted to json)
        :type data: dict
        :return: yappy api response
        :rtype: Response
        """
        url = self._make_url(api_method)
        logging.info('{} Trying to POST data to {}. Data is: {}'.format(self.LOG_PREFIX, url, data))
        response = self.session.post(url, json=data)
        logging.info('{} Response status from {} was {}'.format(self.LOG_PREFIX, url, response.status_code))
        response.raise_for_status()
        return response

    def _stub(self):
        """
        Generates stub message for non-implemented response fields
        :return: stub message
        :rtype: str
        """
        return 'To be implemented'

    def create_release(self, release_data):
        """
        Sends sandbox release to yappy api
        :param release_data: info about sandbox release
        :type release_data: dict
        :return: release operation result
        :rtype: dict
        """
        response = self._post('create_release', release_data)
        result = {}
        if response.status_code not in self.SUCCESSFUL_RELEASE_STATUSES:
            result['errors'] = ['Server returns {code} after {retries} retries'.format(
                code=response.status_code, retries=self.MAX_RETRIES)]
        response_data = response.json()
        result.update({
            'release_link': self._stub(),
            'release_type': self._stub(),
            'release_id': self._stub(),
            'release_status': response.status_code,
            'original_release': release_data,
            'accepted': response.status_code in self.SUCCESSFUL_RELEASE_STATUSES,
            '_original_response_json': response_data
        })
        return result
