import tarfile
import os
import time
import logging

from sandbox import sdk2
from sandbox.sdk2.helpers.process import subprocess
from sandbox.sdk2.helpers.process import ProcessLog


class NodeManager(object):
    """
    Sandbox-CI-like manager for NodeJS environment.

    Provides:
        - install_modules
        - run_script

    You have to manually call `setup()` method before anything else.

    Example usage:
        >>> # noinspection PyUnresolvedReferences
        >>> node = NodeManager(task, registry='http://npm.yandex-team.ru')
        >>> node.setup()
        >>> node.install_modules(cwd='/path/to/sources/root')
        >>> node.run_script('test', cwd='/path/to/sources/root')
    """

    def __init__(self, task, registry=None):
        """
        :type task: sdk2.Task
        :type registry: str | String
        """
        self.task = task
        self.registry = registry

        self.installation_dir = None
        self.sources_root = None

        self.node_bin = None
        self.npm_bin = None

    def setup(self, sources_root, node_archive):
        """
        Fetches and installs NodeJS and NPM.

        :rtype: NoneType
        """
        with self.task.info_section('<setup> node & npm'):
            self.sources_root = sources_root
            self.installation_dir = str(self.task.path('node_installation_dir'))

            node_archive_path = str(sdk2.ResourceData(node_archive).path)
            with tarfile.open(node_archive_path) as a:
                a.extractall(self.installation_dir)

            self.node_bin = os.path.abspath(os.path.join(self.installation_dir, 'bin', 'node'))
            self.npm_bin = os.path.join(os.path.dirname(self.node_bin), 'npm')

            logging.info('>>> NodeManager.setup finished\n_node_bin={}\n_npm_bin={}'.format(
                self.node_bin, self.npm_bin
            ))

    def teardown(self):
        """
        Reserved for future uses.
        """

    def install_modules(self, modules=None, cwd=None, env=None):
        """
        Runs "npm install".
        If :param:`modules` is passed, only selected modules will be installed.
        :returns process return code, stdout and stderr.

        :type modules: tuple | list
        :type cwd: str
        :type env: dict
        :rtype: Tuple[int, str, str]
        """
        modules = modules if modules is not None else ()

        install_command = [self.npm_bin, 'install', '--verbose', '--registry', self.registry]
        install_command.extend(modules)
        return self._call_node_bin(install_command, cwd=cwd, env=env)

    def run_script(self, script_name, script_options=None, cwd=None, env=None):
        """
        Runs "npm run :param:`script`".
        :returns process return code, stdout and stderr.

        :type script_name: str
        :type script_options: tuple | list
        :type cwd: str
        :type env: dict
        :rtype: Tuple[int, str, str]
        """
        script_command = [self.npm_bin, 'run', script_name]
        if script_options:
            script_command.extend(script_options)

        return self._call_node_bin(script_command, cwd=cwd, env=env)

    def run_npm_command(self, command, cwd=None, env=None):
        script_command = [self.npm_bin]
        script_command.extend(command)
        return self._call_node_bin(script_command, cwd=cwd, env=env)

    def _update_env(self, env=None):
        """
        Sets correct PATH environment variable.

        :type env: dict
        :rtype: dict
        """
        env = env if env is not None else {}

        result = os.environ.copy()
        result.update(env)
        result['PATH'] = '{}:{}'.format(os.path.dirname(self.node_bin), result['PATH'])

        return result

    def _call_node_bin(self, command, cwd=None, env=None):
        """
        Runs given :param:`command` with NodeJS binary.

        :type command: tuple | list
        :type cwd: str
        :type env: dict
        :rtype: Tuple[int, str, str]
        """
        log_name = 'call_node_binary_{}'.format(time.time())

        env = self._update_env(env)
        final_command = [self.node_bin]
        final_command.extend(command)

        with self.task.info_section('<node> {}'.format(' '.join(command))):
            with ProcessLog(self.task, logger=log_name) as pl:
                logging.info('>>> call_node_binary: {}\n\nCWD:\n{}\n\nENV:\n{}\n\nLOG:\n{}\n\n'.format(
                    command, cwd, env, log_name
                ))

                process = subprocess.Popen(final_command, stdout=pl.stdout, stderr=pl.stderr, cwd=cwd, env=env)
                out, err = process.communicate()
                out = out.decode() if out is not None else ''
                err = err.decode() if err is not None else ''

                if process.returncode != 0:
                    raise RuntimeError('call_node_binary failed with code {}\n\nOUT:\n{}\n\nERR:\n{}\n\n'.format(
                        process.returncode, out, err
                    ))

                return process.returncode, out, err
