# coding: utf-8
import os
import logging
import tarfile
import time
import platform

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types

from sandbox.sandboxsdk import ssh

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.environments import GCCEnvironment
from sandbox.projects.common.environments import PipVirtualenvEnvironment
from sandbox.projects.common.environments import SandboxPythonDevEnvironment
from sandbox.projects.common.environments import SandboxLibSnappyEnvironment
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import remove_path
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.paths import which
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.utils import get_bsconfig


class GitTagParameter(SandboxStringParameter):
    name = 'tag'
    description = 'Git branch/tag'
    default_value = ''
    required = True


class DoNotCreateShardParameter(SandboxBoolParameter):
    name = 'do_not_create_shard'
    description = 'Do Not Create Shard'
    default_value = False


class BaseBuildYasmTask(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'BASE_BUILD_GOLOVAN_TASK'

    environment = [GCCEnvironment(), PipVirtualenvEnvironment(), SandboxPythonDevEnvironment(), SandboxLibSnappyEnvironment(), ]

    input_parameters = [GitTagParameter, DoNotCreateShardParameter]

    execution_space = 3000
    client_tags = ctc.Tag.LINUX_PRECISE

    SERVICE_NAME = 'redefine_me'  # e.g. 'yasmserver'
    TGZ_PATH = 'redefine_me'  # e.g. '{}.tar.gz'.format(SERVICE_NAME)
    RESOURCE_TYPE = resource_types.OTHER_RESOURCE
    SHARD_TYPE = resource_types.OTHER_RESOURCE
    COPY_DIRS = {}
    GIT_URL = 'ssh://git@bb.yandex-team.ru/search_infra/yasm.git'
    VAULT_OWNER = "YASM"
    SSH_PRIVATE_KEY_VAULT_NAME = "robot-yasm-golovan-ssh-private-key"
    CHECKOUT_PATH = 'yasm'
    REMOVE_DIRS = [CHECKOUT_PATH]
    VENV_NAME = 'python'
    BUILD_DIR = 'build_dir'
    VERSION_FILE_PATH = ''
    PLATFORM_INDEPENDENT = False
    NEED_YASMCONF = True
    VERSION = ''

    def _checkout(self):
        logging.info('Checkout code...')
        tag = self.ctx.get('tag')
        assert tag, 'Trying to fetch project from git, but no tag specified'
        with ssh.Key(self, self.VAULT_OWNER, self.SSH_PRIVATE_KEY_VAULT_NAME):
            run_process(
                ['git', 'clone', self.GIT_URL, self.CHECKOUT_PATH],
                log_prefix='git_clone',
                shell=True,
            )
            run_process(
                ['git', 'checkout', tag],
                work_dir=self.CHECKOUT_PATH,
                log_prefix='git_checkout',
                shell=True)

    def _assemble_bundle(self, copy_dirs, remove_dirs):
        logging.info('Assemble bundle...')
        for copy_dir in copy_dirs.keys():
            copy_path(os.path.join(self.CHECKOUT_PATH, copy_dir), os.path.join(self.BUILD_DIR, copy_dirs[copy_dir]))
        for remove_dir in remove_dirs:
            remove_path(remove_dir)

    def _install_requirements(self, virtualenv_path):
        logging.info('Installing requirements in to the virtualenv...')
        logging.info("ENV VARIABLES: %s" % os.environ)
        pip_path = os.path.join(virtualenv_path, 'bin', 'pip')
        # install service dependencies into virtualenv
        run_process([pip_path, 'install',
                     '-i', 'https://pypi.yandex-team.ru/simple/',
                     '-r', 'requirements.txt'],
                    log_prefix='install_requirements',
                    work_dir=self.CHECKOUT_PATH)

    def _make_resource(self, path, version):
        logging.info('Creating tgz file...')
        with tarfile.open(self.TGZ_PATH, 'w:gz') as tar:
            for entry in os.listdir(path):
                tar.add(os.path.join(path, entry), entry)
        if self.PLATFORM_INDEPENDENT:
            attributes = {'version': version}
            arch = 'any'
        else:
            attributes = {'version': version, 'platform': platform.platform()}
            arch = 'linux'
        self.create_resource(
            description='{} tarball version {} from branch/tag {}'.format(self.SERVICE_NAME, version, self.ctx['tag']),
            resource_path=self.TGZ_PATH,
            resource_type=self.RESOURCE_TYPE,
            attributes=attributes,
            arch=arch
        )

    def _create_shard(self, version):
        logging.info('Creating shard dir and sharing it...')
        shard_name = '{0}-{1}-{2}'.format(self.SERVICE_NAME, '000', version)
        self.ctx['shard_name'] = shard_name
        shard_path = self.path(shard_name)
        remove_path(shard_path)
        make_folder(shard_path)
        copy_path(self.TGZ_PATH, shard_name)
        # pre initialize shard description file with install procedure
        with open(os.path.join(shard_name, 'shard.conf'), 'w') as f:
            f.write(
                '%install\n'
                'tar -xzf {}\n'.format(self.TGZ_PATH)
            )
        # do initialize shard
        run_process(
            ['perl', get_bsconfig(), 'shard_init', '--torrent', shard_name],
            log_prefix="creating_shard",
            work_dir=self.path()
        )
        # in order to get same rbtorrent both for shard and resource we remove
        # some leftovers which are not shared by bsconfig
        # for i in ['_lock.tmp', 'shard.state']:
        #    os.remove(os.path.join(shard_path, i))
        # now we can create resource
        if self.PLATFORM_INDEPENDENT:
            attributes = {'version': version}
            arch = 'any'
        else:
            attributes = {'version': version, 'platform': platform.platform()}
            arch = 'linux'
        self.create_resource(
            description='{} {} shard'.format(self.SERVICE_NAME, self.ctx['tag']),
            resource_path=shard_path,
            resource_type=self.SHARD_TYPE,
            attributes=attributes,
            arch=arch
        )

    def _set_service_version(self, version_file_path):
        logging.info('Set version...')
        version = time.strftime("%Y%m%d%H%M")
        run_process(['sed', '-i',
                     '"s/DEVELOP/%s/"' % version, version_file_path],
                    shell=True,
                    work_dir=self.CHECKOUT_PATH)
        return version

    def _prepare_environment(self):
        logging.info('Preparing environment (gcc, g++)...')
        compilers = self.environment[0].get_compilers()
        os.environ['CC'] = gcc_binary = compilers['CC']
        os.environ['CXX'] = gxx_binary = compilers['CXX']
        logging.info('Fetched compilers are: CC=%s CXX=%s', gcc_binary, gxx_binary)
        make_folder(self.path('custom_bin'))
        os.symlink(gcc_binary, os.path.join(self.path('custom_bin'), 'gcc'))
        os.symlink(gxx_binary, os.path.join(self.path('custom_bin'), 'g++'))
        os.environ['PATH'] = self.path('custom_bin') + ':' + os.environ['PATH']
        logging.info(os.environ)

    def _prepare_virtualenv(self):
        logging.info('Creating virtualenv...')
        virtualenv_path = os.path.join(self.path(self.BUILD_DIR), self.VENV_NAME)
        # create virtualenv using *system* python and local virtualenv script
        virtualenv_script = which('virtualenv')
        run_process(['python', virtualenv_script, virtualenv_path], log_prefix='virtualenv_creating')
        self._install_requirements(virtualenv_path)
        run_process(['python', virtualenv_script, '--relocatable', virtualenv_path], log_prefix='making_virtualenv_relocatable')

    def _generate_yasmconf(self):
        generator_path = os.path.join(self.path(self.CHECKOUT_PATH), "scripts", "conf_scripts", "makeyasmconf2.py")

        run_process([generator_path,
                     '--yasmagent-dir=%s' % self.path(self.CHECKOUT_PATH),
                     '--yasmutil-dir=%s' % self.path(self.CHECKOUT_PATH),
                     '--conf-dir=%s' % os.path.join(self.path(self.CHECKOUT_PATH), 'CONF'),
                     '--out=%s' % os.path.join(self.path(self.BUILD_DIR), 'yasm.conf'),
                     '--log-level=INFO'],
                    log_prefix='build_config',
                    work_dir=self.CHECKOUT_PATH)

    def arcadia_info(self):
        """
        Hacky way to allow this task to be released: provide tag, other fields are not checked.
        """
        return None, self.ctx.get('tag'), None

    def on_execute(self):
        """
        Plan is:
        * set CC and CXX to our gcc
        * git clone and checkout specified branch/tag
        * create virtualenv
        * set version
        * install requirements
        * make virtualenv relocatable
        * assemble bundle with only needed files
        * create tarball with bundle
        * create shard
        """
        self._prepare_environment()

        self._checkout()

        make_folder(self.path(self.BUILD_DIR))

        self._prepare_virtualenv()

        self.VERSION = self._set_service_version(self.VERSION_FILE_PATH)

        if self.NEED_YASMCONF:
            self._generate_yasmconf()

        self._assemble_bundle(copy_dirs=self.COPY_DIRS, remove_dirs=self.REMOVE_DIRS)

        self._make_resource(self.path(self.BUILD_DIR), self.VERSION)

        if not self.ctx.get('do_not_create_shard'):
            self._create_shard(self.VERSION)
