from __future__ import absolute_import

import logging
import os
from sandbox import sdk2
from collections import namedtuple
from sandbox.projects.common.gnupg import GpgKey2
from sandbox.sdk2.helpers import subprocess
from sandbox.sandboxsdk.process import run_process

from shutil import copyfile


GpgSettings = namedtuple(
    'GpgSettings',
    ['vault_secrets_owner', 'secret_key_name', 'public_key_name']
)


class Environments(object):
    DEV = 'development'
    PROD = 'production'
    PRESTABLE = 'prestable'
    TEST = 'testing'


class EncryptMixin(object):

    def encrypt(self, path, settings):
        """
        :type path: str or unicode
        :type settings: GpgSettings
        """
        base_name = os.path.basename(path)

        with GpgKey2(settings.vault_secrets_owner, settings.secret_key_name,
                     settings.public_key_name):
            with sdk2.helpers.ProcessLog(
                    self, logging.getLogger('gpg-zip_log')
            ) as pl:
                output_path = '.'.join([base_name, 'gpg'])
                process = subprocess.Popen(
                    [
                        'gpg-zip',
                        '-e',
                        '--recipient', settings.vault_secrets_owner,
                        '--output', output_path,
                        '--gpg-args', '--no-tty',
                        '--gpg-args', '--batch',
                        '--gpg-args', '--trust-model',
                        '--gpg-args', 'always',
                        base_name,
                    ],
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                )
                exitcode = process.wait()
                if exitcode:
                    pl.logger.error(
                        'Failed to encrypt %s, exitcode %s',
                        path,
                        exitcode,
                        exc_info=True,
                    )
                    raise Exception('Encryption failed')

        return output_path

    def decrypt(self, path, settings):
        """
        :type path: str or unicode
        :type settings: GpgSettings
        """
        # waiting path.gpg
        gpg_name = os.path.basename(path)
        if os.path.abspath(path) != os.path.join(os.getcwd(), gpg_name):
            copyfile(path, gpg_name)
        result_path, gpg_ending = os.path.splitext(gpg_name)

        with GpgKey2(settings.vault_secrets_owner, settings.secret_key_name,
                     settings.public_key_name):
            with sdk2.helpers.ProcessLog(
                    self, logging.getLogger('gpg-zip_log')
            ) as pl:
                process = subprocess.Popen(
                    [
                        "gpg-zip",
                        "-d",
                        "--recipient", settings.vault_secrets_owner,
                        "--gpg-args", "--no-tty",
                        "--gpg-args", "--batch",
                        "--gpg-args", "--trust-model",
                        "--gpg-args", "always",
                        gpg_name,
                    ],
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                )
                exitcode = process.wait()
                if exitcode:
                    pl.logger.error(
                        'Failed to decrypt %s, exitcode %s',
                        path,
                        exitcode,
                        exc_info=True,
                    )
                    raise Exception('Decryption failed')

        return result_path


def prepare_certs():
    run_process(
        ["mkdir", "-p", "/home/sandbox/.postgresql"],
        work_dir=".",
        log_prefix="mkdir"
    )
    run_process(
        [
            "curl",
            "https://crls.yandex.net/allCAs.pem",
            "-o",
            "/home/sandbox/.postgresql/root.crt"
        ],
        work_dir=".",
        log_prefix="curl"
    )
