# -*- coding: utf-8 -*-

from sandbox import common
import sandbox.common.types.misc as ctm
import logging
import os
from sandbox import sdk2
import shutil
from sandbox.sdk2 import parameters
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.common.yc.resources import PostgresqlDump
from sandbox.projects.common.yc.common.helpers import EncryptMixin, GpgSettings, Environments, prepare_certs


class PgDumpUsingRamDrive(sdk2.Task, EncryptMixin):
    """ Run pg_dump using ram drive as temporary storage and gpg encryption"""

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group("PgDump") as pg_dump_block:
            dbname = parameters.String('Database', required=True)
            host = parameters.String('Host', required=True)
            port = parameters.Integer(
                'Port', required=True, default=6432
            )
            username = parameters.String('Username', required=True)
            password = parameters.String(
                'Password', required=True, description='Vault Secret Name'
            )
            output_format = parameters.String(
                'Dump format',
                choices=[
                    ('plain', 'plain'),
                    ('custom', 'custom'),
                    ('directory', 'directory'),
                    ('tar', 'tar'),
                ],
                required=True, default='directory'
            )
            optional_parameters = parameters.List(
                'Optional parameters',
                default=[
                    '--no-privileges',
                    '--no-subscriptions',
                    '--clean',
                    '--if-exists',
                ]
            )
        with sdk2.parameters.Group("Task") as sandbox_block:
            abc_service = parameters.String(
                'ABC Service', required=True,
                description='Your service name from https://abc.yandex-team.ru'
            )
            ramdrive_size = parameters.Integer('Ram drive size(GiB)', default=4)
            env = parameters.String(
                'Environment',
                choices=[
                    (Environments.DEV, Environments.DEV),
                    (Environments.TEST, Environments.TEST),
                    (Environments.PRESTABLE, Environments.PRESTABLE),
                    (Environments.PROD, Environments.PROD),
                ],
                required=True, default=Environments.PROD
            )
        with sdk2.parameters.Group("Encryption") as encryption_block:
            vault_secrets_owner = parameters.String(
                'Vault secrets owner',
                required=True
            )
            vault_pgp_public_key = parameters.String(
                'Public key',
                required=True, description='Vault Secret Name'
            )
            vault_pgp_private_key = parameters.String(
                'Private key',
                required=True, description='Vault Secret Name'
            )

    def on_enqueue(self):
        sdk2.Task.on_enqueue(self)
        if self.Parameters.ramdrive_size:
            self.Requirements.ramdrive = ctm.RamDrive(
                ctm.RamDriveType.TMPFS,
                int(self.Parameters.ramdrive_size) << 10,
                None
            )

    def setup_ramdrive(self):
        if self.ramdrive:
            logging.info(
                'Setup RamDrive size: %s path: %s',
                common.utils.size2str(self.ramdrive.size << 20),
                self.ramdrive.path,
            )
            os.chdir(str(self.ramdrive.path))

    def dump_database(self):
        dump_path = os.path.join(os.getcwd(), '{}_dump'.format(
            self.Parameters.dbname))
        with sdk2.helpers.ProcessLog(
                self, logging.getLogger('pg_dump')
        ) as pl:
            psql_password = sdk2.Vault.data(self.Parameters.password)
            args = [
                'pg_dump',
                '--dbname={}'.format(self.Parameters.dbname),
                '--host={}'.format(self.Parameters.host),
                '--port={}'.format(self.Parameters.port),
                '--username={}'.format(self.Parameters.username),
                '--password',
                '--format={}'.format(self.Parameters.output_format),
                '--file={}'.format(dump_path)
            ]
            args.extend(self.Parameters.optional_parameters)
            env = {'PGSSLMODE': 'verify-full'}
            dump = subprocess.Popen(
                args,
                env=env,
                stdin=subprocess.PIPE,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            dump.communicate(psql_password)
            exitcode = dump.wait()
        if not os.path.exists(dump_path):
            raise Exception('Dump failed with exitcode %s', exitcode)
        return dump_path

    def on_execute(self):
        prepare_certs()
        task_path = os.getcwd()
        settings = GpgSettings(
            vault_secrets_owner=self.Parameters.vault_secrets_owner,
            secret_key_name=self.Parameters.vault_pgp_private_key,
            public_key_name=self.Parameters.vault_pgp_public_key,
        )
        self.setup_ramdrive()
        dump_path = self.dump_database()
        encrypted_dump = self.encrypt(dump_path, settings)

        dump_ramdrive_path = os.path.join(os.getcwd(), encrypted_dump)
        os.chdir(task_path)
        if dump_ramdrive_path != os.path.join(os.getcwd(), encrypted_dump):
            shutil.copyfile(dump_ramdrive_path, encrypted_dump)

        resource_metadata = PostgresqlDump(
            self,
            "{}/{}".format(
                self.Parameters.abc_service,
                self.Parameters.dbname,
            ),
            encrypted_dump,
            env=self.Parameters.env,
            abc_service=self.Parameters.abc_service,
            database=self.Parameters.dbname,
        )
        resource_data = sdk2.ResourceData(resource_metadata)
        resource_data.ready()
