# -*- coding: utf-8 -*-

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
import logging
import os
from sandbox import sdk2
from sandbox.sdk2 import parameters
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.common.yc.resources import PostgresqlDump
from sandbox.projects.common.yc.common.helpers import EncryptMixin, GpgSettings, Environments, prepare_certs


class PgRestoreUsingRamDrive(sdk2.Task, EncryptMixin):
    """ Run pg_restore using ram drive as temporary storage and gpg encryption"""

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group("PgRestore") as pg_restore_block:
            dbname = parameters.String('Database', required=True)
            host = parameters.String('Host', required=True)
            port = parameters.Integer(
                'Port', required=True, default=6432
            )
            username = parameters.String('Username', required=True)
            password = parameters.String(
                'Password', required=True, description='Vault Secret Name'
            )
            output_format = parameters.String(
                'Dump format',
                choices=[
                    ('plain', 'plain'),
                    ('custom', 'custom'),
                    ('directory', 'directory'),
                    ('tar', 'tar'),
                ],
                required=True, default='directory'
            )
            optional_parameters = parameters.List(
                'Optional parameters',
                default=[
                    '--no-privileges',
                    '--single-transaction',
                    '--clean',
                ]
            )
        with sdk2.parameters.Group("Task") as sandbox_block:
            abc_service = parameters.String(
                'ABC Service', required=True
            )
            ramdrive_size = parameters.Integer('Ram drive size(GiB)', default=4)
            env = parameters.String(
                'Dump Environment',
                choices=[
                    (Environments.DEV, Environments.DEV),
                    (Environments.TEST, Environments.TEST),
                    (Environments.PRESTABLE, Environments.PRESTABLE),
                    (Environments.PROD, Environments.PROD),
                ],
                required=True, default=Environments.PROD
            )
        with sdk2.parameters.Group("Encryption") as encryption_block:
            vault_secrets_owner = parameters.String(
                'Vault secrets owner',
                required=True
            )
            vault_pgp_public_key = parameters.String(
                'Public key',
                required=True, description='Vault Secret Name'
            )
            vault_pgp_private_key = parameters.String(
                'Private key',
                required=True, description='Vault Secret Name'
            )

    def on_enqueue(self):
        sdk2.Task.on_enqueue(self)
        if self.Parameters.ramdrive_size:
            self.Requirements.ramdrive = ctm.RamDrive(
                ctm.RamDriveType.TMPFS,
                int(self.Parameters.ramdrive_size) << 10,
                None
            )

    def setup_ramdrive(self):
        if self.ramdrive:
            logging.info(
                'Setup RamDrive size: %s path: %s',
                common.utils.size2str(self.ramdrive.size << 20),
                self.ramdrive.path,
            )
            os.chdir(str(self.ramdrive.path))

    def get_dump_path(self):
        dump_resource = sdk2.Resource.find(
            PostgresqlDump,
            attrs={'env': self.Parameters.env,
                   'abc_service': self.Parameters.abc_service,
                   'database': self.Parameters.dbname
                   },
            state=ctr.State.READY
        ).order(
            -PostgresqlDump.id,
        ).first()
        result = str(sdk2.ResourceData(dump_resource).path)
        logging.info('Using dump resource: %s', result)
        settings = GpgSettings(
            vault_secrets_owner=self.Parameters.vault_secrets_owner,
            secret_key_name=self.Parameters.vault_pgp_private_key,
            public_key_name=self.Parameters.vault_pgp_public_key,
        )
        return self.decrypt(result, settings)

    def restore_database(self, dump_path):
        with sdk2.helpers.ProcessLog(
                self, logging.getLogger('pg_restore')
        ) as pl:
            psql_password = sdk2.Vault.data(self.Parameters.password)
            args = [
                "pg_restore",
                '--dbname={}'.format(self.Parameters.dbname),
                '--host={}'.format(self.Parameters.host),
                '--port={}'.format(self.Parameters.port),
                '--username={}'.format(self.Parameters.username),
                '--password',
                '--format={}'.format(self.Parameters.output_format),
                dump_path
            ]
            args.extend(self.Parameters.optional_parameters)
            env = {'PGSSLMODE': 'verify-full'}
            restore = subprocess.Popen(
                args,
                env=env,
                stdin=subprocess.PIPE,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            restore.communicate(psql_password)
            exitcode = restore.wait()
        if exitcode:
            raise Exception(
                'Failed to restore database, exitcode: {}'.format(
                    exitcode
                )
            )
        logging.info("Restore was ended with exit code {}".format(exitcode))

    def on_execute(self):
        prepare_certs()
        self.setup_ramdrive()
        dump_path = self.get_dump_path()
        self.restore_database(dump_path)
