import logging

from sandbox.common.errors import TaskError


def run_query(query_id, query, yql_token, db, wait=True, syntax_version=1):
    """
    Run YQL query without results
    before using this function include yql packet in environment (see https://wiki.yandex-team.ru/sandbox/cookbook/#python-libs-task-code)

    :param query_id: query identifier
    :param query: YQL query
    :param yql_token: your yql auth token
    :param db: name of yt cluster e.g. 'hahn'
    :param wait: whether you need to wait for the request to finish
    :param syntax_version: syntax_version of yql query
    :return: None if wait == True else request that was running.
    """
    logging.debug('YQL query with id %s:\n%s', query_id, query)
    from yql.api.v1.client import YqlClient
    yql_client = YqlClient(
        db=db,
        token=yql_token,
    )
    request = yql_client.query(query, syntax_version=syntax_version)
    request.run()
    if wait:
        request.get_results(wait=wait)
        logging.debug('request = {}'.format(request))
        if not request.is_success:
            raise TaskError('YQL query with id {} query failed'.format(query_id))
        logging.info('YQL query with id %s finished successful', query_id)
    else:
        return request


def run_batch(batch, yql_token, db, syntax_version=1):
    """
    Run batch of YQL queries without results
    before using this function include yql packet in environment (see https://wiki.yandex-team.ru/sandbox/cookbook/#python-libs-task-code)

    :param batch: list of tuples (query_id, query) params of run_query:
    :param yql_token: your yql auth token
    :param db: name of yt cluster e.g. 'hahn'
    :param syntax_version: syntax_version of yql query
    """
    requests = [(query_id, run_query(query_id, query, yql_token, db, False, syntax_version)) for (query_id, query) in batch]
    batch_size = len(requests)
    successful = 0
    for query_id, request in requests:
        request.get_results(wait=True)
        if not request.is_success:
            logging.info('YQL query with id %s query failed', query_id)
        else:
            successful += 1
            logging.info('YQL query with id %s finished successful', query_id)

    logging.info('batch is finished! %i / %i successful queries', successful, batch_size)
    if successful < batch_size:
        raise TaskError('{successful} / {batch_size} successful queries'.format(successful=successful, batch_size=batch_size))
