import os
import logging
import subprocess
import sys
import re
import time
import shutil
import textwrap

from distutils.version import LooseVersion

from sandbox.common.errors import SandboxException

from sandbox.sandboxsdk import network
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import make_folder

MEM_PER_PROCESS = 10 << 30
CPU_PER_PROCESS = 1
MIN_HOSTS = 3
MAX_HOSTS = 20

REQUIRED_TMPFS_SIZE = 6 << 30


class YTPackage(object):
    def __init__(self, tar_path, root_dir, use_tar_python=True):
        self._root_dir = os.path.abspath(root_dir)
        self._yt_root_dir = os.path.join(self._root_dir, "yt")
        self._tar_path = tar_path
        self._use_tar_python = use_tar_python

    def install(self):
        make_folder(self._yt_root_dir, delete_content=True)

        run_process(
            ['tar', 'xf', self._tar_path, '--overwrite', '-C', self._yt_root_dir, '--no-overwrite-dir'],
            shell=False, check=True, wait=True
        )
        sys.path[:0] = self.get_pythonpaths()
        os.environ['PATH'] = ':'.join(self.get_paths() + [os.environ.get('PATH', '')])
        if self._use_tar_python:
            os.environ['PYTHONPATH'] = ':'.join(self.get_pythonpaths() + [os.environ.get('PYTHONPATH', '')])

    def get_pythonpaths(self):
        return [os.path.join(self._yt_root_dir, 'python')]

    def get_paths(self):
        return [os.path.join(self._yt_root_dir, 'bin'), os.path.join(self._yt_root_dir, 'node', 'bin')]

    def path_mr_client(self):
        return self.path_to_bin("mapreduce-yt")

    def path_to_bin(self, bname):
        return os.path.join(self._yt_root_dir, 'bin', bname)

    def get_yt_root_dir(self):
        return self._yt_root_dir

    def get_root_dir(self):
        return self._root_dir


class YTRunner(object):
    def __init__(self, yt_package, client_info, log_dir, **kwargs):
        self._is_running = False
        self._root_dir = yt_package.get_root_dir()
        self._yt_root_dir = yt_package.get_yt_root_dir()
        self._log_dir = log_dir
        self._client_info = client_info
        self._calc_usage_parameters()
        self._custom_configs = dict()
        self._debug_logging = kwargs.get("debug_logging", False)
        self._yt_testable = kwargs.get("yt_testable", False)
        self._tmpfs = kwargs.get("tmpfs", None)
        self._forbid_chunk_storage_in_tmpfs = kwargs.get("forbid_chunk_storage_in_tmpfs", False)
        self._node_chunk_store_quota = kwargs.get("node_chunk_store_quota", None)
        self._yt_local_version = kwargs.get("yt_local_version", "0")
        self._yt_local_process = None

    def _calc_usage_parameters(self):
        ncpu = int(self._client_info['ncpu'])
        physmem = int(self._client_info['physmem']) - (4 << 30) - REQUIRED_TMPFS_SIZE
        nproc_by_mem = int(float(physmem) / MEM_PER_PROCESS)
        nproc_by_cpu = int(float(ncpu) / CPU_PER_PROCESS)
        self._nproc = min(MAX_HOSTS, max(MIN_HOSTS, min(nproc_by_mem, nproc_by_cpu - 2)))
        self._mem_per_process = int(float(physmem) / self._nproc)

    def start(self):
        self._create_custom_configs()
        self._run_server()

    def _get_environ(self):
        return {
            'NODE_PATH': os.path.join(self._yt_root_dir, 'node_modules'),
            'PYTHONPATH': os.environ.get('PYTHONPATH', ''),
            'PATH': os.environ.get('PATH', ''),
            'YT_LOCAL_THOR_PATH': os.path.join(self._yt_root_dir, 'yt-thor'),
            'YT_LOCAL_ROOT_PATH': self._yt_root_dir
        }

    def with_yt_local(self, args, **kwargs):
        outkwargs = {
            'shell': False,
            'check': True,
            'wait': True,
            'log_prefix': 'yt_local',
        }
        outkwargs.update(kwargs)
        return run_process(
            ['/skynet/python/bin/python', os.path.join(self._yt_root_dir, 'bin/yt_local')] + args,
            environment=self._get_environ(), **outkwargs
        )

    def _create_custom_configs(self):
        name = os.path.join(self._yt_root_dir, "scheduler.custom.cfg")
        with open(name, "w") as out:
            out.write(textwrap.dedent("""
                {scheduler={
                   sort_operation_options = {
                       spec_template = {
                           sort_job_io = {
                               table_writer = {
                                   max_key_weight = 262144;
                               }
                           };
                           merge_job_io = {
                               table_writer = {
                                   max_key_weight = 262144;
                               }
                           }
                       }
                   };
                   operation_options = {
                       spec_template = {
                           map_job_io = {
                               table_writer = {
                                   max_key_weight = 262144;
                               }
                           };
                           reduce_job_io = {
                               table_writer = {
                                   max_key_weight = 262144;
                               }
                           };
                           sort_job_io = {
                               table_writer = {
                                   max_key_weight = 262144;
                               }
                           };
                           job_io = {
                               table_writer = {
                                   max_key_weight = 262144;
                               }
                           }
                       }
                   };
                   fair_share_preemption_timeout=1000000000;
                   fair_share_preemption_timeout_limit=1000000000;
                   max_running_operation_count_per_pool=30;
                   enable_tmpfs=%false;
                }}"""))
        self._custom_configs["scheduler"] = name
        name = os.path.join(self._yt_root_dir, "node.custom.cfg")
        with open(name, "w") as out:
            out.write(textwrap.dedent("""
                {exec_agent={
                   job_controller={
                       resource_limits={
                           user_slots=%d;
                           cpu=%d;
                       }
                   }
                }}""" % (self._nproc, self._nproc)
            ))
        self._custom_configs["node"] = name

        name = os.path.join(self._yt_root_dir, "proxy.custom.cfg")
        with open(name, "w") as out:
            out.write(
                '{"thread_limit":256}'
            )
        self._custom_configs["proxy"] = name

        name = os.path.join(self._yt_root_dir, "controller_agent.custom.cfg")
        with open(name, "w") as out:
            out.write("""
                {controller_agent={
                   sort_operation_options = {
                       spec_template = {
                           sort_job_io = {
                               table_writer = {
                                   max_key_weight = 262144;
                               }
                           };
                           merge_job_io = {
                               table_writer = {
                                   max_key_weight = 262144;
                               }
                           }
                       }
                   };
                   operation_options = {
                       spec_template = {
                           map_job_io = {
                               table_writer = {
                                   max_key_weight = 262144;
                               }
                           };
                           reduce_job_io = {
                               table_writer = {
                                   max_key_weight = 262144;
                               }
                           };
                           sort_job_io = {
                               table_writer = {
                                   max_key_weight = 262144;
                               }
                           };
                           job_io = {
                               table_writer = {
                                   max_key_weight = 262144;
                               }
                           }
                       }
                   }
                }}""")
        # The version is not precise:
        # --controller-agent-config could have been introduced earlier,
        # but certainly after 0.0.63-0
        if LooseVersion(self._yt_local_version) >= LooseVersion("0.0.71-0"):
            self._custom_configs["controller-agent"] = name

    def _wait_cluster_started(self, max_wait=90):
        name = os.path.join(self._yt_root_dir, "test_instance/started")
        logging.info("Waiting for local yt to start (checking {})".format(name))
        stop_at = time.time() + max_wait
        while not os.path.exists(name) and not self.master_failure():
            if time.time() > stop_at:
                break
            time.sleep(5)
            logging.info("Still not started")
        if not os.path.exists(name):
            raise SandboxException("Local yt instance is not started...")
        logging.info("Local yt to started")

    def _run_server(self):
        run_process('free', shell=True, wait=True, check=False, log_prefix='free_mem')
        self.with_yt_local(['start', '--help'], log_prefix='yt_local_help')
        self._port_range_start = self._find_port_range(25000, 30000, 10) or 35000

        args = [
            'start',
            '--id', 'test_instance',
            '--node-count', str(1),
            '--operations-memory-limit', str(self._mem_per_process * self._nproc),
            '--port-range-start', str(self._port_range_start),
            '--sync'
        ]
        for key, file_name in self._custom_configs.items():
            args.append('--{}-config'.format(key))
            args.append(file_name)
        if self._debug_logging:
            args.append('--enable-debug-logging')
        if self._tmpfs:
            args += ['--tmpfs-path', self._tmpfs]
        if self._forbid_chunk_storage_in_tmpfs:
            args.append('--forbid-chunk-storage-in-tmpfs')
        if self._node_chunk_store_quota:
            args += ['--node-chunk-store-quota', str(self._node_chunk_store_quota)]

        self._yt_local_process = self.with_yt_local(args, check=False, wait=False)
        self._wait_cluster_started()

        getproxy = self.with_yt_local(['get_proxy', 'test_instance'], stdout=subprocess.PIPE, outputs_to_one_file=False)
        (ps, _) = getproxy.communicate()
        self._proxy_string = ps.strip()
        self._is_running = True

    def teardown_server(self):
        if self._is_running:
            logging.info("Waiting for local yt to stop")
            self.with_yt_local(['stop', 'test_instance'], check=False)
            while self._yt_local_process.poll() is None:
                time.sleep(10)
            # just for logs, yt_local dies with EC=1 on yt_local stop
            self.master_failure()
            self._yt_local_process = None
        logging.info("Called teardown")
        self.save_yt_logs(self._root_dir, self._log_dir)

    @staticmethod
    def save_yt_logs(root_dir, log_dir):
        instance_root = os.path.join(root_dir, 'yt/test_instance')
        logging.info("Will save logs from {}".format(instance_root))
        for (path, dirs, files) in os.walk(instance_root):
            if 'chunk_store' in path or 'changelogs' in path:
                continue
            for f in files:
                ok = False
                if f.startswith("stderr."):
                    ok = True
                if f.endswith('.log') and not re.match(r'[0-9]+[.]log', f):
                    ok = True
                if not ok:
                    continue

                dname = os.path.relpath(path, instance_root)
                dest = os.path.join(log_dir, dname)
                if not os.path.exists(dest):
                    make_folder(dest)
                shutil.copyfile(os.path.join(path, f), os.path.join(dest, f))

    def get_proxy_string(self):
        return self._proxy_string

    def master_failure(self):
        logging.info("Checking for yt master_failure...")
        if self._yt_local_process is None:
            logging.info("OK, Not started")
            return None

        ret_code = self._yt_local_process.poll()
        if ret_code is None:
            logging.info("OK, still running")
            return None

        if ret_code == 0:
            logging.info("OK, exited normally")
            return None

        logging.info("FAIL, died with code {}".format(ret_code))

        return "yt_local died with code {}, see logs".format(ret_code)

    def _find_port_range(self, start, end, required_count):
        last_bad = start - 1
        cur = start
        while cur < end:
            if not network.is_port_free(cur):
                cur += required_count
                last_bad = cur - 1
            if cur - last_bad >= required_count:
                return last_bad + 1
            cur += 1
        return None
