import json
import logging

from sandbox.common import errors

from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import requests_wrapper


class Z2Client(object):

    def __init__(self, api_key, config_id, dry_run=True):
        self.api_key = api_key
        self.config_id = config_id
        self.dry_run = dry_run
        logging.info("created Z2 client for {}".format(config_id))

    def init_payload(self):
        return {
            "apiKey": self.api_key,
            "configId": self.config_id,
        }

    def edit_items(self, items):
        """
        Edit items
        :return: True if success, False otherwise
        """
        edited_items = [{"name": item.name, "version": item.version} for item in items]
        logging.info("Going to edit items in config {}: {}".format(
            self.config_id,
            str(edited_items)
        ))
        if self.dry_run:
            return True
        payload = self.init_payload()
        payload["items"] = json.dumps(edited_items)
        response = self.request("editItems", payload, "POST")
        j = json.loads(response.text)
        return j.get("success", False)

    def update(self):
        """
        Update configuration
        :return: True if success, False otherwise
        """
        logging.info("Going to update config {}".format(self.config_id))
        if self.dry_run:
            return True
        self.request("update", self.init_payload(), "POST")
        return True

    def update_status(self):
        if self.dry_run:
            return True
        response = self.request("updateStatus", self.init_payload(), "GET")
        logging.info("updateStatus response: {}".format(response.text))
        j = response.json()
        if not j.get("success", False):
            return False
        response = j.get("response", {})
        if response.get("updateStatus", "") != "FINISHED":
            return False
        eh.ensure(response.get("result", "") == "SUCCESS", "Z2 should report successful update")
        return True

    @staticmethod
    @decorators.retries(max_tries=3, delay=5, raise_class=errors.TaskError)
    def request(action, payload, method):
        url = Z2Client.make_url(action)
        if method == "POST":
            response = requests_wrapper.post(url, data=payload)
        else:
            response = requests_wrapper.get(url, params=payload)
        response.raise_for_status()
        return response

    @staticmethod
    def make_url(action):
        base_url = "https://z2.yandex-team.ru/api/v1"
        return "{}/{}".format(base_url, action)
