import jinja2
import json
import logging
import os
import tarfile
import multiprocessing as mp

from sandbox import sdk2
from sandbox import common
from sandbox.projects.common.nanny import nanny

from sandbox.common.types.misc import DnsType
import sandbox.common.types.client as ctc


class ConductorBundle(sdk2.Resource):
    """ Conductor tarball """
    releasers = ["noiseless", "sivanichkin", "nanny-robot", "warwish", "antivabo"]
    releasable = True
    share = True


class ConductorJugglerBundle(sdk2.Resource):
    """ Conductor tarball """
    releasers = ["noiseless", "sivanichkin", "nanny-robot", "warwish", "antivabo"]
    releasable = True
    share = True


class RakeReportResource(sdk2.Resource):
    """ RSpec test results """
    releasable = False
    share = False


class BuildConductorTarball(nanny.ReleaseToNannyTask2, sdk2.Task):
    """Build conductor tarball"""

    class Requirements(sdk2.Requirements):
        dns = DnsType.DNS64
        disk_space = 3072
        client_tags = (
            ctc.Tag.GENERIC | ctc.Tag.LINUX_XENIAL
        )
        cores = 4

    class Parameters(sdk2.Task.Parameters):
        _container = sdk2.parameters.Container(
            "Environment container resource",
            default_value=1085837934,
            required=True
        )
        ref_id = sdk2.parameters.String('Git ref id', default='master', required=True)
        ref_sha = sdk2.parameters.String('Git ref SHA', required=False)
        release = sdk2.parameters.Bool('Create resources', default=False)

    REPO_URL = 'https://{username}:{password}@bb.yandex-team.ru/scm/search_infra/conductor.git'
    CHECKOUT_PATH = 'src'
    BUILD_PATH = 'pkg'
    ARCHIVE_NAME = 'conductor-bundle.tar.gz'
    MYSQL_PATH = 'mysql'
    GEM_DIR = '.gems'

    COMMAND_LIST = [
        ('/opt/nodejs/0.10/bin/node /opt/nodejs/0.10/bin/npm install --registry=https://npm.yandex-team.ru', [CHECKOUT_PATH, 'src', 'public']),
        ('/opt/nodejs/0.10/bin/node ./node_modules/.bin/bower install', [CHECKOUT_PATH, 'src', 'public']),
        ('/opt/nodejs/0.10/bin/node ./node_modules/.bin/enb make', [CHECKOUT_PATH, 'src', 'public']),
    ]

    FILES_FOR_REMOVE = [
        'config/database.yml',
        'config/local.database.yml',
        'config/initializers/local_settings.rb',
        'public/static/libs.*',
        'public/static/blocks',
        'node_modules',
        'vendor/gems',
        'vendor/bundle/ruby/2.*/gems/mysql2*/spec/ssl'
    ]

    FILE_TO_COPY = [
        {
            'from': [
                '.bundle',
                GEM_DIR,
                'app',
                'config',
                'db',
                'lib',
                'bin',
                'public',
                'vendor',
                'Gemfile',
                'Gemfile.lock',
                'Rakefile',
                'config.ru'
            ],
            'to': [BUILD_PATH]
        },
        {
            'from': ['test/old_factories'],
            'to': [BUILD_PATH, 'test']
        },
        {
            'from': [
                '/var/lib/cays/blocks',
                '/var/lib/cays/libs.min.css',
                '/var/lib/cays/libs.min.js'
            ],
            'to': [BUILD_PATH, 'public', 'static']
        },
        {
            'from': [
                '../debian/database.yml'
            ],
            'to': [BUILD_PATH, 'config']
        }
    ]

    @property
    def get_env(self):
        env = os.environ.copy()
        env['GEM_HOME'] = self.gem_path
        env['RAILS_ENV'] = 'test'
        return env

    @property
    def gem_path(self):
        return self.path().joinpath(self.CHECKOUT_PATH, 'src', self.GEM_DIR).as_posix()

    def _get_repo_url(self):
        oauth_token = sdk2.Vault.data('RCCS-ADMINS', 'robot_conductor_bb_oauth_token')
        return self.REPO_URL.format(username='x-oauth-token', password=oauth_token)

    def checkout(self):
        logging.info('Fetching data from repository')
        clone_command = 'git clone {repo} {path}'.format(repo=self._get_repo_url(), path=self.CHECKOUT_PATH)
        fetch_command = 'git fetch origin'
        checkout_command = 'git checkout -f {ref_id}'.format(ref_id=self.Parameters.ref_sha or self.Parameters.ref_id)
        with sdk2.helpers.ProcessLog(self, logger='git_checkout') as pl:
            if not os.path.exists(self.CHECKOUT_PATH):
                sdk2.helpers.subprocess.Popen(
                    clone_command.split(),
                    stdout=pl.stdout,
                    stderr=pl.stderr
                ).wait()
            else:
                sdk2.helpers.subprocess.Popen(
                    fetch_command.split(),
                    cwd=self.CHECKOUT_PATH,
                    stdout=pl.stdout,
                    stderr=pl.stderr
                ).wait()
            sdk2.helpers.subprocess.Popen(
                checkout_command.split(),
                cwd=self.CHECKOUT_PATH,
                stdout=pl.stdout,
                stderr=pl.stderr
            ).wait()

    def prepare_env(self):
        logging.info('Preparing environment')
        commands = [
            '/usr/bin/gem2.1 install --no-document --install-dir {} -E bundler -v 1.17.3'.format(self.gem_path),
            '{}/bin/bundle install --no-deployment --jobs={}'.format(self.gem_path, mp.cpu_count()),
            '{}/bin/bundle install --deployment --jobs={}'.format(self.gem_path, mp.cpu_count())
        ]
        with sdk2.helpers.ProcessLog(self, logger='prepare') as pl:
            for cmd in commands:
                sdk2.helpers.subprocess.Popen(
                    cmd.split(),
                    cwd=self.path().joinpath(self.CHECKOUT_PATH, 'src').as_posix(),
                    stdout=pl.stdout,
                    env=self.get_env,
                    stderr=pl.stderr
                ).wait()
        common_path = self.path().joinpath(self.MYSQL_PATH)
        dbpath = common_path.joinpath('db')
        logpath = common_path.joinpath('logs')
        for path in common_path, dbpath, logpath:
            path.mkdir(mode=0o775, parents=True, exist_ok=True)
        install_db_cmd = '/usr/sbin/mysqld --no-defaults --initialize-insecure --user=mysql --basedir=/usr --datadir={dbpath}'.format(dbpath=dbpath.as_posix())
        mysql_cmd = (
            '/usr/sbin/mysqld '
            '--no-defaults '
            '--datadir={dbpath} '
            '--log-error={logpath}/mysqld-3306_err.log '
            '--pid-file={common_path}/mysqld-3306.pid '
            '--socket={common_path}/mysqld-3306.sock '
            '--port=33306 '
            '--sql-mode=ONLY_FULL_GROUP_BY,STRICT_TRANS_TABLES,NO_ZERO_IN_DATE,NO_ZERO_DATE,ERROR_FOR_DIVISION_BY_ZERO,NO_AUTO_CREATE_USER,NO_ENGINE_SUBSTITUTION'
        ).format(
            dbpath=dbpath.as_posix(),
            logpath=logpath.as_posix(),
            common_path=common_path.as_posix()
        )
        # Patching some files. TODO: delete this
        with open(self.path().joinpath(self.CHECKOUT_PATH, 'src', '.rspec').as_posix(), 'r+') as f:
            data = f.read().replace('--color', '--color\n--format json\n--out out.json\n--format html\n--out out.html')
            f.seek(0)
            f.write(data)
        with open(self.path().joinpath(self.CHECKOUT_PATH, 'src', 'config', 'database.yml').as_posix(), 'r+') as f:
            data = f.read().replace('encoding: utf8', 'encoding: utf8\n  socket: {}/mysqld-3306.sock'.format(common_path.as_posix()))
            f.seek(0)
            f.write(data)
        with sdk2.helpers.ProcessLog(self, logger='mysql') as pl:
            sdk2.helpers.subprocess.Popen(
                install_db_cmd.split(),
                stdout=pl.stdout,
                stderr=pl.stderr
            ).wait()
        self.Context.mysqld_pid = sdk2.helpers.subprocess.Popen(mysql_cmd.split()).pid
        self.Context.mysqld_socket_path = '{common_path}/mysqld-3306.sock'.format(common_path=common_path.as_posix())

    def run_tests(self):
        cmds = [
            'bin/rake db:drop db:create db:migrate',
            'bin/rake spec'
        ]
        with sdk2.helpers.ProcessLog(self, logger='rake') as pl:
            for cmd in cmds:
                process = sdk2.helpers.subprocess.Popen(
                    cmd.split(),
                    cwd=self.path().joinpath(self.CHECKOUT_PATH, 'src').as_posix(),
                    env=self.get_env,
                    stdout=pl.stdout,
                    stderr=pl.stderr
                )
                process.wait()
        self.Context.json_report = open(self.path().joinpath(self.CHECKOUT_PATH, 'src', 'out.json').as_posix()).read().strip()
        sdk2.ResourceData(RakeReportResource(
            self,
            'RSpec test results',
            self.path().joinpath(self.CHECKOUT_PATH, 'src', 'out.html').as_posix()
        ))
        # sdk2.helpers.subprocess.Popen(
        #    'mysqladmin -u root -S {socket} shutdown'.format(socket=self.Context.mysqld_socket_path).split()
        # )
        if process.returncode != 0:
            raise common.errors.TaskFailure("At least one test failed.\nSee footer for brief or report tab for detailed info")

    def build(self):
        logging.info('Building conductor')
        env = os.environ.copy()
        env['GEM_HOME'] = self.gem_path
        self.path().joinpath(self.BUILD_PATH).mkdir(mode=0o775, parents=True, exist_ok=True)
        with sdk2.helpers.ProcessLog(self, logger='conductor_build') as pl:
            for command in self.COMMAND_LIST:
                sdk2.helpers.subprocess.Popen(
                    command[0].split(),
                    cwd=self.path().joinpath(*command[1]).as_posix(),
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                    env=env
                ).wait()
        logging.info('Preparing bundle')
        with sdk2.helpers.ProcessLog(self, logger='prepare') as pl:
            sdk2.helpers.subprocess.Popen(
                ['rm', '-vrf'] + self.FILES_FOR_REMOVE,
                shell=False,
                cwd=self.path().joinpath(self.CHECKOUT_PATH, 'src').as_posix(),
                stdout=pl.stdout,
                stderr=pl.stderr
            ).wait()
            for item in self.FILE_TO_COPY:
                sdk2.helpers.subprocess.Popen(
                    ['cp', '-vr'] + item['from'] + [self.path().joinpath(*item['to']).as_posix()],
                    shell=False,
                    cwd=self.path().joinpath(self.CHECKOUT_PATH, 'src').as_posix(),
                    stdout=pl.stdout,
                    stderr=pl.stderr
                ).wait()
        # version, _ = sdk2.helpers.subprocess.Popen(
        #    ['dpkg-parsechangelog', '-S', 'Version'],
        #    stdout=sdk2.helpers.subprocess.PIPE,
        #    cwd=self.path().joinpath(self.CHECKOUT_PATH).as_posix()
        # ).communicate()
        if self.Parameters.release:
            version = self.Parameters.ref_id.split('/', 2)[-1]
        else:
            version = 'testing-{}'.format(self.Parameters.ref_sha[:8])
        with open(self.path(self.BUILD_PATH).joinpath('config', 'conductor.version').as_posix(), 'w') as version_file:
            version_file.write(version.strip())
        archive_path = self.path(self.ARCHIVE_NAME)
        if archive_path.exists():
            archive_path.unlink()
        with tarfile.open(archive_path.as_posix(), 'w:gz') as tar:
            for entry in self.path(self.BUILD_PATH).iterdir():
                tar.add(entry.as_posix(), entry.name)
        sdk2.ResourceData(ConductorBundle(
            self,
            'Conductor tarball, version {}'.format(version.strip()),
            archive_path.as_posix(),
            ttl=180 if self.Parameters.release else 30
        ))

    @sdk2.footer()
    def build_footer(self):
        if not self.Context.json_report:
            report_file_path = self.path().joinpath(self.CHECKOUT_PATH, 'src', 'out.json').as_posix()
            if not os.path.exists(report_file_path):
                return "No report file found"
            report_file = open(report_file_path, 'r').read().strip()
        result = self.Context.json_report or report_file
        data = json.loads(result)
        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path), extensions=['jinja2.ext.do'])
        return env.get_template("footer.html").render(data)

    def on_execute(self):
        self.checkout()
        self.prepare_env()
        self.run_tests()
        self.build()

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
