import os
import logging
import tarfile
import multiprocessing as mp

from sandbox import sdk2

from sandbox.common.types.misc import DnsType
import sandbox.common.types.client as ctc


class ConductorRubyEnvironment(sdk2.Resource):
    """ Ruby environment for conductor """
    releasers = ["qwizzy"]
    releasable = True
    share = False


class BuildRubyEnv(sdk2.Task):
    """Build Ruby Environment"""

    dns = DnsType.DNS64

    class Requirements(sdk2.Requirements):
        disk_space = 3072
        client_tags = (
            ctc.Tag.GENERIC | ctc.Tag.LINUX_XENIAL
        )
        cores = 4

    resources = {
        'zlib': 540384823,
        # 'zlib': 10,
        'ncurses': 540385939,
        # 'ncurses': 11,
        'readline': 540387715,
        # 'readline': 12,
        'openssl': 540404806,
        # 'openssl': 13,
        'ruby': 540382347
        # 'ruby': 14
    }

    PREFIX = 'ruby_prefix'

    ARCHIVE_NAME = 'ruby2.1.tgz'

    THREADS = str(mp.cpu_count())
    # THREADS = "1"

    execution_space = 1000

    BUILD_PARAMS = {
        'zlib': {
            'configure': './configure --prefix={prefix}',
            'make': 'make -j{threads}',
            'make_install': 'make -j{threads} install'
        },
        'ncurses': {
            'env': {'CFLAGS': '-fPIC'},
            'configure': './configure --with-shared --prefix={prefix}',
            'make': 'make -j{threads}',
            'make_install': 'make -j{threads} install'
        },
        'readline': {
            'env': {'CFLAGS': '-fPIC'},
            'configure': './configure --with-curses --prefix={prefix}',
            'make': 'make -j{threads}',
            'make_install': 'make -j{threads} install'
        },
        'openssl': {
            'env': {'CFLAGS': '-fPIC'},
            'configure': './config shared --prefix={prefix}',
            'make': 'make -j{threads}',
            'make_install': 'make -j{threads} install'
        },
        'ruby': {
            'configure': (
                './configure --program-suffix=2.1 --sysconfdir=/etc --localstatedir=/var --sharedstatedir=/var/lib --libexecdir=/usr/lib/ruby '
                '--enable-shared --disable-rpath --disable-werror --disable-install-capi --disable-install-rdoc '
                '--with-openssl-dir={prefix} --with-readline-dir={prefix} --with-zlib-dir={prefix} --prefix={prefix}'
            ),
            'make': 'make -j{threads} ruby',
            'make_install': 'make -j{threads} install-nodoc'
        }
    }

    @property
    def _get_path(self, path):
        return self.path(path).as_posix()

    def _get_sandbox(self, name, resource_id, srcpath):
        resource = sdk2.Resource.find(id=resource_id).first()
        resource_path = sdk2.ResourceData(resource).path
        srcpath.mkdir(mode=0o775, exist_ok=True)
        logging.debug('resource_path: mode {} owner {}:{}'.format(oct(resource_path.stat().st_mode), resource_path.owner(), resource_path.group()))
        logging.debug('srcpath: mode {} owner {}:{}'.format(oct(srcpath.stat().st_mode), srcpath.owner(), srcpath.group()))
        with sdk2.helpers.ProcessLog(self, logger='download_{}'.format(name)) as pl:
            sdk2.helpers.subprocess.Popen(
                ["tar", "xf", resource_path.as_posix(), "--strip-components=1", "--directory", srcpath.as_posix()],
                stdout=pl.stdout,
                stderr=pl.stderr
            ).wait()

    def build(self, name):
        env = os.environ.copy()
        if 'env' in self.BUILD_PARAMS[name]:
            env.update(self.BUILD_PARAMS[name]['env'])
        work_dir = self.path(name)
        with sdk2.helpers.ProcessLog(self, logger=name) as pl:
            self._get_sandbox(name, self.resources[name], work_dir)
            for command in ['configure', 'make', 'make_install']:
                logging.debug('Running {} for {}'.format(command, name))
                sdk2.helpers.subprocess.Popen(
                    self.BUILD_PARAMS[name][command].format(prefix=self.path(self.PREFIX).as_posix(), threads=self.THREADS).split(),
                    cwd=work_dir.as_posix(),
                    env=env,
                    stdout=pl.stdout,
                    stderr=pl.stderr
                ).wait()

    def make_archive(self):
        env = os.environ.copy()
        env.update({"LD_LIBRARY_PATH": "{}".format(self.path(self.PREFIX).joinpath('lib').as_posix())})
        stdout, _ = sdk2.helpers.subprocess.Popen(
            [self.path(self.PREFIX).joinpath('bin', 'ruby2.1').as_posix(), '--version'],
            stdout=sdk2.helpers.subprocess.PIPE,
            env=env
        ).communicate()
        archive_path = self.path(self.ARCHIVE_NAME)
        if archive_path.exists():
            archive_path.unlink()
        with tarfile.open(archive_path.as_posix(), 'w:gz') as tar:
            for entry in self.path(self.PREFIX).iterdir():
                tar.add(entry.as_posix(), entry.name)
        sdk2.ResourceData(ConductorRubyEnvironment(
            self,
            stdout.strip(),
            archive_path.as_posix()
        ))

    def on_execute(self):
        for pkg in ['zlib', 'ncurses', 'readline', 'openssl', 'ruby']:
            self.build(pkg)
        self.make_archive()
