# coding: utf-8
import os

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.common import constants as consts
from sandbox.projects.common import utils
from sandbox.projects.common.build import parameters as build_params
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.nanny import const as nanny_const
from sandbox.projects.common.nanny import nanny

import sandbox.projects.context.resource_types as resource_types


SERVICE_BUILD_DIR_NAME = 'app_service'
SERVICE_BINARY_NAME = 'app_service'
YACOSHARE_BINARY_NAME = 'yacoshare'
YACOSHARE_APP_NAME = 'yacoshare'


class ServicePath(sdk2.parameters.String):
    name = 'service_path'
    description = 'Service path'
    required = True
    default_value = 'ads/yacontext/apps/sample/{}'.format(SERVICE_BUILD_DIR_NAME)


class ArtefactName(sdk2.parameters.String):
    name = 'artefact_name'
    description = 'Artefact name'
    required = True
    default_value = SERVICE_BINARY_NAME


class ReleaseStatusParam(sdk2.parameters.RadioGroup):
    name = 'release_status'
    description = 'Release status'
    required = True
    choices = [
        ('stable', ctt.ReleaseStatus.STABLE),
        ('testing', ctt.ReleaseStatus.TESTING),
        ('unstable', ctt.ReleaseStatus.UNSTABLE),
        ('no release', ctt.ReleaseStatus.CANCELLED),
    ]
    default_value = ctt.ReleaseStatus.CANCELLED


class BuildContextBinary(nanny.ReleaseToNannyTask, YaMakeTask):
    """
    Build the binary file of a service at context
    """

    type = 'BUILD_CONTEXT_BINARY'
    input_parameters = [
        ServicePath,
        ArtefactName,
        ReleaseStatusParam,
        build_params.BuildType,
        build_params.ArcadiaUrl,
        build_params.ClearBuild,
        build_params.ArcadiaPatch,
        build_params.DefinitionFlags,
        build_params.LTO,
        build_params.ThinLTO,
        build_params.YtStore,
    ]

    def get_app_path(self):
        service_path = utils.get_or_default(self.ctx, ServicePath)
        app_path = service_path
        if os.path.basename(app_path) == SERVICE_BUILD_DIR_NAME:
            app_path = os.path.dirname(app_path)
        return app_path

    def get_artefact_path(self):
        service_path = utils.get_or_default(self.ctx, ServicePath)
        return os.path.join(service_path, utils.get_or_default(self.ctx, ArtefactName))

    def initCtx(self):
        YaMakeTask.initCtx(self)
        self.ctx[consts.USE_AAPI_FUSE] = True
        self.ctx[consts.ALLOW_AAPI_FALLBACK] = True
        self.ctx[build_params.BuildSystem.name] = consts.SEMI_DISTBUILD_BUILD_SYSTEM

    def get_resources(self):
        app_path = self.get_app_path()
        artefact_name = utils.get_or_default(self.ctx, ArtefactName)
        if artefact_name == YACOSHARE_BINARY_NAME:
            resource_type = resource_types.YacoshareClient
        elif os.path.basename(app_path) == YACOSHARE_APP_NAME:
            resource_type = resource_types.YacoshareServer
        else:
            resource_type = resource_types.ContextServiceBinary
        return {
            'project': {
                'description': '',
                'resource_path': artefact_name,
                'resource_type': resource_type,
            }
        }

    def get_targets(self):
        return [utils.get_or_default(self.ctx, ServicePath)]

    def get_arts(self):
        app_path = self.get_app_path()
        self.ctx[nanny_const.LABELS] = {'service': os.path.basename(app_path)}
        artefact_path = self.get_artefact_path()
        return [{'dest': '', 'path': artefact_path}]

    def on_success(self):
        YaMakeTask.on_success(self)
        if utils.get_or_default(self.ctx, ReleaseStatusParam) == ctt.ReleaseStatus.CANCELLED:
            return
        nanny.ReleaseToNannyTask.on_release(
            self,
            dict(
                releaser=self.author,
                release_status=utils.get_or_default(self.ctx, ReleaseStatusParam),
                release_subject="Context binary release",
                email_notifications=dict(to=[], cc=[]),
                release_comments="Context binary release",
            ),
        )
