# coding=utf-8

from __future__ import unicode_literals

import logging
import requests
from operator import itemgetter

from sandbox.projects.cornholio.TicketWeightAutoincrement.process_modifier import process_modifier, process_deadline
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox import sdk2


class TicketWeightAutoincrementReport(sdk2.Resource):
    ttl = 14


class TicketWeightAutoincrement(sdk2.Task):
    """
    Таска для автоматического пересчёта весов тикетов в трекере
    """
    oauth_token = None

    class Parameters(sdk2.Task.Parameters):
        st_query = sdk2.parameters.String(
            "Запрос в трекере",
            required=True
        )

        field_modifiers = sdk2.parameters.List(
            "Модификаторы по свойствам тикетов",
            default={
                "priority/id:f+50",
                "votes:f+8",
                "__age:f+0.15",
                "tags/я_нужен_из-за_факапа:*1.3",
                "type/key==bug:*1.2",
            },
            description="Формат: поле/вложенное_поле:модификатор.\n"
                        "Модификатор может начинаться на '*' или '+': *2 или +50.\n"
                        "Если добавить f перед модификатором, то для числового значения поля, "
                        "он применится, умноженный на число.\n"
                        "Модификаторы приеняются последовательно.\n"
                        "__age — особый, количество дней с момента создания"
        )

        deadline_addon = sdk2.parameters.Integer(
            "Чем ближе к дедлайну, тем ближе к этому числу.",
            default=300
        )

        deadline_reaction = sdk2.parameters.Integer(
            "Чем выше, тем раньше и медленнее будет расти вес. "
            "При 30, вес начнёт расти за 200 дней, "
            "и наберёт треть максимума в 50 дней до дедлайна",
            default=30
        )

        dry_run = sdk2.parameters.Bool(
            "Dry run",
            default=True,
            description="Не проставлять вес тикетам, просто вывести результат"
        )

        weight_field_name = sdk2.parameters.String(
            "Ключ поля веса",
            default="weight",
            description="Ключ поля для веса. По умолчанию вес проставляется в поле Вес - weight."
        )

        with sdk2.parameters.Group("Данные робота"):
            useragent = sdk2.parameters.String(
                "Имя роботного пользователя",
                required=True,
                default="robot-metatron"
            )

            vault_oauth_id = sdk2.parameters.String(
                "id OAuth-токена в Vault",
                required=True,
                default="robot-metatron-st-token"
            )

    class Requirements(sdk2.Task.Requirements):
        environments = [
            PipEnvironment('yandex_tracker_client', version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment('startrek_client', version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    def on_execute(self):
        super(TicketWeightAutoincrement, self).on_execute()

        self.oauth_token = sdk2.Vault.data(self.Parameters.vault_oauth_id)

        from startrek_client import Startrek
        st = Startrek(self.Parameters.useragent, token=self.oauth_token)
        all_tickets = list(st.issues.find(self.Parameters.st_query))

        result = []

        logging.debug("Found tickets count is '{}'".format(len(all_tickets)))

        for ticket in all_tickets:
            logging.debug("Calcluating weights for '{}'".format(ticket.key))
            overall_score = 1

            for mod in self.Parameters.field_modifiers:
                logging.debug("Applying modifier '{}'".format(mod))
                overall_score = process_modifier(ticket, overall_score, mod)

                logging.debug("Applying modifier '{}' complete; overall score now is {}".format(mod, overall_score))

            if int(self.Parameters.deadline_addon) > 0:
                logging.debug("Applying deadline modifier")
                overall_score = process_deadline(
                    ticket, overall_score, self.Parameters.deadline_addon, self.Parameters.deadline_reaction
                )

                logging.debug("Applying deadline modifier complete; overall score now is {}".format(overall_score))

            overall_score = int(round(overall_score))
            logging.debug(
                "Calcluating weights for '{}' complete; overall score is {}".format(ticket.key, overall_score)
            )

            result.append((ticket, overall_score))

        html_result = "<head><meta charset=\"UTF-8\"><style>" \
                      "table tr td {border: 1px solid; padding: 5px} table {border-collapse: collapse;}" \
                      "</style></head><body><table>" \
                      "<thead><tr><td>Вес</td><td>Номер</td><td>Описание</td></tr></thead><tbody>"

        for ticket in sorted(result, key=itemgetter(1), reverse=True):
            html_result += "<tr><td>{score}</td><td><a target=\"_blank\" href=\"https://st.yandex-team.ru/{key}\">{key}</a></td><td>{summary}</td></tr>".format(
                score=ticket[1],
                key=ticket[0].key,
                summary=ticket[0].summary,
            )

            if not self.Parameters.dry_run:
                self.set_weight(ticket[0].key, ticket[1], self.Parameters.weight_field_name)

        html_result += "</tbody></table></body>"

        with open("result.html", "w") as f:
            f.write(html_result.encode('utf-8'))

        report_res = TicketWeightAutoincrementReport(self, "Рассчитанные веса", "result.html")
        report_res_data = sdk2.ResourceData(report_res)
        report_res_data.ready()

        self.set_info(
            "Новые веса тикетов: <a href=\"{url}\">{url}</a>".format(url=report_res.http_proxy),
            do_escape=False
        )

    def set_weight(self, ticket_id, weight, weight_field_name):
        logging.debug(
            "Setting {} weight for '{}'".format(ticket_id, weight)
        )

        values = {};
        values[weight_field_name] = weight;

        # https://st.yandex-team.ru/TOOLSUP-27161#1515675424000 кажется, только через балк и не через python-клиент
        r = requests.post("https://st-api.yandex-team.ru/v2/bulkchange/_update", json={
            "issues": [ticket_id],
            "values": values
        }, headers={
            "Authorization": "OAuth {}".format(self.oauth_token)
        })

        try:
            r.raise_for_status()
        except requests.HTTPError:
            error_msg = "ST HTTP error: response: \"{}\"; status code: {}".format(r.text, r.status_code)
            logging.error(error_msg)
            self.set_info(error_msg)
            raise
