# -*- coding: utf-8 -*-
import os
import logging
import sys
import shutil


import sandbox.sandboxsdk.svn as sdk_svn

from sandbox.projects.ab_testing import RUN_ABT_RESOURCES
from sandbox.projects.cross_stability import CROSS_STABILITY_BASKET, CROSS_STABILITY_RESAULT
from sandbox.projects.resource_types import YA_PACKAGE


from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.resource as ctr
import datetime
import random


from sandbox import sdk2


class CalcCrossStability(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        cluster = sdk2.parameters.String(
            "YT cluster",
            default_value="hahn",
            description="YT cluster",
            required=True
        )
        browser_resourse_id = sdk2.parameters.LastReleasedResource(
            "browser_resourse_id",
            resource_type=RUN_ABT_RESOURCES,
            state=(ctr.State.READY, ctr.State.NOT_READY),
            required=True
        )

        basket_resource_id = sdk2.parameters.LastReleasedResource(
            "basket_resource_id",
            resource_type=CROSS_STABILITY_BASKET,
            state=(ctr.State.READY, ctr.State.NOT_READY),
            required=True
        )

        cross_stability = sdk2.parameters.LastReleasedResource(
            "add_current_data_to_basket, cross_metrics_off",
            resource_type=YA_PACKAGE,
            state=(ctr.State.READY, ctr.State.NOT_READY),
            required=True
        )

        vault_group = sdk2.parameters.String(
            "Vault group of yt token.",
            default_value="AB-TESTING",
            description="Vault group of yt token.",
            required=True
        )

        vault_name = sdk2.parameters.String(
            "Vault name of yt token.",
            default_value="yt-token",
            description="Vault name",
            required=True
        )

        yt_pool = sdk2.parameters.String(
            "YT pool for calculation.",
            default_value="abt-prod-other",
            description="",
            required=False
        )

        stab_prefix = sdk2.parameters.String(
            "Stab prefix.",
            default_value="",
            description="Stab prefix.",
            required=True
        )

        cgi = sdk2.parameters.String(
            "Add cgi to url &test-id=1,... ",
            default_value="",
            description="Add cgi to url &test-id=1,... ",
            required=False
        )

        delta = sdk2.parameters.Integer(
            "delta between request",
            default_value=60,
            description="delta between request",
            required=True
        )

        count_requests = sdk2.parameters.Integer(
            "count requests",
            default_value=60,
            description="count requests",
            required=True
        )

        tex = sdk2.parameters.Bool(
            "tex requests",
            default_value=False,
            description="tex requests",
            required=True
        )

        stat = sdk2.parameters.Bool(
            "Statistics",
            default_value=False,
            description="Statistics",
            required=True
        )

        kill_timeout = 5 * 3600

    def run_prepare_patrons(self, conf):
        '''YT_PROXY=hahn MR_RUNTIME=YT ./add_current_data_to_basket -d ../scripts/dump.txt -o ../scripts/patrons.txt -u browser.xml'''
        cmd = [
            conf['add_current_data_to_basket'],
            '-d', conf['basket'],
            '-o',  'patrons.txt',
            '-u', conf['browser']
        ]
        logging.info('run_prepare_patron %s', ' '.join(cmd))

        with sdk2.helpers.ProcessLog(self, logger="run_prepare_patron") as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=sp.STDOUT)

    def run_stability_tank(self, conf):
        '''python stability_tank.py --us user_sessions.txt --patrons patrons.txt --delta 60 --cgi "&test-id=1" --prefix notest_9 --logs logs1.txt --tex'''

        cmd = [
            conf['stability_tank.py'],
            '--patrons', 'patrons.txt',
            '--delta',  conf['delta'],
            '--cgi', conf['cgi'],
            '--prefix',  conf['prefix'],
            '--logs', conf['log'],
            '--num', conf['num']
        ]
        if conf.get('tex'):
            cmd.extend(['--tex'])
        if conf.get('us'):
            cmd.extend(['--us', conf['us']])

        logging.info('run_stability_tank {}'.format(cmd))
        with sdk2.helpers.ProcessLog(self, logger="run_stability_tank") as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=sp.STDOUT)

    def run_statistics(self, conf):
        cmd = ["sort", conf['us'], "-u", "-o", "sort_us.txt"]
        with sdk2.helpers.ProcessLog(self, logger="run_sort_user_sessions") as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=sp.STDOUT)

        './cross_metrics_off  -i ../scripts/user_sessions.txt  --prefix "" --local -o r.txt --log 600 -e ../scripts/logs1.txt'
        cmd = [
            conf['cross_metrics_off'],
            '-i', "sort_us.txt",
            '--local',
            '-o', conf['statistics'],
            '--log', conf['count_sample_log'],
            '-e', conf['log'],
            '-d', conf['detail']
        ]
        with sdk2.helpers.ProcessLog(self, logger="run_cross_metrics_off") as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=sp.STDOUT)

    def run_detail(self, conf):
        '''python rang.py --factors detail --output  report_blender --stable  main:type_b_1 --prefix all'''
        for stable_param, stable_report in conf['detail_params']:

            cmd = [
                conf['rang.py'],
                '--factors', conf['detail'],
                '--output', stable_report,
                '--stable', stable_param,
                '--prefix', 'all'
            ]

            logging.info('run_detail {}'.format(cmd))
            with sdk2.helpers.ProcessLog(self, logger="run_detail") as pl:
                sp.check_call(cmd, stdout=pl.stdout, stderr=sp.STDOUT)

    def on_execute(self):
        shellabt_path = sdk_svn.Arcadia.get_arcadia_src_dir("arcadia:/arc/trunk/arcadia/quality/ab_testing/scripts/shellabt/")
        sys.path.append(shellabt_path)

        import shellabt

        conf = shellabt.FreeConfig()

        conf['log'] = "logs.txt"
        conf["count_sample_log"] = '600'
        conf['statistics'] = "stat.txt"

        package_id = self.Parameters.cross_stability
        logging.info('Syncing resource: %d', package_id)
        package_path = str(sdk2.ResourceData(package_id).path)

        os.environ["YT_PROXY"] = self.Parameters.cluster

        conf['delta'] = str(self.Parameters.delta)
        conf['tex'] = self.Parameters.tex
        conf['num'] = str(self.Parameters.count_requests)
        str_date = datetime.datetime.strftime(datetime.datetime.today(), '%Y%m%d')
        str_rand = str(random.randint(0, 10000))
        stab_prefix = str(self.Parameters.stab_prefix).format(date=str_date, rnd=str_rand)

        conf['prefix'] = stab_prefix
        conf['cgi'] = str(self.Parameters.cgi)

        stat = self.Parameters.stat
        if stat:
            conf['us'] = "user_sessions.txt"
        conf['count_sample_log'] = '600'
        conf['log'] = 'logs.txt'
        conf['detail'] = 'detail'
        conf['detail_params'] = [("main:type_b_1", "report_blender_1.html"), ("main:type_b_3", "report_blender_3.html"), ("main:type_b_5", "report_blender_5.html")]

        browser_resource_id = self.Parameters.browser_resourse_id
        resources = sdk2.ResourceData(browser_resource_id)
        conf['browser'] = os.path.join(str(resources.path), 'browser.xml')

        basket_resource_id = self.Parameters.basket_resource_id
        resources = sdk2.ResourceData(basket_resource_id)
        conf['basket'] = str(resources.path)

        extract_to = './'
        logging.info('Extracting: %s to %s', package_path, extract_to)
        paths = shellabt.DebianYaPackagePaths(package_path, '/Berkanavt/ab_testing/bin', extract_to)
        conf.merge(paths)
        paths = shellabt.DebianYaPackagePaths(package_path, '/Berkanavt/ab_testing/scripts/cross_stability', extract_to)
        conf.merge(paths)

        os.environ['YT_TOKEN'] = sdk2.Vault.data(self.Parameters.vault_group, self.Parameters.vault_name)
        os.environ['MR_RUNTIME'] = 'YT'
        yt_pool = self.Parameters.yt_pool
        if yt_pool:
            os.environ['YT_POOL'] = yt_pool

        self.run_prepare_patrons(conf)
        self.run_stability_tank(conf)
        if conf.get('us'):
            self.run_statistics(conf)
            self.run_detail(conf)

            logging.info('Create resource')
            name = 'cross_stability_report'
            statistics = os.path.abspath(conf['statistics'])
            us = os.path.abspath(conf['us'])
            resource = CROSS_STABILITY_RESAULT(self, "Cross stability", name, ttl=365)
            data = sdk2.ResourceData(resource)
            data.path.mkdir(0o755, parents=True, exist_ok=True)
            logging.info('%s and %s to %s' % (statistics, us, str(data.path)))
            shutil.copy2(statistics, str(data.path))
            shutil.copy2(us, str(data.path))
            logging.info('%s.n and %s.f to %s' % (conf['detail'], conf['detail'], str(data.path)))
            shutil.copy2(conf['detail'] + ".n", str(data.path))
            shutil.copy2(conf['detail'] + ".f", str(data.path))
            for _, stable_report in conf['detail_params']:
                logging.info('%s to %s' % (stable_report, str(data.path)))
                shutil.copy2(stable_report, str(data.path))
            data.ready()
