import logging

from sandbox import sandboxsdk
from sandbox.projects.common.build import YaPackage


class AutoReleaseBlock(sandboxsdk.parameters.SandboxInfoParameter):
    description = "AutoRelease"


class AutoReleaseEnabled(sandboxsdk.parameters.SandboxBoolParameter):
    name = "auto_release"
    description = "Enable auto release"
    required = False
    default_value = False


class AutoReleaseEnv(sandboxsdk.parameters.SandboxSelectParameter):
    STABLE = "stable"
    PRESTABLE = "prestable"
    TESTING = "testing"
    UNSTABLE = "unstable"
    CANCELLED = "cancelled"

    name = "auto_release_env"
    description = "Autorelease to environment"
    required = False
    choices = [
        (STABLE, STABLE),
        (PRESTABLE, PRESTABLE),
        (TESTING, TESTING),
        (UNSTABLE, UNSTABLE),
        (CANCELLED, CANCELLED),
    ]
    default_value = TESTING


class AutoReleaseAuthor(sandboxsdk.parameters.SandboxStringParameter):
    name = "auto_release_author"
    description = "Autorelease author"
    required = False


class CryptaBuildAndRelease(YaPackage.YaPackage, object):
    """
        Task to run YA_PACKAGE and auto-release resources to the given environment
    """

    type = "CRYPTA_BUILD_AND_RELEASE"
    input_parameters = YaPackage.YaPackage.input_parameters + [
        AutoReleaseBlock,
        AutoReleaseEnabled,
        AutoReleaseEnv,
        AutoReleaseAuthor,
    ]

    def on_execute(self):
        logging.info("Run original ya_package")
        logging.info(
            "Autorelease branch set to: %s, enabled: %s",
            self.ctx.get(AutoReleaseEnv.name),
            self.ctx.get(AutoReleaseEnabled.name),
        )
        YaPackage.YaPackage.on_execute(self)
        if self.ctx.get(AutoReleaseEnabled.name):
            logging.info("Autoreleasing!")
            YaPackage.YaPackage.on_release(
                self,
                {
                    "release_status": self.ctx.get(AutoReleaseEnv.name),
                    "releaser": (self.ctx.get(AutoReleaseAuthor.name) or self.owner),
                    "release_comments": "automatic release",
                    "release_subject": "",
                    "email_notifications": {"cc": [], "to": []},
                },
            )


__Task__ = CryptaBuildAndRelease
