import glob
import os
import re

import sandbox.projects.common.gnupg
import sandbox.projects.common.debpkg
from sandbox import sandboxsdk
from sandbox import sdk2

from sandbox.sdk2.helpers import subprocess


REPOS = ["crypta-precise", "crypta-trusty", "crypta-common", "crypta-qa"]


class CryptaDebPackage(sdk2.Resource):
    pass


class CryptaBuildDebPackage(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        privileged = True

    class Parameters(sdk2.Task.Parameters):
        _container = sdk2.parameters.Container("Environment container resource", required=True)
        path_to_arcadia_dir = sdk2.parameters.String("Package path, related to arcadia", required=True)
        path_to_debian = sdk2.parameters.String("Path to 'debian' catalog, related to package", required=True)
        build_script = sdk2.parameters.String("Build script", multiline=True, required=True)
        changelog = sdk2.parameters.String("Changelog", multiline=True, required=False)
        vault_items_owner = sdk2.parameters.String("Vault items owner", required=True)
        robot_login = sdk2.parameters.String("Robot login (without @yandex-team.ru)", required=True)
        gpg_private_key = sdk2.parameters.String("GPG private key (vault item)", required=True)
        gpg_public_key = sdk2.parameters.String("GPG public key (vault item)", required=True)
        publish = sdk2.parameters.Bool("Publish")
        do_not_remove_resources = sdk2.parameters.Bool("Do not remove resources", required=True, default=False)
        version_prefix = sdk2.parameters.String("Package version prefix", default="")
        with publish.value[True]:
            publish_to = sdk2.parameters.String("Publish to", choices=[(name, name) for name in REPOS], required=True)
            ssh_private_key = sdk2.parameters.String("SSH private key (vault item)", required=True)

    class Context(sdk2.Task.Context):
        pass

    def on_prepare(self):
        self.Context.workspace = os.path.join(os.getcwd(), "workspace")
        self.Context.debian = os.path.abspath(
            os.path.join(self.Context.workspace, self.Parameters.path_to_debian, "debian")
        )
        arcadia_url = sdk2.svn.Arcadia.trunk_url(self.Parameters.path_to_arcadia_dir)
        info = sdk2.svn.Arcadia.info(arcadia_url)
        self.Context.svn_revision = info["entry_revision"]
        sdk2.svn.Arcadia.export(arcadia_url, self.Context.workspace, depth="infinity")

    def on_execute(self):
        self._run_build_script()

        with sandbox.projects.common.gnupg.GpgKey(
            self, self.Parameters.vault_items_owner, self.Parameters.gpg_private_key, self.Parameters.gpg_public_key
        ):
            self._run_dch()
            self._run_pbuilder_satisfydepends()
            self._run_debuild()
            self._save_resources()

            if self.Parameters.publish:
                self._run_dupload()

    def get_vault_data(self, vault_item_owner, vault_item_name):
        """
        Added for compatibility with sandbox.projects.common.gnupg, sandboxsdk.ssh
        """
        return sdk2.Vault.data(vault_item_owner, vault_item_name)

    def _run_build_script(self):
        build_script_file = "build_script.sh"
        with open(build_script_file, "w") as f:
            f.write(self.Parameters.build_script)

        env = os.environ.copy()
        env["WORKSPACE"] = self.Context.workspace
        cmd = ["/bin/bash", build_script_file]
        self._run_cmd(cmd, os.getcwd(), "build_script", env)

    def _parse_changelog(self):
        with sdk2.helpers.ProcessLog(self, logger="dpkg-parsechangelog") as pl:
            cmd = ["dpkg-parsechangelog", "-l{}".format(os.path.join(self.Context.debian, "changelog"))]
            parse_changelog_process = subprocess.Popen(cmd, shell=False, stdout=subprocess.PIPE, stderr=pl.stdout)
            assert not parse_changelog_process.wait()
            changelog = []
            changelog_field_pattern = re.compile("^(\\w+):\\s*(.*)")
            for line in parse_changelog_process.stdout.readlines():
                changelog += changelog_field_pattern.findall(line.strip())
            return dict(changelog)

    def _run_dch(self):
        changelog = self._parse_changelog()

        env = os.environ.copy()
        env["DEBEMAIL"] = "{}@yandex-team.ru".format(self.Parameters.robot_login)
        cmd = [
            "dch",
            "--check-dirname-level",
            "0",
            "-b",
            "--distribution",
            changelog["Distribution"],
            "--newVersion",
            "{}{}-trunk".format(self.Parameters.version_prefix, self.Context.svn_revision),
            "--",
            self.Parameters.changelog,
        ]
        self._run_cmd(cmd, self.Context.debian, "dch", env)

    def _run_pbuilder_satisfydepends(self):
        cmd = ["sudo", "/usr/lib/pbuilder/pbuilder-satisfydepends", "--control", "control"]
        self._run_cmd(cmd, self.Context.debian, "pbuilder-satisfydepends")

    def _run_debuild(self):
        cmd = [
            "debuild",
            "--check-dirname-level",
            "0",
            "--no-tgz-check",
            "-k{}".format("{}@yandex-team.ru".format(self.Parameters.robot_login)),
        ]
        self._run_cmd(cmd, self.Context.debian, "debuild")

    def _save_resources(self):
        deb_dir = os.path.dirname(os.path.dirname(self.Context.debian))
        os.chdir(deb_dir)
        for deb in glob.glob("*.deb"):
            path = os.path.join(deb_dir, deb)
            sdk2.ResourceData(
                CryptaDebPackage(self, deb, path, ttl="inf" if self.Parameters.do_not_remove_resources else 30)
            )

    def _run_dupload(self):
        dupload_conf = {}
        for name in REPOS:
            dupload_conf[name] = {
                "fqdn": "{}.dupload.dist.yandex.ru".format(name),
                "method": "scpb",
                "incoming": "/repo/{}/mini-dinstall/incoming/".format(name),
                "dinstall_runs": 1,
            }
        os.chdir(os.path.dirname(self.Context.debian))
        with sandbox.projects.common.debpkg.DebRelease(dupload_conf, login=self.Parameters.robot_login) as deb:
            with sandboxsdk.ssh.Key(self, self.Parameters.vault_items_owner, self.Parameters.ssh_private_key):
                deb.debrelease(["--to", self.Parameters.publish_to])

    def _run_cmd(self, cmd, directory, logger, env=None):
        os.chdir(directory)
        with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
            subprocess.check_call(cmd, shell=False, stdout=pl.stdout, stderr=pl.stdout, env=env)
