# -*- coding: utf-8 -*-
"""
Common module to use Yabs Graphite. Suitable for any other Graphite
installation if host list is known (look at "hosts" parameter of
Graphite.__init__ for details).

Simplest usage example:
>>> import time
>>> from graphite import Graphite, one_min_metric
>>> test_metric = one_min_metric('sandbox_graphite_sender', 'test')
>>> metric_string = '{} {} {}'.format(test_metric('test1'), 10, time.time())
>>> metric_tuple = (test_metric('test2'), 15, time.time())
>>> metric_dict = {'name': test_metric('test3'), 'timestamp': time.time(), 'value': 100500}
>>> Graphite().send([metric_string, metric_tuple, metric_dict])

Also module provides metric_point function to help remember order of arguments.
All following variants are equivalent:
>>> from graphite import metric_point
>>> n, t, v = test_metric('test'), time.time(), 1
>>> metric_string_1 = '{} {} {}'.format(n, v, t)             # manual formatting variant
>>> metric_string_2 = metric_point(n, v, t)                  # better compact variant
>>> metric_string_3 = metric_point(n, value=v, timestamp=t)  # explicit variant

More examples can be found in unit-test module "tests/test_graphite.py".
"""

import socket
import inspect
import logging

import six

YABS_SERVERS = [
    'mega-graphite-man.search.yandex.net:2024',
    'mega-graphite-sas.search.yandex.net:2024'
]


class _DenyExternalInheritanceMetaclass(type):
    def __new__(cls, name, bases, dct):
        if dct.get('__module__') != __name__:
            raise TypeError('Metric classes must not be subclassed in other modules, use specificated global classes directly (one_min_metric, five_sec_metric, etc.)')
        return super(_DenyExternalInheritanceMetaclass, cls).__new__(cls, name, bases, dct)


class _MetricName(six.with_metaclass(_DenyExternalInheritanceMetaclass, object)):
    def __init__(self, *metric_parts, **optional_parts):
        if not hasattr(self, 'frequency'):
            raise TypeError('MetricName class itself must not be instantiated, use specificated global classes instead (one_min_metric, five_sec_metric, etc.)')
        if not metric_parts:
            raise ValueError('Metric must have at least one component after global namespace "<frequency>.<hostname>"')
        self.metric_parts = metric_parts
        self.hostname = optional_parts.get('hostname', 'bs')

    def __call__(self, *metric_parts, **optional_parts):
        if 'hostname' not in optional_parts:
            optional_parts['hostname'] = self.hostname
        return self.__class__(*(self.metric_parts + metric_parts), **optional_parts)

    def __str__(self):
        metric_parts = [self.frequency]
        if self.hostname:
            metric_parts.append(normalize_hostname(self.hostname))
        metric_parts.extend(self.metric_parts)
        return '.'.join(metric_parts)


class Graphite(object):
    @staticmethod
    def _make_metric_point(obj):
        if isinstance(obj, six.string_types):
            obj = obj.strip().split()  # pass to next if to validate parts
        if isinstance(obj, (tuple, list)):
            return metric_point(*obj)
        if isinstance(obj, dict):
            return metric_point(**obj)
        raise RuntimeError("Don't know how to convert {} object to metric point".format(obj))

    def __init__(self, hosts=YABS_SERVERS, timeout=5, logger=None):
        if logger is None:
            logger = logging
        self.hosts = hosts
        self.timeout = timeout
        self.logger = logger

    def send(self, points):
        data = ''.join(
            self._make_metric_point(obj) + '\n'
            for obj in points
        )
        if not data:
            return
        for host in self.hosts:
            hostname, port = host.split(':')
            try:
                self.logger.info("Sending metrics %s to host %s", data, host)

                sock = socket.create_connection((hostname, int(port)), self.timeout)
                sock.sendall(data)
                sock.close()
            except socket.error as err:
                self.logger.warn('Send metrics to %s failed: %s', host, err)


def normalize_hostname(hostname):
    return hostname.replace('.', '_')


def metric_point(name, value, timestamp, **kwargs):
    """
    Validate that all objects are of proper type and build whole metric string.
    Explicit "kwargs" added to make calls like metric_point(**dct) possible for
    dictionaries with extra fields.
    """
    if inspect.isclass(name):
        raise ValueError('Can not use class as a metric name (perhaps you forgot to instantiate it?)')
    name = str(name)
    value = str(float(value))
    timestamp = str(int(float(timestamp)))
    return ' '.join([name, value, timestamp])


def _create_metric_classes():
    for freq in ('one_sec', 'one_min', 'one_hour', 'one_day', 'five_sec', 'five_min'):
        class CurFreqMetric(_MetricName):
            frequency = freq
        globals()[freq + '_metric'] = CurFreqMetric


# Adding on import variables such as one_min_metric to global module scope
_create_metric_classes()
