import logging
import os
import tarfile

import jinja2
from sandbox import sdk2

from sandbox.projects.crypta import resources


STABLE = "stable"
PRESTABLE = "prestable"
TESTING = "testing"
RELEASE_STATUS = (STABLE, PRESTABLE, TESTING)

MINUTE = 60
HOUR = 60 * MINUTE

logger = logging.getLogger(__name__)


def get_last_released_resource_unsafe(resource_type, resource_release_status, **additional_attrs):
    """
    Returns last released resource with given type and with
    release status that is less or equals to the given one.

    Release statuses have the hierarchy:
    stable < prestable < testing.
    """
    logger.info("Looking for resources %s with release status %s", resource_type, resource_release_status)

    last_released_resources = []

    for release_status in RELEASE_STATUS:
        attrs = dict(released=release_status, **additional_attrs)
        resource = resource_type.find(state='READY', attrs=attrs).order(-sdk2.Resource.id).first()
        last_released_resources.append(resource)
        if release_status == resource_release_status:
            break

    last_released_resources = [x for x in last_released_resources if x]

    logger.info("Found released resources: %s", last_released_resources)

    if not last_released_resources:
        return None

    last_released_resource = max(last_released_resources, key=lambda x: x.id)

    logger.info("Last released resource: %s", last_released_resource)

    return last_released_resource


def get_last_released_resource(resource_type, resource_release_status):
    """
    Returns last released resource with given type and with
    release status that is less or equals to the given one.

    Release statuses have the hierarchy:
    stable < prestable < testing.

    Raises exception if can't find resource.
    """
    resource = get_last_released_resource_unsafe(resource_type, resource_release_status)
    assert resource, "Can't find {} resource with released attribute <= {}".format(resource_type, resource_release_status)
    return resource


def get_last_released_universal_resource(name, resource_release_status):
    """
    Returns last released CryptaUniversalBundle with given name and with
    release status that is less or equals to the given one.

    Release statuses have the hierarchy:
    stable < prestable < testing.

    Raises exception if can't find resource.
    """
    resource = get_last_released_resource_unsafe(resources.CryptaUniversalBundle, resource_release_status, name=name)
    assert resource, "Can't find {} resource with released attribute <= {}".format(name, resource_release_status)
    return resource


def download_resource(resource):
    """
    Returns path to downloaded resource
    """
    logger.info("Downloading resource %s", resource)
    return sdk2.ResourceData(resource).path.as_posix()


def untar_path(path):
    """
    Unarchives given tar archive to the current work directory
    """
    logger.info("Untaring %s", path)
    tarfile.open(path, "r").extractall()


def render_template_str(template, **context):
    """
    Render jinja2 template with given context
    """
    return jinja2.Template(template, trim_blocks=True, lstrip_blocks=True, undefined=jinja2.StrictUndefined).render(**context)


def render_template(path, **context):
    """
    Render jinja2 template file with given context
    """
    dirname, filename = os.path.split(os.path.abspath(path))
    loader = jinja2.FileSystemLoader(dirname)
    environment = jinja2.Environment(loader=loader, undefined=jinja2.StrictUndefined, trim_blocks=True, lstrip_blocks=True)
    template = environment.get_template(filename)
    rendered_template = template.render(**context)

    logger.info("Rendered %s:\n%s", path, rendered_template)

    with open(path, "w") as f:
        f.write(rendered_template)


def get_abspath(path):
    return lambda _: os.path.abspath(path)


def get_log_path():
    return lambda task: task.log_path().as_posix()


def get_environment_tag(environment):
    return "crypta/{}".format(environment).upper()


def get_possible_environment_tags():
    return [get_environment_tag(env) for env in RELEASE_STATUS]
