import datetime
import logging
import os
import six
import time

from textwrap import dedent

from sandbox import sdk2
from sandbox.common.types import task
from sandbox.projects.common.juggler import jclient
from sandbox.projects.crypta.common import (
    basic_task,
    graphite,
    helpers,
    resource_selector,
    vault
)
from sandbox.projects.crypta.common.crypta_api.crypta_task_status_api import CryptaTaskStatusApi
from sandbox.sdk2.helpers import subprocess


FAKE_HOSTNAME = {
    helpers.STABLE: "class01i.haze.yandex.net",
    helpers.TESTING: "class01it.crypta.yandex.net",
}
GRAPHITE_HOSTS = [
    "mega-graphite-man.search.yandex.net:2024",
    "mega-graphite-sas.search.yandex.net:2024"
]
JUGGLER_HOSTS = {
    helpers.STABLE: "crypta-sandbox",
    helpers.TESTING: "crypta-sandbox-testing",
}

logger = logging.getLogger(__name__)


class OutputResourceConfig(object):
    def __init__(self, type, path, attrs=None):
        self.type = type
        self.path = path
        self.attrs = attrs or {}

    def to_dict(self):
        return self.__dict__


class CryptaTask(basic_task.CryptaBasicTask):
    class Parameters(basic_task.CryptaBasicTask.Parameters):
        task_name = sdk2.parameters.String("Task name")

    class CryptaOptions(object):
        # Download and use resource of this type.
        bundle_resource_type = None

        # Render this files before execution phase.
        # If you want to generate templates dynamically you need to override
        # get_templates method of CryptaTask.
        templates = None

        # Run this command on execution phase.
        # If you want to generate command dynamically you need to override
        # get_cmd method of CryptaTask.
        cmd = None

        # Update common environment with this additional variables.
        # If you want to generate additional environment dynamically you
        # need to override get_additional_env method of CryptaTask.
        additional_env = None

        # Use or not semaphore for sequential run.
        # Sequential run option in scheduler's settings sometimes doesn't work.
        # If you want to generate semaphore name dynamically you need to override
        # get_semaphore_name method of CryptaTask.
        use_semaphore = True

        # Task name for backward compability with crypta graphite monitoring.
        # Task doesn't send metrics if it is None.
        # If you want to generate graphite monitoring task name dynamically you need to override
        # get_graphite_monitoring_task_name method of CryptaOptions.
        graphite_monitoring_task_name = None

        # if you need to register task status in crypta api (it registers in in Reactor, etc)
        report_status_to_crypta_api = False

        cmd_task_extras = None

        output_resources = None

        template_rendering_context = None

    def on_enqueue(self):
        if self.get_use_semaphore():
            logger.info("Acquiring semaphore %s", self._get_semaphore_name())
            self.Requirements.semaphores = task.Semaphores(
                acquires=[
                    task.Semaphores.Acquire(
                        self._get_semaphore_name(),
                        weight=1,
                        capacity=1
                    )
                ]
            )

    def on_prepare(self):
        self.send_metric("started", 1)

        bundle = self.get_bundle_resource()
        bundle_path = helpers.download_resource(bundle)
        helpers.untar_path(bundle_path)

        self.set_info(str(bundle))
        self.set_info(os.path.basename(bundle_path))

        self.Context.bundle = dict(
            type=str(bundle.type),
            id=bundle.id,
        )
        self.Context.save()

        for template in self.get_templates():
            self.render_template(self.render_template_str(template))

    @sdk2.report("Report")
    def bundle_report(self):
        if not self.Context.bundle:
            return "No bundle information available"

        bundle = self.Context.bundle

        return dedent("""
            <ul>
                <li>Using <code>{bundle_type}</code> (<a href="/resource/{bundle_id}">#{bundle_id}</a>)</li>
                <li>YQL operations filter (<a href="{yql_url}">{yql_mark}</a>)</li>
            </ul>
        """).format(
            bundle_type=bundle['type'],
            bundle_id=bundle['id'],
            yql_url=self._get_yql_filter_url(),
            yql_mark=self._get_yql_mark()
        )

    def on_execute(self):
        with sdk2.helpers.ProcessLog(self, logger="crypta_task") as pl:
            cmd = self.render_cmd(self.get_cmd())
            env = self.get_env()
            subprocess.check_call(cmd, shell=False, stdout=pl.stdout, stderr=pl.stderr, env=env)

            bundle_dict = resource_selector.get_bundle_dict()
            for config in self.get_output_resources():
                res = bundle_dict[config.type](self, "", config.path)
                for k, v in six.iteritems(config.attrs):
                    setattr(res, k, v)

    def get_crypta_api_params(self):
        """ Override to determine extra Crypta API parameters """
        return {}

    def on_save(self):
        self.Parameters.tags = [
            x for x in self.Parameters.tags
            if x.upper() not in helpers.get_possible_environment_tags()
        ]

        environment_tag = helpers.get_environment_tag(self.Parameters.environment)
        logger.info("Setting tag %s", environment_tag)
        self.Parameters.tags.append(environment_tag)

        if self.CryptaOptions.report_status_to_crypta_api:
            logger.info("Setting Crypta API parameters in context")
            params = {key: val for key, val in self.Parameters.user_crypta_api_params.items()}
            params["task_id"] = str(self.id)
            params.update(self.get_crypta_api_params())
            self.Context.crypta_api_params = params

    def on_success(self, prev_status):
        self.send_metric("done", 1)
        self.send_juggler_event(status="OK", description="OK")
        self.report_to_crypta_status_api("SUCCESS")

    def on_failure(self, prev_status):
        self.send_metric("failed", 1)
        self.report_to_crypta_status_api("FAILURE")

    def on_break(self, prev_status, status):
        self.send_metric("failed", 1)
        self.report_to_crypta_status_api("FAILURE")

    def send_metric(self, name, value):
        graphite_monitoring_task_name = self._get_graphite_monitoring_task_name()
        if graphite_monitoring_task_name is None:
            return

        hostname = FAKE_HOSTNAME[self.Parameters.environment]
        metric = graphite.one_min_metric("monitor", graphite_monitoring_task_name, name, hostname=hostname)
        metric_point = graphite.metric_point(metric, value, time.time())

        # TODO: creates api object per method call, because I have no idea how to make it a field in sdk2.Task
        graphite.Graphite(hosts=GRAPHITE_HOSTS).send([metric_point])
        self.set_info(str(metric_point))

    def send_juggler_event(self, status, description):
        juggler_host = self._get_juggler_host()
        juggler_service = self._get_juggler_service()

        if juggler_host is None or juggler_service is None:
            return

        jclient.send_events_to_juggler(juggler_host, juggler_service, status, description)
        self.set_info("Send juggler event. Host='{}', service='{}', status='{}', description='{}'".format(juggler_host, juggler_service, status, description))

    def report_to_crypta_status_api(self, status):
        if not self.CryptaOptions.report_status_to_crypta_api:
            return

        params = self.Context.crypta_api_params
        task_type = self.type.name

        try:
            # TODO: creates api object per method call, because I have no idea how to make it a field in sdk2.Task
            crypta_task_api = CryptaTaskStatusApi("no_oauth_yet", self.Parameters.environment)
            crypta_task_api.report_task_status(task_type, status, [], params)
        except:
            message = "Unable to report task '%s' status %s to crypta api" % (task_type, status)
            logging.exception(message)
            self.set_info(message)

    def get_env(self):
        env = {
            "YT_TOKEN": vault.get_yt_token(self.Parameters.environment),
            "YQL_TOKEN": vault.get_yql_token(self.Parameters.environment),
            "MR_RUNTIME": "YT",
            "YT_PREFIX": "//",
            "YQL_MARK": self._get_yql_mark(),
            "SOLOMON_TOKEN": vault.get_solomon_token(),
        }
        additional_env = self.get_additional_env()
        assert isinstance(additional_env, dict), "Additional environment must be instance of dict"
        env.update({k: v(self) if callable(v) else v for k, v in six.iteritems(additional_env)})
        return env

    def get_bundle_resource(self):
        if self.Parameters.override_basic_params and self.Parameters.overridden_bundle_resource:
            return self.Parameters.overridden_bundle_resource
        else:
            return helpers.get_last_released_resource(self.CryptaOptions.bundle_resource_type,
                                                      self.Parameters.environment)

    def get_cmd(self):
        cmd_items = self.CryptaOptions.cmd
        if self.CryptaOptions.cmd_task_extras:
            cmd_items += self.CryptaOptions.cmd_task_extras.get(self.Parameters.task_name, [])
        return cmd_items

    def render_cmd(self, cmd_items):
        return [self.render_template_str(item(self) if callable(item) else item) for item in cmd_items]

    def get_additional_env(self):
        return self.CryptaOptions.additional_env or {}

    def get_templates(self):
        return self.CryptaOptions.templates or []

    def get_template_rendering_context(self):
        return self.CryptaOptions.template_rendering_context or {}

    def render_template(self, template):
        helpers.render_template(template, environment=self.Parameters.environment, **self.get_template_rendering_context())

    def render_template_str(self, template):
        context = {
            "environment": self.Parameters.environment,
            "task_name": self.Parameters.task_name,
            "cwd": os.getcwd(),
            "logdir": str(self.log_path()),
        }
        return helpers.render_template_str(template, **context)

    def _get_semaphore_name(self):
        if self.Parameters.override_basic_params:
            return self.Parameters.overridden_semaphore
        else:
            return self.get_semaphore_name()

    def get_semaphore_name(self):
        return CryptaTask._join_not_empty(self.__class__.__name__, self.Parameters.environment, self.Parameters.task_name)

    def _get_graphite_monitoring_task_name(self):
        if self.Parameters.override_basic_params:
            return self.Parameters.overridden_monitor if self.Parameters.override_use_monitor else None
        else:
            return self.get_graphite_monitoring_task_name()

    def get_graphite_monitoring_task_name(self):
        return CryptaTask._join_not_empty(self.CryptaOptions.graphite_monitoring_task_name, self.Parameters.task_name)

    def _get_juggler_service(self):
        if self.Parameters.override_basic_params:
            return self.Parameters.override_juggler_service
        else:
            return self.get_juggler_service()

    def get_juggler_service(self):
        return self.get_class_juggler_service(self.Parameters.task_name)

    @classmethod
    def get_class_juggler_service(cls, task_name=None):
        return CryptaTask._join_not_empty(cls.__name__, task_name)

    def get_use_semaphore(self):
        return self.CryptaOptions.use_semaphore

    def get_output_resources(self):
        return self.CryptaOptions.output_resources or {}

    def _get_juggler_host(self):
        if self.Parameters.override_basic_params:
            return self.Parameters.override_juggler_host
        else:
            return JUGGLER_HOSTS[self.Parameters.environment]

    def _get_yql_filter_url(self):
        return (
            'https://yt.yandex-team.ru/hahn/operation?'
            'filter={yql_mark}&from={task_start_time}&to={task_end_time}'
            '&dataMode=archive&user=all&type=all&state=all'
        ).format(
            yql_mark=self._get_yql_mark(),
            task_start_time=self.created.strftime('%s'),
            task_end_time=(self.updated + datetime.timedelta(days=1)).strftime('%s'),
        )

    def _get_yql_mark(self):
        return 'crypta-yql-sndbx-{task_id}'.format(task_id=self.id)

    @staticmethod
    def _join_not_empty(*vargs):
        return "_".join(str(arg) for arg in vargs if (arg is not None) and (arg != ""))


class CryptaBaseYqlTask(CryptaTask):

    """ Base yql task """

    TASK = ""
    BIN_FILE_NAME = ""
    YT_POOL = None

    class Parameters(CryptaTask.Parameters):
        task = sdk2.parameters.String("Task name", required=False)
        task_args = sdk2.parameters.Dict("Task parameters")
        conf_args = sdk2.parameters.Dict("Conf parameters")
        pool = sdk2.parameters.String("YT pool", required=False)

    def task_parameters(self):
        for (key, value) in (self.Parameters.task_args or {}).items():
            yield "--arg"
            yield "{}={}".format(key, value)

    def conf_parameters(self):
        for (key, value) in (self.Parameters.conf_args or {}).items():
            yield "--{}".format(key)
            yield str(value)

    def get_additional_env(self):
        env_type = self.Parameters.environment.replace("stable", "production").upper()
        pool = self.Parameters.pool if self.Parameters.pool else self.YT_POOL
        additional_env = {"CRYPTA_ENVIRONMENT": env_type}
        if pool is not None:
            additional_env.update(YT_POOL=pool)
        return additional_env

    def get_cmd(self):
        binary_path = os.path.abspath(self.BIN_FILE_NAME)
        cmd = [binary_path, ] + list(self.conf_parameters()) + [
            "run",
            "--task",
            self.Parameters.task or self.TASK
        ] + list(self.task_parameters())
        return cmd
