import yaml

from sandbox import sdk2
from sandbox.common.types import misc
from sandbox.projects.crypta.common import (
    helpers,
    task
)
from sandbox.projects.crypta.dmp.adobe.common import bundle


INDEX_FILENAME = "index.yaml"
CONFIG_FILENAME = "config.yaml"


class CryptaAdobeTask(task.CryptaTask):
    class Requirements(task.CryptaTask.Requirements):
        dns = misc.DnsType.DNS64

    class CryptaOptions(task.CryptaTask.CryptaOptions):
        bundle_resource_type = bundle.CryptaAdobeBundle

    class Parameters(task.CryptaTask.Parameters):
        customer = sdk2.parameters.String(
            "Adobe customer",
            required=True
        )
        ext_id_type = sdk2.parameters.String(
            "Ext id type",
            required=True
        )

    def get_semaphore_name(self):
        return "{}_{}_{}".format(super(CryptaAdobeTask, self).get_semaphore_name(), self.Parameters.customer, self.Parameters.ext_id_type)

    def get_juggler_service(self):
        return self.get_juggler_service_for_customer(self.Parameters.customer, self.Parameters.ext_id_type)

    def get_graphite_monitoring_task_name(self):
        return "adobe__{}__{}__{}".format(self.__class__.__name__,  self.Parameters.customer, self.Parameters.ext_id_type)

    def on_prepare(self):
        super(CryptaAdobeTask, self).on_prepare()

        self.render_template(INDEX_FILENAME)
        with open(INDEX_FILENAME) as f:
            index = yaml.load(f)

        context = get_context(index, self.Parameters.customer, self.Parameters.ext_id_type)
        helpers.render_template(CONFIG_FILENAME, environment=self.Parameters.environment, **context)

    @classmethod
    def get_juggler_service_for_customer(cls, customer, ext_id_type):
        return "{}_{}_{}".format(cls.get_class_juggler_service(), customer, ext_id_type)


def get_context(index, customer, ext_id_type):
    for context in index:
        if context["customer"] == customer and context["ext_id_type"] == ext_id_type:
            context["acl"] = yaml.dump(context["acl"]).strip("\n")
            context["hierarchy"] = yaml.dump(context["hierarchy"]).strip("\n")
            return context

    raise Exception("Customer {} with ext id type {} not found in index".format(customer, ext_id_type))
