# -*- coding: utf-8 -*-
from __future__ import print_function

import os

from sandbox import sdk2

from sandbox.projects.crypta.common import helpers
from sandbox.projects.crypta.common import vault
from sandbox.projects.crypta.common.task import CryptaTask


class CryptaGraphBochkaBundle(sdk2.Resource):
    releasable = True


kGB = 1 << 10  # megabytes in gigabyte
kHOUR = 60 * 60  # seconds in hour


kTASKS = (
    ("Yt2BB", "crypta.graph.bochka.lib.tasks.Yt2BBPushTask"),
    ("Yt2BBProfileCleaner", "crypta.graph.bochka.lib.tasks.Yt2BBProfileCleanerTask"),
    ("Yt2RTMR", "crypta.graph.bochka.lib.tasks.Yt2RTMRPushTask"),
    ("Soup2LB", "crypta.graph.bochka.lib.tasks.YtSoup2LBPushTask"),
    ("IdStorage2LB", "crypta.graph.bochka.lib.tasks.YtIdsStorage2LBPushTask"),
    ("Soup2Vulture", "crypta.graph.bochka.lib.tasks.YtSoup2VultureTask"),
    ("not set", ""),
)


class CryptaGraphBochka(CryptaTask):

    """ Run upload from yt to logbroker in sandbox """

    class CryptaOptions(CryptaTask.CryptaOptions):

        bundle_resource_type = CryptaGraphBochkaBundle
        use_semaphore = True
        report_status_to_crypta_api = True

    class Parameters(CryptaTask.Parameters):

        kill_timeout = 5 * kHOUR

        task = sdk2.parameters.String("Task", required=False, choices=kTASKS, default=kTASKS[-1][1])
        pool = sdk2.parameters.String("YT pool", required=False)
        source_path = sdk2.parameters.String("Source path", required=False)
        lb_topic = sdk2.parameters.String("LB Topic", required=False)
        options = sdk2.parameters.Dict("Additional params for run binary", required=False)
        args = sdk2.parameters.Dict("Additional args for run task", required=False)
        config = sdk2.parameters.String("Config file pb.txt (will override all params)")

    def get_cmd(self):
        cmd = [os.path.abspath("crypta-bochka")]

        if self.Parameters.config:
            cmd.extend(["--config", os.path.abspath(self.Parameters.config.lstrip("/"))])

        if self.Parameters.lb_topic:
            cmd.extend(["--lb-topic-topic-name", self.Parameters.lb_topic])

        for key, value in self.Parameters.options.items():
            cmd.extend([key, value])

        cmd.extend(["run"])

        if self.Parameters.task:
            cmd.extend(["--task", self.Parameters.task])

        if self.Parameters.source_path:
            cmd.extend(["--arg", "source_path={0}".format(self.Parameters.source_path)])

        for key, value in self.Parameters.args.items():
            cmd.extend(["--arg", "{0}={1}".format(key, value)])

        return cmd

    def get_semaphore_name(self):
        return "{cls}_{env}_{path}".format(
            cls=self.__class__.__name__, env=self.Parameters.environment, path=self.source_table_name
        )

    def get_juggler_service(self):
        return None

    def get_additional_env(self):
        env_type = self.Parameters.environment
        return {
            "YT_POOL": self.yt_pool,
            "CRYPTA_ENVIRONMENT": env_type.replace("stable", "production").upper(),
            # tvm id 2012336
            "CRYPTA_YT2LB_TVM_SECRET": vault.get_vault_item("sec-01dq7mh7039z77mgws7sds4t0a[client_secret]"),
        }

    @property
    def yt_pool(self):
        if self.Parameters.pool:
            return self.Parameters.pool
        return {helpers.STABLE: "crypta_graph", helpers.TESTING: "crypta_graph_testing"}[self.Parameters.environment]

    @property
    def source_table_name(self):
        if self.Parameters.config:
            return self.Parameters.config.split(".")[0].replace("/", "_")
        return self.Parameters.source_path.split("/")[-1]
