import os

from sandbox import sdk2
from sandbox.projects.crypta.common.task import CryptaTask
from sandbox.projects.crypta.common.vault import get_vault_item, get_reactor_token


class CryptaSoupyIndeviceBundle(sdk2.Resource):
    releasable = True
    ttl_on_release = 30


class CryptaSoupyIndeviceMetricsBundle(sdk2.Resource):
    releasable = True
    ttl_on_release = 30


class Paths(object):
    def __init__(self, env_type):
        if env_type == "stable":
            self.soup_dir = "//home/crypta/production/state/graph/v2/soup"
            self.stream_soup_dir = "//home/crypta/production/state/graph/stream/soup"
            self.output_dir = "//home/crypta/production/state/graph/v2/indevice"
            self.idstorage_dir = "//home/crypta/production/ids_storage"
        else:
            self.soup_dir = "//home/crypta/testing/state/graph/v2/soup"
            self.stream_soup_dir = "//home/crypta/testing/state/graph/stream/soup"
            self.output_dir = "//home/crypta/testing/state/graph/v2/indevice"
            self.idstorage_dir = "//home/crypta/testing/ids_storage"


class CryptaSoupyIndevice(CryptaTask):
    class CryptaOptions(CryptaTask.CryptaOptions):
        bundle_resource_type = CryptaSoupyIndeviceBundle
        use_semaphore = False
        report_status_to_crypta_api = True

    class Parameters(CryptaTask.Parameters):
        keep = sdk2.parameters.Integer("Number of past indevices to keep", default=4)
        resume_for_date = sdk2.parameters.String("Resume an aborted task for a given date", default=None)
        tags = ["soupy_indevice", "task"]
        luigid_url = sdk2.parameters.String("Luigi scheduler url", default=None)
        sparse_run = sdk2.parameters.Bool("Sparse run (only at every odd day)", default_value=True)

    def get_additional_env(self):
        step_token = ""
        if self.Parameters.environment == "stable":
            step_token = get_vault_item("sec-01csvzga1x7wnhpvfdcscheeax[token]")

        luigid_url = (
            self.Parameters.luigid_url
            or {  # W503 # noqa
                "stable": "https://luigi.production.graph.crypta.yandex-team.ru:443",
                "testing": "https://luigi.testing.graph.crypta.yandex-team.ru:443",
            }[self.Parameters.environment]
        )

        return {
            "CRYPTA_ENVIRONMENT": self.Parameters.environment.replace("stable", "production").upper(),
            "ENVIRONMENT": self.Parameters.environment,
            "LUIGID_URL": luigid_url,
            "REACTOR_TOKEN": get_reactor_token(),
            "STATBOX_STEP_OAUTH_TOKEN": step_token,
            "YT_POOL": "crypta_graph_indevice" if self.Parameters.environment == "stable" else "crypta_graph_testing",
            "YT_PROXY": "hahn.yt.yandex.net",
        }

    def get_cmd(self):
        env_type = self.Parameters.environment
        paths = Paths(env_type)

        cmd = [
            os.path.abspath("soupy_indevice"),
            "--soup-dir",
            paths.soup_dir,
            "--output-dir",
            paths.output_dir,
            "--idstorage-dir",
            paths.idstorage_dir,
            "--keep",
            str(self.Parameters.keep),
        ]

        if self.Parameters.sparse_run:
            cmd += ["--sparse-run"]

        if self.Parameters.resume_for_date:
            cmd += ["--resume-for-date", self.Parameters.resume_for_date]

        return cmd


class CryptaSoupyIndeviceMetrics(CryptaTask):
    class CryptaOptions(CryptaTask.CryptaOptions):
        bundle_resource_type = CryptaSoupyIndeviceMetricsBundle
        use_semaphore = False
        report_status_to_crypta_api = True

    class Parameters(CryptaTask.Parameters):

        kind_choices = [("Default", "default"), ("Fast", "fast")]

        date = sdk2.parameters.String("Indevice date to calculate metrics for")
        kind = sdk2.parameters.String(
            "Type of indevice matching to calculate metrics for", default="default", choices=kind_choices
        )
        tags = ["soupy_indevice", "metrics"]

    def get_additional_env(self):
        statface_token = ""
        # yql_token = ""
        if self.Parameters.environment == "stable":
            statface_token = get_vault_item("sec-01daxg3xp40d5pr2s52sb22pkm[secret]")

        return {
            "CRYPTA_ENVIRONMENT": self.Parameters.environment.replace("stable", "production").upper(),
            "ENVIRONMENT": self.Parameters.environment,
            "STATFACE_OAUTH": statface_token,
            "YT_POOL": "crypta_graph",
            "YT_PROXY": "hahn.yt.yandex.net",
        }

    def get_cmd(self):
        # env_type = self.Parameters.environment

        cmd = [os.path.abspath("statface")]

        if self.Parameters.kind != "default":
            cmd.append(self.Parameters.kind)

        cmd += ["--date", self.Parameters.date]

        return cmd


class CryptaSoupyIndeviceLinkCounts(CryptaSoupyIndevice):

    """ Calculate link count percentiles on the whole soup """

    def get_cmd(self):
        cmd = super(CryptaSoupyIndeviceLinkCounts, self).get_cmd()
        cmd.append("--link-count-perc")
        return cmd


class CryptaSoupyIndeviceFast(CryptaSoupyIndevice):
    def get_cmd(self):
        cmd = super(CryptaSoupyIndeviceFast, self).get_cmd()
        paths = Paths(self.Parameters.environment)
        cmd += ["--fast-match", "--stream-soup-root", paths.stream_soup_dir, "--full-indevice-root", paths.output_dir]

        return cmd


class CryptaSoupyIndeviceTimedCheckFast(CryptaTask):
    class CryptaOptions(CryptaTask.CryptaOptions):
        bundle_resource_type = CryptaSoupyIndeviceBundle
        use_semaphore = False
        report_status_to_crypta_api = True

    class Parameters(CryptaTask.Parameters):
        hour = sdk2.parameters.Integer(
            "Hour of the day by which the output table should have been created", default=12
        )

    def get_additional_env(self):
        return {
            "CRYPTA_ENVIRONMENT": self.Parameters.environment.replace("stable", "production").upper(),
            "ENVIRONMENT": self.Parameters.environment,
            "YT_POOL": "crypta_graph_indevice" if self.Parameters.environment == "stable" else "crypta_graph_testing",
            "YT_PROXY": "hahn.yt.yandex.net",
        }

    def get_cmd(self):
        paths = Paths(self.Parameters.environment)

        cmd = [
            os.path.abspath("soupy_indevice"),
            "--output-dir",
            paths.output_dir + "/fast",
            "--check-output-table-exists-by",
            str(self.Parameters.hour),
        ]

        return cmd
