import logging
import os

from sandbox import sdk2
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.projects.crypta.common import helpers
from sandbox.projects.crypta.common.helpers import TESTING
from sandbox.projects.crypta.common.task import CryptaTask
from sandbox.projects.crypta.graph.matching import get_matching_env

custom_tmp_config_file = '_custom_config.yaml'


class CryptaGraphHumanMatchingBuild(sdk2.Resource):
    releasable = True


class CryptaGraphHumanMatchingYaml(CryptaTask):
    class Requirements(CryptaTask.Requirements):
        environments = [SandboxJavaJdkEnvironment('17.0.2')]

    class CryptaOptions(CryptaTask.CryptaOptions):
        bundle_resource_type = CryptaGraphHumanMatchingBuild
        graphite_monitoring_task_name = "crypta-graph-human-matching"
        report_status_to_crypta_api = True

    class Parameters(CryptaTask.Parameters):
        kill_timeout = 60 * 60 * 48  # 48 hours

        config_choices = [
            ('config-prod.yaml', 'config-prod.yaml'),
            ('config-prod2.yaml', 'config-prod2.yaml'),
            ('config-prestable.yaml', 'config-prestable.yaml'),
            ('config-sample-random.yaml', 'config-sample-random.yaml'),
            ('Custom YAML config...', custom_tmp_config_file)
        ]
        config_file = sdk2.parameters.String("Config file", required=True,
                                             choices=config_choices,
                                             default=config_choices[0][1])

        with config_file.value[custom_tmp_config_file]:
            custom_yaml_config = sdk2.parameters.String("Custom config text", multiline=True)

        copy_to_prod = sdk2.parameters.Bool("Copy to dicts", required=True, default=True)

        is_experiment = sdk2.parameters.Bool("Is experiment", required=True, default=False)

        run_id = sdk2.parameters.String("Run ID", default_value="")

        with is_experiment.value[True]:
            experiment_name = sdk2.parameters.String("Name of experiment")

        wait_sec = sdk2.parameters.Integer('Delay task execution in seconds', required=True, default=0)

    def get_cmd(self):
        cmd = [
            os.path.abspath("run.sh"),
            "--config", self.Parameters.config_file,
            "--copyToDict", str(self.Parameters.copy_to_prod),
        ]
        if self.Parameters.is_experiment:
            cmd.extend([
                "--isExperiment", str(self.Parameters.is_experiment),
                "--experimentName", self.Parameters.experiment_name
            ])
        if len(str(self.Parameters.run_id)) > 0:
            cmd.extend([
                "--runId", str(self.Parameters.run_id)
            ])

        return cmd

    def get_additional_env(self):
        return get_matching_env(self.Parameters.environment, self.Parameters.is_experiment)

    def get_semaphore_name(self):
        semaphore_name = super(CryptaGraphHumanMatchingYaml, self).get_semaphore_name()
        suffix = "exp" if self.Parameters.is_experiment else "prod"
        return "{}_{}".format(semaphore_name, suffix)

    def get_bundle_resource(self):
        if self.Parameters.override_basic_params and self.Parameters.overridden_bundle_resource:
            return self.Parameters.overridden_bundle_resource
        elif self.Parameters.is_experiment:
            # always prefer testing build in experiment
            return helpers.get_last_released_resource(self.CryptaOptions.bundle_resource_type,
                                                      TESTING)
        else:
            return super(CryptaGraphHumanMatchingYaml, self).get_bundle_resource()

    def get_crypta_api_params(self):
        return {
            "is_experiment": self.Parameters.is_experiment,
        }

    def get_juggler_service(self):
        suffix = "_exp" if self.Parameters.is_experiment else ""
        return "{}{}".format(super(CryptaGraphHumanMatchingYaml, self).get_juggler_service(), suffix)

    def on_prepare(self):
        if self.Parameters.custom_yaml_config:
            with open(os.path.abspath(custom_tmp_config_file), 'w') as f:
                f.write(self.Parameters.custom_yaml_config)

        super(CryptaGraphHumanMatchingYaml, self).on_prepare()

    def on_enqueue(self):
        super(CryptaGraphHumanMatchingYaml, self).on_enqueue()

        if self.Parameters.wait_sec:
            with self.memoize_stage.wait_once(commit_on_entrance=False):
                logging.info("Sleep for %d seconds to delimit YT resources" % self.Parameters.wait_sec)
                raise sdk2.WaitTime(self.Parameters.wait_sec)
