import os

from sandbox import sdk2
from sandbox.projects.crypta.common.task import CryptaTask
from sandbox.projects.crypta.common.vault import get_vault_item


CUSTOM_METRICS_YAML_FILE = "custom_metrics.yaml"


class CryptaGraphHumanMatchingPublishBuild(sdk2.Resource):
    releasable = True


class CryptaGraphHumanMatchingPublish(CryptaTask):
    def get_additional_env(self):
        env_type = self.Parameters.environment.replace("stable", "production").upper()
        stat_token = get_vault_item("sec-01daxg3xp40d5pr2s52sb22pkm[secret]")

        return {"ENV_TYPE": env_type, "STATFACE_OAUTH": stat_token}

    class CryptaOptions(CryptaTask.CryptaOptions):
        bundle_resource_type = CryptaGraphHumanMatchingPublishBuild
        report_status_to_crypta_api = True

    class Parameters(CryptaTask.Parameters):
        kill_timeout = 60 * 60 * 4  # 4 hours

        yt_proxy = sdk2.parameters.String("YT proxy", default_value="hahn.yt.yandex.net")
        yt_pool = sdk2.parameters.String("YT pool", default_value="crypta_graph_human_matching")

        input = sdk2.parameters.String("Input YT path to directory", required=True)
        output = sdk2.parameters.String("Output YT path to directory", required=True)
        history_dir = sdk2.parameters.String(
            "History dir YT path",
            default_value="//home/crypta/archive/graph/vertices_no_multi_profile")
        skip_checks = sdk2.parameters.Bool("Skip checks", required=True, default=False)

        metrics_choices = [
            ("metrics.yaml", "metrics.yaml"),
            ("Custom metrics YAML ...", CUSTOM_METRICS_YAML_FILE),
        ]

        metrics_yaml = sdk2.parameters.String("Metrics config file",
                                              choices=metrics_choices,
                                              default=metrics_choices[0][1])

        with metrics_yaml.value[CUSTOM_METRICS_YAML_FILE]:
            custom_metrics_yaml_config = sdk2.parameters.String("Custom metrics config", multiline=True)

    def get_cmd(self):
        cmd = [
            os.path.abspath("publish"),
            "--Yt.Proxy", self.Parameters.yt_proxy,
            "--Yt.Pool", self.Parameters.yt_pool,
            "--Input", self.Parameters.input,
            "--Output", self.Parameters.output,
            "--HistoryDir", self.Parameters.history_dir,
        ]

        if self.Parameters.skip_checks:
            cmd.append("--SkipChecks")

        if self.Parameters.metrics_yaml:
            cmd.extend(["--MetricsYaml", self.Parameters.metrics_yaml, ])

        return cmd

    def on_prepare(self):
        if self.Parameters.custom_metrics_yaml_config:
            with open(os.path.abspath(CUSTOM_METRICS_YAML_FILE), "w") as f:
                f.write(self.Parameters.custom_metrics_yaml_config)

        super(CryptaGraphHumanMatchingPublish, self).on_prepare()

    def on_failure(self, prev_status):
        super(CryptaGraphHumanMatchingPublish, self).on_failure(prev_status)
        self.send_juggler_event(status="CRIT", description="Failed quality check")

    def on_break(self, prev_status, status):
        super(CryptaGraphHumanMatchingPublish, self).on_break(prev_status, status)
        self.send_juggler_event(status="CRIT", description="Didn't pass quality check")
