# -*- coding: utf-8 -*-
from __future__ import print_function
import os

from sandbox import sdk2
from sandbox.common.types import task as sndbx_task
from sandbox.projects.crypta.common import vault, task


class CryptaGraphRtsklejkaBundle(sdk2.Resource):
    releasable = True


kHOUR = 60 * 60  # seconds in hour

kRTSKLEJKA_IMPORT_CHOICES = [("CryptaIdTable", "crypta.graph.rtsklejka.adhoc.lib.tasks.CreateCryptaIdTableTask")]


class CryptaGraphRtsklejka(task.CryptaTask):

    """ Run import part of stream logs """

    class CryptaOptions(task.CryptaTask.CryptaOptions):

        bundle_resource_type = CryptaGraphRtsklejkaBundle
        use_semaphore = True
        report_status_to_crypta_api = True

    class Parameters(task.CryptaTask.Parameters):

        kill_timeout = int(3 * kHOUR)
        tags = ["rtsklejka", "rtcrypta"]

        pool = sdk2.parameters.String("YT pool", required=False, default="crypta_graph")
        task = sdk2.parameters.String("Task", required=True, choices=kRTSKLEJKA_IMPORT_CHOICES)
        options = sdk2.parameters.Dict("Additional params for run binary", required=False)
        extra_args = sdk2.parameters.Dict(
            "Additional args for bt task (will be added to cmd as [--arg, key=value])", required=False
        )

    def get_cmd(self):
        cmd = [os.path.abspath("resharder-utils")]

        for key, value in self.Parameters.options.items():
            cmd.extend([key, value] if value else [key])

        cmd.extend(["run", "--task", self.Parameters.task])

        for key, value in self.Parameters.extra_args.items():
            cmd.extend(["--arg", "{key}={value}".format(key=key, value=value)])

        return cmd

    def get_semaphore_name(self):
        return "{cls}_{env}_{task}".format(
            cls=self.__class__.__name__, env=self.Parameters.environment, task=self.Parameters.task
        )

    def on_enqueue(self):
        base_semaphore = self.get_semaphore_name()
        acquires = []
        acquires.append(sndbx_task.Semaphores.Acquire(base_semaphore, weight=1, capacity=1))
        self.Requirements.semaphores = sndbx_task.Semaphores(acquires=acquires)

    @classmethod
    def get_juggler_service_for_task(cls, task):
        service = "{}_{}".format(cls.get_class_juggler_service(), task.split(".")[-1])
        return service

    def get_juggler_service(self):
        return self.get_juggler_service_for_task(self.Parameters.task)

    def get_additional_env(self):
        env_type = self.Parameters.environment
        return {
            "YT_POOL": self.yt_pool,
            "CRYPTA_ENVIRONMENT": env_type.replace("stable", "production"),
            # override with robot-crypta tokens (because for some logs no access for robot-crypta-testing)
            # and no rights to mount dynamic tables for robot-crypta-testing
            # TODO: got correct access and rm next 2 line
            "YT_TOKEN": vault.get_yt_token("stable"),
            "YQL_TOKEN": vault.get_yql_token("stable"),
        }

    @property
    def yt_pool(self):
        return self.Parameters.pool
