# -*- coding: utf-8 -*-
from __future__ import print_function
import os
import json

from sandbox import sdk2
from sandbox.common import rest
from sandbox.common.proxy import OAuth
from sandbox.common.types import task as sndbx_task
from sandbox.projects.crypta.common import helpers, vault, task


class CryptaGraphDataImportStreamBundle(sdk2.Resource):
    releasable = True
    ttl_on_release = 30


kHOUR = 60 * 60  # seconds in hour

# NOTE: add juggler checks crypta/graph/monitorings/aggregate_checks
kSTREAM_IMPORT_CHOICES = [
    ("AccessLog", "crypta.graph.data_import.access_log.lib.AccessLogImportTask"),
    ("AppMetrika", "crypta.graph.data_import.app_metrica_day.lib.task.AppMetrikaTask"),
    ("BarNavigLog", "crypta.graph.data_import.bar_navig_log.lib.BarNavigImportTask"),
    ("ExportAccessLog", "crypta.graph.data_import.export_access_log.lib.EalImportTask"),
    ("PostbackLogTask", "crypta.graph.data_import.postback_log.lib.task.PostbackLogTask"),
    ("RTBLog", "crypta.graph.data_import.rtb_log.lib.task.RTBLogTask"),
    ("RedirLog", "crypta.graph.data_import.redir_log.lib.RedirLogImportTask"),
    ("WatchLog", "crypta.graph.data_import.watch_log.lib.WatchLogImportTask"),
    ("AddDaySoup", "crypta.graph.data_import.soup.lib.task.SoupTask"),
]


kSTREAM_IMPORT = {
    "stable": [
        "AccessLog",
        "AppMetrika",
        "ExportAccessLog",
        "PostbackLogTask",
        "RedirLog",
        "RTBLog",
        "WatchLog",
        "BarNavigLog",
    ],
    "testing": [
        "AccessLog",
        "AppMetrika",
        "ExportAccessLog",
        "PostbackLogTask",
        "RedirLog",
        "RTBLog",
        "WatchLog",
        "BarNavigLog",
    ],
}


class CryptaGraphDataImportStream(task.CryptaTask):

    """ Run import part of stream logs """

    class CryptaOptions(task.CryptaTask.CryptaOptions):

        bundle_resource_type = CryptaGraphDataImportStreamBundle
        use_semaphore = True
        report_status_to_crypta_api = True

    class Parameters(task.CryptaTask.Parameters):

        kill_timeout = int(7 * kHOUR)
        tags = ["data-import", "stream", "rtcrypta"]

        pool = sdk2.parameters.String("YT pool", required=False)
        task = sdk2.parameters.String("Task", required=True, choices=kSTREAM_IMPORT_CHOICES)
        source = sdk2.parameters.String("Source paths (json list)", required=False, multiline=True)
        options = sdk2.parameters.Dict("Additional params for run binary", required=False)
        extra_args = sdk2.parameters.Dict(
            "Additional args for bt task (will be added to cmd as [--arg, key=value])", required=False
        )

    def get_cmd(self):
        binary_path = os.path.abspath("crypta-data-import-stream")
        cmd = [binary_path]

        for key, value in self.Parameters.options.items():
            cmd.extend([key, value] if value else [key])

        cmd.extend(["--yql-embedded-is-embedded"])
        cmd.extend(["run", "--task", self.Parameters.task])

        if self.Parameters.source:
            assert isinstance(json.loads(self.Parameters.source), list), "Source table should be a json list"
            cmd.extend(["--arg", "source_tables={value}".format(value=self.Parameters.source)])

        for key, value in self.Parameters.extra_args.items():
            cmd.extend(["--arg", "{key}={value}".format(key=key, value=value)])

        return cmd

    def get_semaphore_name(self):
        return "{cls}_{env}_{task}".format(
            cls=self.__class__.__name__, env=self.Parameters.environment, task=self.Parameters.task
        )

    def on_enqueue(self):
        base_semaphore = self.get_semaphore_name()
        acquires = []
        if self.Parameters.task == dict(kSTREAM_IMPORT_CHOICES)["AddDaySoup"]:
            for log_source in self.get_log_sources():
                acquires.append(
                    sndbx_task.Semaphores.Acquire(
                        "{base}_{source}".format(base=base_semaphore, source=log_source.lower().strip()),
                        weight=1,
                        capacity=1,
                    )
                )
        else:
            acquires.append(sndbx_task.Semaphores.Acquire(base_semaphore, weight=1, capacity=1))
        self.Requirements.semaphores = sndbx_task.Semaphores(acquires=acquires)

    def get_log_sources(self):
        return self.Parameters.extra_args.get("log_sources", "all").split(",")

    @classmethod
    def get_juggler_service_for_task(cls, task, log_sources):
        service = "{}_{}".format(cls.get_class_juggler_service(), task.split(".")[-1])
        if task.endswith("SoupTask"):
            service = "{}_{}".format(service, "_".join(log_sources))

        return service

    def get_juggler_service(self):
        return self.get_juggler_service_for_task(self.Parameters.task, self.get_log_sources())

    def get_additional_env(self):
        env_type = self.Parameters.environment
        return {
            "YT_POOL": self.yt_pool,
            "CRYPTA_ENVIRONMENT": env_type.replace("stable", "production"),
            # override with robot-crypta tokens (because for some logs no access for robot-crypta-testing)
            # and no rights to mount dynamic tables for robot-crypta-testing
            # TODO: got correct access and rm next 2 line
            "YT_TOKEN": vault.get_yt_token("stable"),
            "YQL_TOKEN": vault.get_yql_token("stable"),
            "METRIKA_PRIVATE_KEY": vault.get_vault_item("sec-01csvzeyhdgwmk5dq5bsntedh6[secret]"),  # for wl sockets
        }

    @property
    def yt_pool(self):
        if self.Parameters.pool:
            return self.Parameters.pool
        return {helpers.STABLE: "crypta_graph_data_import_stream", helpers.TESTING: "crypta_graph_testing"}[
            self.Parameters.environment
        ]


class CryptaGraphDataImportRunThemAll(task.CryptaTask):

    """ The one task to run all importers """

    class CryptaOptions(task.CryptaTask.CryptaOptions):

        bundle_resource_type = None
        use_semaphore = True
        report_status_to_crypta_api = True

    class Parameters(task.CryptaTask.Parameters):

        kill_timeout = 900  # 15 min
        tags = ["data-import", "stream", "rtcrypta"]
        skip_run = sdk2.parameters.String("Skip run task (csv)", required=False)

    def on_prepare(self):
        self.send_metric("started", 1)
        # no download bundle resource

    def on_execute(self):
        sandbox_oauth = vault.get_snbx_token("stable")  # self.Parameters.environment)
        client = rest.Client(auth=OAuth(sandbox_oauth))
        task_ids = []
        import_tasks = kSTREAM_IMPORT[self.Parameters.environment]
        task_details = [x for x in kSTREAM_IMPORT_CHOICES if x[0] in import_tasks]
        for task_name, task_path in task_details:
            if task_name in self.Parameters.skip_run.split(","):
                continue

            task_id = client.task({"type": "CRYPTA_GRAPH_DATA_IMPORT_STREAM"})["id"]
            extra_args = {
                # skip_stream_on_testing - take only 1/10 of logs on testing (default: True)
                # enable_clout - use tentative cloud tree (default: True)
            }

            client.task[task_id] = {
                "owner": "CRYPTA",
                "tags": self.Parameters.tags + ["task/{task_name}".format(task_name=task_name)],
                "description": "Run task {task_name}".format(task_name=task_name),
                "max_restarts": 3,
                "custom_fields": [
                    {"name": "environment", "value": self.Parameters.environment},
                    {"name": "task", "value": task_path},
                    {"name": "extra_args", "value": extra_args},
                ],
                "notifications": [
                    {
                        "transport": "email",
                        "recipients": ["crypta-cryptaid@yandex-team.ru"],
                        "statuses": ["EXCEPTION", "EXPIRED", "FAILURE", "TIMEOUT"],
                    }
                ],
            }
            task_ids.append(task_id)

        response = client.batch.tasks.start.update(task_ids)
        for sub_task in response:
            assert sub_task["status"] == "SUCCESS", "Sandbox should start with success status!"
