# -*- coding: utf-8 -*-
import datetime
import logging
import os
import pytz

from sandbox import sdk2

from sandbox.projects.crypta.common import helpers
from sandbox.projects.crypta.common import vault
from sandbox.projects.crypta.common.task import CryptaTask

logger = logging.getLogger(__name__)


class CryptaGraphV1Bundle(sdk2.Resource):
    releasable = True
    ttl_on_release = 30


class CryptaGraphV1ExecDir(sdk2.Resource):
    pass


class RunType:
    V1 = "v1"
    V2 = "v2"
    VALL = "vall"
    CUSTOM = "custom"
    POSTROCK = "postrock"


K_GB = 1 << 10  # megabytes in gigabyte
K_DAY = 60 * 60 * 24  # seconds in day


class FakeVault(object):

    """Use to safe escape ENV class before print"""

    def __getattr__(self, key):
        return lambda *args, **kwargs: "*** FAKE VALUE ***"


class CryptaGraphV1(CryptaTask):

    """Run skleyka 1.0 in sandbox"""

    class CryptaOptions(CryptaTask.CryptaOptions):

        bundle_resource_type = CryptaGraphV1Bundle
        use_semaphore = True

    class Requirements(CryptaTask.Requirements):

        disk_space = 120 * K_GB
        ram = 60 * K_GB
        cores = 16

    class Parameters(CryptaTask.Parameters):

        kill_timeout = 3 * K_DAY
        tags = ["v1"]

        date = sdk2.parameters.String("Run date", required=True, default="yesterday")
        pool = sdk2.parameters.String("YT pool", required=False)
        run_type = sdk2.parameters.String(
            "Run type",
            required=True,
            choices=(
                ("Run v1 graph", RunType.V1),
                ("Run v2 graph", RunType.V2),
                ("Run all(v1/v2) graph", RunType.VALL),
                (u"Play Post Rock! 🤘", RunType.POSTROCK),
                ("Custom (specify under options)", RunType.CUSTOM),
            ),
            default=RunType.VALL,
        )
        options = sdk2.parameters.List("Additional params for run binary", required=False)

        with sdk2.parameters.Output:
            result_resource = sdk2.parameters.Integer("Result resource id", required=True)

    def get_cmd(self):
        binary_path_v1 = "opt/crypta/crypta-graph-v1/crypta-graph-v1"
        binary_path_vall = "opt/crypta/crypta-graph-v1/crypta-graph-vall"
        if self.Parameters.run_type == RunType.V1:
            binary_path = binary_path_v1
        else:
            binary_path = binary_path_vall
        os.rename(os.path.abspath(binary_path_v1), os.path.abspath(binary_path))

        logger.info(self._env_class(vault=FakeVault())._to_string())
        self._prepare_dirs()
        cmd = [os.path.abspath(binary_path)]

        if self.Parameters.run_type == RunType.V1:
            cmd.extend(["--graph", self.run_date, "--graph-task", "MainV1Task"])
        elif self.Parameters.run_type == RunType.V2:
            cmd.extend(["--graph", self.run_date, "--graph-task", "MainV2Task"])
        elif self.Parameters.run_type == RunType.VALL:
            cmd.extend(["--graph", self.run_date, "--graph-task", "MainTask"])
        elif self.Parameters.run_type == RunType.POSTROCK:
            cmd.extend(["--postrock", "--force-yes"])

        for option in self.Parameters.options:
            cmd.extend(option.strip().split())
        return cmd

    def on_execute(self, *args, **kwargs):
        if self.Parameters.environment == helpers.PRESTABLE:
            raise RuntimeError("No PRESTABLE environment for skleika in SB")
        try:
            exec_dir = CryptaGraphV1ExecDir(self, "Graph V1 for date {dt}".format(dt=self.run_date), "output")
            exec_dir_data = sdk2.ResourceData(exec_dir)
            super(CryptaGraphV1, self).on_execute(*args, **kwargs)
        finally:
            try:
                exec_dir_data.ready()
                self.Parameters.result_resource = exec_dir.id
            except Exception:
                logger.exception("Could not set ready for output directory resource")

    def get_semaphore_name(self):
        return "{cls}_{env}_{run_type}_{date}_{options}".format(
            cls=self.__class__.__name__,
            env=self.Parameters.environment,
            run_type=self.Parameters.run_type,
            date=self.run_date,
            options="_".join(self.Parameters.options),
        )

    def get_juggler_service(self):
        return self.get_juggler_service_for_run_type(self.Parameters.run_type, self.Parameters.environment)

    @classmethod
    def get_juggler_service_for_run_type(cls, run_type, env):
        return "{}_{}_{}".format(cls.get_class_juggler_service(), run_type, env)

    @property
    def run_date(self):
        if self.Parameters.date == "yesterday":
            created = self.created
            # NOTE(k-zaitsev): Sandbox gives out times in UTC. However sometimes created is naive, sometimes tz-aware.
            # I've found it to be naive during on_enqueue (calls get_semaphore_name) and tz-aware during on_execute
            if created.tzinfo is None:
                created = pytz.utc.localize(created)

            local_created = created.astimezone(tz=pytz.timezone("Europe/Moscow"))
            date_str = (local_created - datetime.timedelta(days=1)).strftime("%Y-%m-%d")
            logger.info("Requested to run for 'yesterday'. Task created %s, running for %s", self.created, date_str)
            return date_str
        return self.Parameters.date

    @property
    def yt_pool(self):
        if self.Parameters.pool:
            return self.Parameters.pool
        if self.Parameters.environment == helpers.TESTING:
            return "crypta_graph_testing"
        if self.Parameters.environment == helpers.STABLE:
            if self.Parameters.run_type == RunType.V1:
                return "crypta_graph_v1"
            return "crypta_graph"
        return "crypta_graph_testing"

    def get_additional_env(self):
        return self._env_class()._as_dict()

    def _prepare_dirs(self):
        env_class = self._env_class()
        base_path = env_class._BASE_PATH

        dirs = [
            value for value in filter(lambda value: value.startswith(base_path), env_class._as_dict().values())
        ] + [
            env_class.RTCRYPTA_PYSTARTER_LOG_FOLDER,
            env_class.MONRUN_FOLDER,
            env_class.MONRUN_DATE_FOLDER,
            env_class.RTCRYPTA_PYSTARTER_LOG_FOLDER + "luigi",
            env_class.RTCRYPTA_PYSTARTER_LOG_FOLDER + "yt",
        ]

        for directory in dirs:
            if not os.path.exists(directory):
                os.makedirs(directory)

    def _env_class(self, vault=vault):
        class ENV:

            _BASE_PATH = os.path.abspath("output")
            IS_SANDBOX = True

            SOUP_NORMALIZE_LAZY = "yes"

            CRYPTA_ENV = self.Parameters.environment.replace("stable", "production").lower()
            ENV_TYPE = CRYPTA_ENV
            CRYPTA_ENVIRONMENT = CRYPTA_ENV

            IDSERVICE = {
                helpers.TESTING: "http://idserv-test.rtcrypta.yandex.net:8080",
                helpers.STABLE: "http://idserv.rtcrypta.yandex.net:8080",
            }[self.Parameters.environment]

            CRYPTA_PATH_PREFIX = os.path.abspath("opt/crypta")
            RTCRYPTA_MAPREDUCE_EXEC = "{CRYPTA_PATH_PREFIX}/mapreduce-yt".format(**locals())

            RTCRYPTA_MR_SERVER = "hahn.yt.yandex.net"
            AUDIENCE_MR_SERVER = "hahn.yt.yandex.net"

            # for skleyka 1.0 is important to run as robot-crypta,
            # not robot-crypta-testing, so use production tokes always :|
            if self.Parameters.run_type == RunType.V1:
                YT_TOKEN = vault.get_yt_token(helpers.TESTING)
                YQL_TOKEN = vault.get_yql_token(helpers.TESTING)
            else:
                YT_TOKEN = vault.get_yt_token(helpers.STABLE)
                YQL_TOKEN = vault.get_yql_token(helpers.STABLE)

            REACTOR_TOKEN = vault.get_reactor_token()
            AUDIENCE_YT_TOKEN = YT_TOKEN
            COPY_TO_HAHN_TOKEN = YT_TOKEN

            YT_PREFIX = "//"
            MR_RUNTIME = "YT"

            YT_SMART_FORMAT = 1
            YT_RETRY_READ = 1

            YT_POOL = self.yt_pool

            # matrixnet apllying binary
            # MX_OPS_BIN = '/usr/bin/mx_ops'  # TODO: ??

            RTCRYPTA_PYSTARTER_LOG_FOLDER = "{_BASE_PATH}/logs/".format(**locals())
            MONRUN_FOLDER = "{_BASE_PATH}/monrun/".format(**locals())
            MONRUN_DATE_FOLDER = "{_BASE_PATH}/monrun/{dt}/".format(dt=self.run_date, **locals())

            # VW_BIN = '/usr/share/crypta-pystarter/vw'  # TODO: ???
            # MR_CRYPTA_SPLIT_BIN = '/usr/bin/mr_crypta_split'  # TODO: ???
            # MR_CRYPTA_CREATE_FILTER_BIN = '/usr/bin/mr_crypta_create_filter'  # TODO: ???

            # statface
            STATFACE_USERNAME = "robot-crypta"
            STATFACE_OAUTH = vault.get_vault_item("sec-01daxg3xp40d5pr2s52sb22pkm[secret]")

            CRYPTA_GRAPH_CRYPTA_HOME = {
                helpers.TESTING: "//home/crypta/testing",
                helpers.STABLE: "//home/crypta/production",
            }[self.Parameters.environment]

            CRYPTA_NOVOSIB_HOME = {
                helpers.TESTING: "//home/crypta/testing",
                helpers.STABLE: "//home/crypta/production",
            }[self.Parameters.environment]

            CRYPTA_YT_HOME = "//home/crypta/production"

            CRYPTA_GRAPH_LOGFELLER_YT_PATH = "//home/logfeller/logs"

            PASSPORT_SOCIAL_FOLDER = "//home/passport/production/socialism/crypta-dump/"

            LOGFELLER_AUTO_FRONT_LOG_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/autoru-front-log/1d/".format(
                **locals()
            )
            LOGFELLER_BAR_LOG_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/bar-navig-log/1d/".format(**locals())
            LOGFELLER_BS_CHEVENT_LOG_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/bs-chevent-log/1d/".format(**locals())
            LOGFELLER_BS_WATCH_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/bs-watch-log/1d/".format(**locals())
            LOGFELLER_EAL_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/export-access-log/1d/".format(**locals())
            LOGFELLER_KINOPOISK_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/kinopoisk-tskv-front-log/1d/".format(
                **locals()
            )
            LOGFELLER_MOB_REPORT_LOG_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/mobreport-access-log/1d/".format(
                **locals()
            )
            LOGFELLER_MOBILE_TRACKING = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/metrika-mobile-install-log/1d/".format(
                **locals()
            )
            LOGFELLER_MOBILE_TRACKING_PRIVATE = (
                "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/metrika-mobile-install-private-log/1d/".format(**locals())
            )
            LOGFELLER_OAUTH_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/oauth-log/1d/".format(**locals())
            LOGFELLER_PASSPORT_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/passport-log/1d/".format(**locals())
            LOGFELLER_REDIR_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/common-redir-log/1d/".format(**locals())
            LOGFELLER_REQANS_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/reqans-log/1d/".format(**locals())
            LOGFELLER_REQANS_NEW_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/search-proto-reqans-log/1d/".format(
                **locals()
            )
            LOGFELLER_REQANS_ALICE_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/search-report-alice-log/1d/".format(
                **locals()
            )
            LOGFELLER_RTB_LOG_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/bs-rtb-log/1d/".format(**locals())
            LOGFELLER_RTGEO_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/crypta-rt-geo-log/1d/".format(**locals())
            LOGFELLER_SBAPI_MITB_LOG_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/sbapi-access-mitb-log/1d/".format(
                **locals()
            )
            LOGFELLER_SENDER_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/sendr-click-log/1d/".format(**locals())
            # CRYPTR-1712
            LOGFELLER_VISIT_V2_LOG_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/visit-v2-log/1d/".format(**locals())
            LOGFELLER_VISIT_V2_PRIVATE_LOG_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/visit-v2-private-log/1d/".format(
                **locals()
            )
            LOGFELLER_PASSPORT_PHONE_LOG_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/passport-phone-log/1d/".format(
                **locals()
            )
            LOGFELLER_PASSPORT_SENSITIVE_LOG_FOLDER = (
                "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/passport-sensitive-log/1d/".format(**locals())
            )
            LOGFELLER_SDK_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/mobile-redirect-bind-id-log/1d/".format(
                **locals()
            )

            PASSPORT_USERDATA = "//home/passport/production/userdata/"
            STATBOX_CUBE_INSTALL = "//statbox/hypercube/mobile/reactivation/v2/by_device_id/"

            POSTBACK_LOG_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/bs-postback-log/1d/".format(**locals())
            POSTCLICK_LOG_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/bs-mobile-postclick-log/1d/".format(**locals())
            BS_XUNIQS_LOG_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/bs-xuniqs-log/1d/".format(**locals())
            BS_HIT_LOG_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/bs-hit-log/1d/".format(**locals())
            BS_RTB_LOG_FOLDER = "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/bs-rtb-log/1d/".format(**locals())
            BS_WEBVISOR_LOG_FOLDER = (
                "{CRYPTA_GRAPH_LOGFELLER_YT_PATH}/metrika-reduced-webvisor-to-crypta-log/1d/".format(**locals())
            )

            PROFILES_EXPORT = "{CRYPTA_NOVOSIB_HOME}/profiles/export/profiles_for_14days".format(**locals())

            PROFILES_STORAGE = "{CRYPTA_GRAPH_CRYPTA_HOME}/profiles/stages/postprocessed-yandexuid/".format(**locals())
            FP_FOLDER = "{CRYPTA_NOVOSIB_HOME}/storage/storage/fp/".format(**locals())
            FRESH_FP_FOLDER = "{CRYPTA_GRAPH_CRYPTA_HOME}/storage/fresh/fp/".format(**locals())
            DIT_MSK_DAY_FOLDER = "{CRYPTA_NOVOSIB_HOME}/dmp/dmp-ditmsk/".format(**locals())  # Novosib source!
            COOKIE_MATCHING_FOLDER = "{CRYPTA_NOVOSIB_HOME}/cookie_matching/tags/".format(
                **locals()
            )  # Novosib source!

            VKID_PUID_DMP_FOLDER = "//home/sherlock/export/crypta/"  # MAIL SOURCE
            WEBVISOR_LOGIN_FOLDER_PROCESSED = "{CRYPTA_GRAPH_CRYPTA_HOME}/state/webvisor_processed/".format(**locals())

            FRAUD_IPS = "{CRYPTA_GRAPH_CRYPTA_HOME}/state/ip_fraud".format(**locals())
            # CRYPTAIS-1533: always take stream_alive from prod
            DATA_TO_CLASSIFY = "{CRYPTA_YT_HOME}/storage/stream_alive".format(**locals())
            PROFILES_BASE_FOLDER = "{CRYPTA_NOVOSIB_HOME}/profiles_base/".format(**locals())

            SOVETNIK_INPUT_FOLDER = "//logs/sovetnik-users-log/1d"
            AUTORU_INPUT_FOLDER = "//home/logfeller/logs/vertis-event-log/1d"
            AUTORU_WAREHOUSE_FOLDER = "//home/verticals/broker/prod/warehouse/auto/events/1d"

            # graph
            GRAPH_PY_FOLDER = "/usr/share/crypta-graph/python/"
            GRAPH_YT_OUTPUT_FOLDER = "{CRYPTA_GRAPH_CRYPTA_HOME}/state/graph/".format(**locals())

            _AUDIENCE_YT_FOLDER_POSTFIX = {
                helpers.TESTING: "state/audience-testing/",
                helpers.STABLE: "state/audience/",
            }[self.Parameters.environment]

            AUDIENCE_YT_FOLDER = "{CRYPTA_GRAPH_CRYPTA_HOME}/{_AUDIENCE_YT_FOLDER_POSTFIX}".format(**locals())
            GRAPH_STREAM_FOLDER = "{CRYPTA_GRAPH_CRYPTA_HOME}/state/graph/stream/".format(**locals())
            GRAPH_YT_STAT_FOLDER = "{CRYPTA_GRAPH_CRYPTA_HOME}/state/graph_stat/".format(**locals())
            GRAPH_YT_DICTS_FOLDER = "{GRAPH_YT_OUTPUT_FOLDER}dicts/".format(**locals())
            CRYPTA_IDS_STORAGE = "{CRYPTA_GRAPH_CRYPTA_HOME}/ids_storage/".format(**locals())
            CRYPTA_SHARED_IDS_FOLDER = "{GRAPH_YT_OUTPUT_FOLDER}shared/".format(**locals())

            CRYPTA_GRAPH_VARLIB = "{_BASE_PATH}/var/lib/crypta-graph/".format(**locals())
            GRAPH_LOCAL_OUTPUT_FOLDER = "{CRYPTA_GRAPH_VARLIB}/data/".format(**locals())
            GRAPH_HIST_FOLDER = "{CRYPTA_GRAPH_VARLIB}/data/".format(**locals())
            GRAPH_LOG_FOLDER = "{_BASE_PATH}/var/log/crypta-graph/".format(**locals())
            GRAPH_ISCRYPTA_FOLDER = "{CRYPTA_GRAPH_CRYPTA_HOME}/state/iscrypta/".format(**locals())

            # Initial source was in x-products
            GRAPH_PARTNERS_DATA = "//home/x-products/production/partners_data/audience/"
            GRAPH_EXTERNAL_DUMPS = "{GRAPH_YT_DICTS_FOLDER}external_dumps/".format(**locals())

            GRAPH_RIPE_LOCAL_DIR = "{_BASE_PATH}/home/shpolsky/ripe/".format(**locals())
            GRAPH_IP_LOCAL_DIR = "{_BASE_PATH}/home/shpolsky/ip/".format(**locals())
            GRAPH_RIPE_FNAME_4 = "ripe.db.inetnum"
            GRAPH_RIPE_FNAME_6 = "ripe.db.inet6num"
            GRAPH_BAD_IPS_FNAME = "bad_ips.lst"
            GRAPH_STORE_DAYS = {
                helpers.TESTING: 3,
                helpers.STABLE: 30,
            }[self.Parameters.environment]
            AUDIENCE_STORE_DAYS = {
                helpers.TESTING: 10,
                helpers.STABLE: 10,
            }[self.Parameters.environment]
            DEV_CID_TABLE = "{GRAPH_ISCRYPTA_FOLDER}deviceid_cryptaid_mapping".format(**locals())
            VERTICES_TABLE = "{GRAPH_YT_DICTS_FOLDER}vertices".format(**locals())

            # TODO: ???
            # JAVA_HOME = '/usr/local/java8'
            # GRAPH_2_JAVA_EXEC = '/usr/share/crypta-graph-v2/graph2'
            # GRAPH_2_DICTS_JAVA_EXEC = '/usr/share/crypta-graph-v2/graph2-dicts'
            # YT_POOL_MAX_OPS = 15

            # radius metrics
            RADIUS_SPLUNK_URL = "pereval.crypta.yandex.net"
            RADIUS_SPLUNK_PORT = "443"
            RADIUS_SPLUNK_USR = "robot-crypta"
            RADIUS_CLIENT_TVM_ID = "2019036"
            RADIUS_SERVER_TVM_ID = "2019036"
            RADIUS_TVM_SECRET = vault.get_vault_item("sec-01eb3w9ezmcm48cpp7ggva9hv5[token]")

            RADIUS_BASE_YT_FOLDER = "{CRYPTA_GRAPH_CRYPTA_HOME}/state/radius/".format(**locals())
            RADIUS_LOG_LOCAL_FOLDER = "{GRAPH_LOCAL_OUTPUT_FOLDER}radius/log/".format(**locals())
            RADIUS_LOG_YT_FOLDER = "{RADIUS_BASE_YT_FOLDER}log/".format(**locals())
            RADIUS_METRICS_LOCAL_FOLDER = "{GRAPH_LOCAL_OUTPUT_FOLDER}radius/metrics/".format(**locals())
            RADIUS_METRICS_YT_FOLDER = "{RADIUS_BASE_YT_FOLDER}metrics/".format(**locals())

            # indevice
            INDEVICE_YT_FOLDER = "{CRYPTA_GRAPH_CRYPTA_HOME}/state/graph/indevice/".format(**locals())
            INDEVICE_LOG_FOLDER = "{GRAPH_LOG_FOLDER}".format(**locals())

            # indevice unperfect matrixnet model
            INDEVICE_UNPERFECT_MODEL = "/usr/share/rtcrypta-indevice-model/indevice.info"

            # https://st.yandex-team.ru/CRYPTA-4955
            METRIKA_INPUT_FOLDER = "//home/metrika/userparams"

            MOBMET_APP_ID_AND_DOMAINS = (
                "//home/metrica-analytics/firstsvet/MOBMET/applications_and_sites/app_id_and_domains"
            )
            MOBMET_COUNTERS_AND_DOMAINS = (
                "//home/metrica-analytics/firstsvet/MOBMET/applications_and_sites/counters_and_domains"
            )

            LUIGID_HOST = {
                helpers.TESTING: "luigi.testing.graph.crypta.yandex-team.ru",
                helpers.STABLE: "luigi.production.graph.crypta.yandex-team.ru",
            }[self.Parameters.environment]
            LUIGID_PORT = 443
            LUIGID_SCHEME = "https"
            LUIGID_URL = "{LUIGID_SCHEME}://{LUIGID_HOST}:{LUIGID_PORT}/".format(**locals())
            LUIGID_WORKERS = 20

            # BB_KAFKA_HOSTS = '%%BB_KAFKA_HOSTS%%'
            # BB_KAFKA_TOPIC = 'crypta_profile.${CRYPTA_DC_LETTER}'
            # BB_KAFKA_TOPIC_FAST = 'crypta_profile_fast.${CRYPTA_DC_LETTER}'
            # BB_KAFKA_NUM_PARTITIONS = 28
            # BB_LOGBROKER_FILE = '/var/lib/crypta-to-bb/logbroker.stream'

            CRYPTA_UPLOAD_ENABLED = "yes"
            CRYPTA_AUDIENCE_ENABLED = "yes"
            CRYPTA_STATFACE_ENABLED = "yes"
            HAS_BSYETI_TABLES = "yes"
            TEST_FAIL_REPORTING = "no"

            RADIUS_SPLUNK_PASSWORD = vault.get_vault_item("sec-01csvzg07kmzgybhtfkdy8azwa[password]")
            METRICA_RSA_KEY = vault.get_vault_item("sec-01csvzeyhdgwmk5dq5bsntedh6[secret]")

            YT_JOB_MAX_MEMORY_BYTES = 1 << 32  # 4 * 1024  * 1024  * 1024 bytes >~ 4 gigabytes
            YT_MEMORY_LIMIT = 1 << 31  # default memory limit for yt jobs ~2gb

            # Sandbox
            SANDBOX_OAUTH = vault.get_vault_item("sec-01csvzg7vtpvb7bgrx18ajsscj[token]")
            TVM_SECRET = vault.get_vault_item("sec-01eb3wtsvqer1ywz2m0pkewzxq[secret]")

            SOUP_BACKUP_FOLDER = "//home/crypta/archive/graph/soup"
            SOUP_BACKUP_ENABLE = {
                helpers.TESTING: "no",
                helpers.STABLE: "yes",
            }[self.Parameters.environment]

            SOUP_DATES_ENABLED = "yes"
            WEBVISOR_TOLERANCE = "yes"
            FUZZY2_USE_UN = "yes"

            STREAMING_LOGS = "mm access redir wl bar eal bs-rtb-log postback-log"

            YQL_FORCE_EMBEDDED = "yes"
            YQL_UDF_SOURCE = "yt"

            if self.Parameters.run_type == RunType.V1:
                LUIGI_RUN_TAGS = "v1"
            elif self.Parameters.run_type == RunType.V2:
                LUIGI_FAIL_TAGS = "v1"

            @classmethod
            def _as_dict(cls):
                """return all non private members as dict"""
                return {key: str(value) for (key, value) in cls.__dict__.items() if not key.startswith("_")}

            @classmethod
            def _to_string(cls):
                assert isinstance(vault, FakeVault)
                output = "=== CRYPTA GRAPH ENVIRONMENT =======\n"
                for (key, value) in sorted(cls._as_dict().items()):
                    output += "{0}: {1}\n".format(key, value)
                return output + "\n"

        return ENV
