import json
import logging
import requests

from sandbox import sdk2

from sandbox.projects.crypta.common.vault import get_vault_item
from sandbox.common.types.task import Semaphores


PLATFORM_API_URL = 'https://platform.yandex-team.ru/api/v1/environment'
DOCKER_INFO = "https://dockinfo.yandex-team.ru/api/docker/resolve?registryUrl={0}&tag={1}"


class CryptaDeployPlatform(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600

        with sdk2.parameters.Group('Docker') as docker_block:
            docker_url = sdk2.parameters.String('Docker image URL', required=True)
            docker_vault_name = sdk2.parameters.String('Docker token value name', required=True)

        with sdk2.parameters.Group('Platform') as platform_block:
            platform_service_name = sdk2.parameters.String('Platform Service', required=True)
            platform_project_name = sdk2.parameters.String('Platform Project', required=True)
            platform_environment_name = sdk2.parameters.String('Platform Environment', required=True)
            platform_component_name = sdk2.parameters.String('Platform Component', required=True)
            platform_vault_name = sdk2.parameters.String('Platform token vault name', required=True)
            platform_wait_deploy = sdk2.parameters.Bool('Block new tasks until the current platform environment is updated', default=False)

        with sdk2.parameters.Group('Queue') as queue_block:
            semaphore_name = sdk2.parameters.String('Semaphore name (optional)', required=False)

    def on_enqueue(self):
        if self.Parameters.platform_wait_deploy:
            if self.Parameters.semaphore_name:
                semaphore_name = self.Parameters.semaphore_name
            else:
                semaphore_name = '{}_{}_{}'.format(self.Parameters.platform_service_name,
                                                   self.Parameters.platform_project_name,
                                                   self.Parameters.platform_environment_name)
            self.Requirements.semaphores = Semaphores(
                acquires=[
                    Semaphores.Acquire(name=semaphore_name, capacity=1)
                ],
            )

    def on_execute(self):
        component = self.Parameters.platform_component_name
        environment_id = ".".join([
            self.Parameters.platform_service_name, self.Parameters.platform_project_name, self.Parameters.platform_environment_name
        ])
        platform_headers = {
            'Authorization': 'OAuth ' + get_vault_item(self.Parameters.platform_vault_name, self.owner).strip(),
            'Content-Type': 'application/json',
        }
        docker_headers = {
            'Authorization': 'OAuth ' + get_vault_item(self.Parameters.docker_vault_name, self.owner).strip(),
            'Content-Type': 'application/json',
        }
        repository = self.Parameters.docker_url

        docker_request = requests.get(
            DOCKER_INFO.format(*repository.split(':')),
            headers=docker_headers,
        )
        docker_request.raise_for_status()
        docker_hash = docker_request.json()['hash']

        state_request = requests.get(
            '{}/{}/{}'.format(PLATFORM_API_URL, 'current', environment_id),
            headers=platform_headers,
        )
        state_request.raise_for_status()
        environment_version = state_request.json()['version']

        deploy_data = [
            {
                "componentName": component,
                "properties": {
                    "repository": repository,
                    "hash": docker_hash,
                }
            },
        ]
        deploy_url = '{}/fast-deploy/{}/{}'.format(PLATFORM_API_URL, environment_id, environment_version)
        logging.info('POSTing %s to %s', deploy_data, deploy_url)
        deploy_request = requests.post(
            deploy_url,
            data=json.dumps(deploy_data),
            headers=platform_headers,
        )
        if deploy_request.status_code != 204:
            self.set_info("<b>Error updating service</b>: {0} {1}".format(deploy_request.status_code, deploy_request.text), do_escape=False)
            raise Exception('Failed to deploy')
