import logging
import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.crypta.common import (
    helpers,
    resource_selector,
    task,
    vault,
)
from sandbox.projects.crypta.run_binary.bundles import *  # noqa


class CryptaRunBinary(task.CryptaTask):
    """Runs any binary"""

    class CryptaOptions(task.CryptaTask.CryptaOptions):
        use_semaphore = False

    class Requirements(task.CryptaTask.Requirements):
        """
        """

        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(task.CryptaTask.Parameters):
        """
        """

        with sdk2.parameters.Group("Binary resource") as resource_bin:
            binary_resource_bundle = resource_selector.ResourceSelector("Bundle", required=False)
            binary_resource_id = sdk2.parameters.String("Resource ID", required=False)
        with sdk2.parameters.Group("Vault") as vault:
            VAULT_OWNER = sdk2.parameters.String("Owner of vault secrets", required=True)
            VAULT_YQL_TOKEN = sdk2.parameters.String("Name of YQL token secret", required=True)
            VAULT_YT_TOKEN = sdk2.parameters.String("Name of YT token secret", required=True)
            VAULT_STATFACE_TOKEN = sdk2.parameters.String("Name of Statface token secret", required=False)
            VAULT_SOLOMON_TOKEN = sdk2.parameters.String("Name of Solomon OAuth token secret", required=False)
            VAULT_TVM_SECRET = sdk2.parameters.String("Name of TVM secret", required=False)

        with sdk2.parameters.Group("Parameters") as params:
            ARGS = sdk2.parameters.String("Arguments")
            EXTRA_ENV = sdk2.parameters.Dict("Additional environment for run binary", required=False)

    class Context(sdk2.Task.Context):
        pass

    def on_execute(self):
        """
        """
        logging.info("Using binary resource %s", self.Parameters.binary_resource_id)
        binary = self.get_binary()

        with sdk2.helpers.ProcessLog(self, logger="crypta_binary") as process_log:
            env = {
                "YQL_TOKEN": self.get_secret(self.Parameters.VAULT_YQL_TOKEN),
                "YT_TOKEN": self.get_secret(self.Parameters.VAULT_YT_TOKEN),
                "TVM_SECRET": self.get_secret(self.Parameters.VAULT_TVM_SECRET),
                "STATFACE_OAUTH": self.get_secret(self.Parameters.VAULT_STATFACE_TOKEN),
                "SOLOMON_TOKEN": self.get_secret(self.Parameters.VAULT_SOLOMON_TOKEN),
                "ENVIRONMENT": self.get_environment(),
            }
            env.update(self.Parameters.EXTRA_ENV)

            os.chmod(binary, 0755)

            subprocess.check_call(
                [binary] + list(self.task_parameters()),
                shell=False,
                stdout=process_log.stdout,
                stderr=process_log.stderr,
                env=env,
            )

    def get_secret(self, what):
        """ Get param from vault or default value """
        if not what:
            return "nonexistent"
        return vault.get_vault_item(what, vault_owner=self.Parameters.VAULT_OWNER)

    def get_environment(self):
        """
        """
        return self.Parameters.environment.lower().replace("stable", "production")

    def task_parameters(self):
        """
        """
        return (self.Parameters.ARGS or "").split()

    def get_binary(self):
        """
        """
        if self.Parameters.binary_resource_id:
            return self._get_binary_by_id()
        if self.Parameters.binary_resource_bundle:
            return self._get_binary_by_bundle()
        raise RuntimeError("Required resource id or bundle")

    def _get_binary_by_id(self):
        resource = sdk2.Resource[self.Parameters.binary_resource_id]
        resource_data = sdk2.ResourceData(resource)
        return resource_data.path.as_posix()

    def _get_binary_by_bundle(self):
        BundleClass = resource_selector.get_bundle_dict()[self.Parameters.binary_resource_bundle]  # N806 # noqa
        bundle = helpers.get_last_released_resource(BundleClass, self.Parameters.environment)
        bundle_path = helpers.download_resource(bundle)
        helpers.untar_path(bundle_path)
        return os.path.abspath("bundle.bin")

    def on_prepare(self):
        pass
