import datetime
import logging
import os
import requests

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.common.build.YaMake2 import YaMake2, YaMakeParameters

from sandbox.projects.crypta.common import vault


def absolute_file_paths(directory):
    for dirpath, _, filenames in os.walk(directory):
        for fname in filenames:
            yield os.path.abspath(os.path.join(dirpath, fname))


def upload(yt_client, fname, remote_dir, attrs, expiration):
    rpath = os.path.join(remote_dir, os.path.basename(fname))
    with yt_client.Transaction():
        if yt_client.exists(rpath):  # unlink to reset creation time
            logging.debug("removing: %s", rpath)
            yt_client.remove(rpath)
        with open(fname, "rb") as ifile:
            yt_client.write_file(rpath, ifile, force_create=True, compute_md5=True)
        for key, value in attrs.iteritems():
            yt_client.set("{0}/@{1}".format(rpath, key), str(value).lower())
        if expiration and "arcadia_revision" in attrs:
            rev_path = "{0}.{1}".format(rpath, attrs["arcadia_revision"])
            ttl = datetime.datetime.now() + datetime.timedelta(days=expiration)
            yt_client.copy(rpath, rev_path, force=True)
            yt_client.set("{0}/@expiration_time".format(rev_path), str(ttl))


def copy(fname, archive_dir, publush_dir, revision, proxy, token):
    headers = {"Authorization": "OAuth {0}".format(token)}
    url = "https://{proxy}/api/v3/copy".format(proxy=proxy)

    rev_path = "{0}.{1}".format(os.path.join(archive_dir, fname), revision)
    d_path = os.path.join(publush_dir, fname)

    response = requests.post(
        url,
        headers=headers,
        verify=False,
        params={"source_path": rev_path, "destination_path": d_path, "force": "true"},
    )
    logging.debug("Copy response %s", response.json())
    response.raise_for_status()


class CryptaYaMakeYt(YaMake2):
    class Parameters(sdk2.Task.Parameters):
        ya_make_default = YaMakeParameters()
        binary_executor_release_type = "stable"

        with sdk2.parameters.Group("Publish YT parameters:") as ya_make_yt_publish_params:
            yt_publish_proxy = sdk2.parameters.String("YT publish proxy", required=True, default="hahn.yt.yandex.net")
            yt_publish_dir = sdk2.parameters.String("YT publish dir", required=True)
            yt_archive_dir = sdk2.parameters.String("YT archive dir", required=True)
            yt_token_owner = sdk2.parameters.String("YT token owner", required=True)
            yt_token_name = sdk2.parameters.String("YT token name", required=True)
            yt_expiration_days = sdk2.parameters.Integer("YT expiration days (0 is inf)", required=True, default=14)

    class Requirements(YaMake2.Requirements):
        environments = [environments.PipEnvironment("yandex-yt")]

    def post_build(self, source_dir, output_dir, pack_dir):
        super(CryptaYaMakeYt, self).post_build(source_dir, output_dir, pack_dir)
        try:
            self._push_on_yt(output_dir)
        except:
            # on any error on yt publish skip
            logging.exception("YA_MAKE_YT.post_build():")
            raise  # ok, let it fail now!

    def on_release(self, additional_parameters):
        super(CryptaYaMakeYt, self).on_release(additional_parameters)
        logging.debug("release params %s", additional_parameters)
        if additional_parameters["release_status"] == "cancelled":
            logging.info("Skip cancelled release (do nothing)")
            # Warning! You should re-release by hands previous task, to recreate udf.
            return
        self._release_on_yt("build/lib")

    def _push_on_yt(self, directory):
        import yt.wrapper as yt

        resources = self.get_resources()
        logging.debug("resources %s", resources)

        yt_client = yt.YtClient(proxy=self.Parameters.yt_publish_proxy, token=self._get_token())
        attrs = self.get_base_resource_attrs()
        attrs.update({"executable": True, "sndbx-task": self.id})

        logging.debug("publish attrs %s", attrs)
        logging.debug("publish paths %s", list(absolute_file_paths(directory)))
        logging.debug("arts %s", self.get_arts())
        logging.debug("arts src %s", self.get_arts_source())

        for art in self.get_arts():
            fpath = os.path.abspath(os.path.join(directory, art["path"]))
            logging.debug("upload file %s", fpath)
            upload(
                yt_client,
                fpath,
                self.Parameters.yt_archive_dir,
                attrs=attrs,
                expiration=self.Parameters.yt_expiration_days,
            )

    def _release_on_yt(self, directory):
        logging.debug("publish paths %s", list(absolute_file_paths(directory)))
        logging.debug("arts %s", self.get_arts())
        logging.debug("arts src %s", self.get_arts_source())
        for fpath in self.get_arts():
            logging.debug("copy file %s", fpath)
            copy(
                os.path.basename(fpath["path"]),
                self.Parameters.yt_archive_dir,
                self.Parameters.yt_publish_dir,
                self.Context.ap_arcadia_revision,
                self.Parameters.yt_publish_proxy,
                self._get_token(),
            )

    def _get_token(self):
        return vault.get_vault_item(
            item_name=self.Parameters.yt_token_name, vault_owner=self.Parameters.yt_token_owner
        )
