import itertools
import logging
import os

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import resource as ctr
from sandbox.projects import resource_types
from sandbox.projects.common import binary_task
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils
from sandbox.projects.common.differ import printers
from sandbox.projects.common.search import requester as search_requester
from sandbox.projects.images.daemons import resources as images_daemons_resources
import sandbox.projects.common.app_host.converter as apphost_converter


class ImagesCompareCbirdaemonApphostResponses(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
        Get cbirdaemon responses (apphost mode)
    """
    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        cbir_daemon_responses1_resource_id = sdk2.parameters.Resource('Cbirdaemon responses #1',
                                                                      resource_type=images_daemons_resources.CBIR_DAEMON2_APPHOST_RESPONSES,
                                                                      required=True)
        cbir_daemon_responses2_resource_id = sdk2.parameters.Resource('Cbirdaemon responses #2',
                                                                      resource_type=images_daemons_resources.CBIR_DAEMON2_APPHOST_RESPONSES,
                                                                      required=True)
        with sdk2.parameters.Output():
            diff_count = sdk2.parameters.Integer("Diff count", default=0)
            total_count = sdk2.parameters.Integer("Total count", default=0)
            compare_result = sdk2.parameters.Bool("Compare result")

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "IMAGES_COMPARE_CBIRDAEMON_APPHOST_RESPONSES", "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    def on_execute(self):
        from sandbox.projects.cvsearch.ImagesCompareCbirdaemonApphostResponses.lib import compare_responses
        super(ImagesCompareCbirdaemonApphostResponses, self).on_execute()
        resource_diff = sdk2.ResourceData(images_daemons_resources.CBIR_DAEMON2_APPHOST_RESPONSES_DIFF(self, "A directory with diff", "diff"))
        resource_diff.path.mkdir(0o755, parents=True, exist_ok=True)
        diff_path = str(resource_diff.path.absolute())
        iterator1 = self.__responses_iterator(self.Parameters.cbir_daemon_responses1_resource_id)
        iterator2 = self.__responses_iterator(self.Parameters.cbir_daemon_responses2_resource_id)

        diff_content = []
        diff_count = 0
        total_count = 0
        for n, responses in enumerate(itertools.izip_longest(iterator1, iterator2)):
            total_count += 1
            response1, response2 = responses
            logging.info("response1 %s" % (response1))
            logging.info("response2 %s" % (response1))
            if response1 is None or response2 is None:
                raise TaskFailure("Different number of responses")

            output = compare_responses.compare(response1, response2, "response #{}".format(n), diff_path)
            if output:
                diff_count += 1
                diff_content.extend(output)

        if diff_content:
            printers.HtmlFile.write_companion(diff_path, "response_diff_style.css")
            printers.HtmlFile.write_companion(diff_path, "response_diff_scripts.js")
            with open(os.path.join(diff_path, "diff.html"), "w") as diff_file:
                printers.HtmlFile.write(diff_file, "".join(diff_content), buttons=False)
            self.Parameters.compare_result = False
        else:
            fu.write_file(os.path.join(diff_path, utils.NO_DIFF_NAME), utils.no_diff_detected())
            self.Parameters.compare_result = True
        self.Parameters.diff_count = diff_count
        self.Parameters.total_count = total_count
        resource_diff.ready()

    def __responses_iterator(self, responses_resource_id):
        responses_path = str(sdk2.ResourceData(sdk2.Resource[responses_resource_id]).path.absolute())
        logging.info("responses_path %s" % (responses_path))
        # TODO: don't use converter released with noapache
        converter_path = str(sdk2.ResourceData(
            resource_types.APP_HOST_TOOL_CONVERTER_EXECUTABLE.find(
                arch='linux',
                attrs={"released": "stable"}
            ).first()).path.absolute())
        logging.info("converter_path %s" % (converter_path))
        converter_func = apphost_converter.convert_input_response
        with open(responses_path) as responses_file:
            for data in search_requester.sequence_binary_data_from_stream(responses_file):
                logging.info("data %s" % (data))
                yield converter_func(converter_path, data)
