import itertools
import logging
import os
import re

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import resource as ctr
from sandbox.projects.common import binary_task
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils
from sandbox.projects.common.differ import printers
from sandbox.projects.images.daemons import resources as images_daemons_resources

_FEATURE_RE = re.compile(r"([^{]+){([^}]+)}")


class ImagesCompareCbirdaemonHttpResponses(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
        Get cbirdaemon responses (http mode)
    """
    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        cbir_daemon_responses1_resource_id = sdk2.parameters.Resource('Cbirdaemon responses #1',
                                                                      resource_type=images_daemons_resources.CBIR_DAEMON2_RESPONSES,
                                                                      required=True)
        cbir_daemon_responses2_resource_id = sdk2.parameters.Resource('Cbirdaemon responses #2',
                                                                      resource_type=images_daemons_resources.CBIR_DAEMON2_RESPONSES,
                                                                      required=True)
        with sdk2.parameters.Output():
            diff_count = sdk2.parameters.Integer("Diff count", default=0)
            total_count = sdk2.parameters.Integer("Total count", default=0)
            compare_result = sdk2.parameters.Bool("Compare result")

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "IMAGES_COMPARE_CBIRDAEMON_HTTP_RESPONSES", "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    def on_execute(self):
        from sandbox.projects.cvsearch.ImagesCompareCbirdaemonApphostResponses.lib import compare_responses
        super(ImagesCompareCbirdaemonHttpResponses, self).on_execute()
        resource_diff = sdk2.ResourceData(images_daemons_resources.CBIR_DAEMON2_RESPONSES_DIFF(self, "A directory with diff", "diff"))
        resource_diff.path.mkdir(0o755, parents=True, exist_ok=True)
        diff_path = str(resource_diff.path.absolute())
        iterator1 = self.__responses_iterator(self.Parameters.cbir_daemon_responses1_resource_id)
        iterator2 = self.__responses_iterator(self.Parameters.cbir_daemon_responses2_resource_id)

        diff_content = []
        diff_count = 0
        total_count = 0
        for n, responses in enumerate(itertools.izip_longest(iterator1, iterator2)):
            total_count += 1
            response1, response2 = responses
            logging.info("response1 %s" % (response1))
            logging.info("response2 %s" % (response1))
            if response1 is None or response2 is None:
                raise TaskFailure("Different number of responses")

            output = compare_responses.compare(response1, response2, "response #{}".format(n), diff_path)
            if output:
                diff_count += 1
                diff_content.extend(output)

        if diff_content:
            printers.HtmlFile.write_companion(diff_path, "response_diff_style.css")
            printers.HtmlFile.write_companion(diff_path, "response_diff_scripts.js")
            with open(os.path.join(diff_path, "diff.html"), "w") as diff_file:
                printers.HtmlFile.write(diff_file, "".join(diff_content), buttons=False)
            self.Parameters.compare_result = False
        else:
            fu.write_file(os.path.join(diff_path, utils.NO_DIFF_NAME), utils.no_diff_detected())
            self.Parameters.compare_result = True
        self.Parameters.diff_count = diff_count
        self.Parameters.total_count = total_count
        resource_diff.ready()

    def __responses_iterator(self, responses_resource_id):
        responses_path = str(sdk2.ResourceData(sdk2.Resource[responses_resource_id]).path.absolute())
        logging.info("responses_path %s" % (responses_path))
        with open(responses_path) as responses_file:
            while True:
                header = responses_file.readline()
                if not header:
                    break
                size = int(header)
                data = responses_file.read(size)
                if len(data) != size:
                    raise TaskFailure("Invalid content in file '{}'".format(responses_path))
                yield self.__parse_response(data)

    def __parse_response(self, data):
        result = {}
        for line in data.strip().split("\n"):
            m = _FEATURE_RE.search(line)
            if not m:
                raise TaskFailure("Unsupported response format: '{}'".format(data))
            result[m.group(1)] = m.group(2)
        # make app-host compatible structure
        return {"answers": [{"signatures": result}]}
