# -*- coding: utf-8 -*-
import requests
import json

import logging

from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
import sandbox.common.types.task as ctt
from sandbox.projects.common import task_env

DATA_TRANSFER_URL = "https://cdc.n.yandex-team.ru"
DATA_TRANSFER_ACTIVATE_API = "/v1/transfer/{}:activate"
DATA_TRANSFER_CHECK_OPERATION_API = "/v1/operation"


class DataTransferActivator(sdk2.Task):
    class Requirements(task_env.TinyRequirements):
        disk_space = 10000
        ram = 4 * 1024

    class Context(sdk2.Context):
        activation_id = None
        attempts_count = 50

    class Parameters(sdk2.Parameters):
        transfer_id = sdk2.parameters.String('Id of transfer to activate', required=True)
        token_name = sdk2.parameters.String('Name of secret with token in vault. Link to oauth token - '
                                            'https://oauth.yandex-team.ru/authorize?response_type=token&client_id=a749413f4bb34e8e855ff242c3693928',
                                            required=True)
        token_owner = sdk2.parameters.String('Owner of secret', required=True)
        wait_transfer_finish = sdk2.parameters.Bool('Wait transfer finish', default=False)
        wait_sessions_check_interval = sdk2.parameters.Integer(
            "Check sessions interval in seconds",
            default_value=600,
        )

    def GetToken(self):
        secret_name = self.Parameters.token_name
        secret_owner = self.Parameters.token_owner
        token = sdk2.Vault.data(secret_owner, secret_name)
        return token

    def sendRequest(self, api, req_type, data=None):
        token = self.GetToken()
        data_transfer_api_url = DATA_TRANSFER_URL + api

        headers = {
            "Content-type": "application/json",
            "Authorization": "{}".format(token)
        }

        retry_strategy = Retry(
            total=3,
            status_forcelist=[400, 429, 500, 502, 503, 504],
            method_whitelist=["GET", "PATCH", "POST"],
            backoff_factor=20
        )

        adapter = HTTPAdapter(max_retries=retry_strategy)
        http = requests.Session()
        http.mount("https://", adapter)
        http.mount("http://", adapter)

        logging.info("Start {} to {}".format(req_type, data_transfer_api_url))

        if req_type == "GET":
            response = http.get(url=data_transfer_api_url, headers=headers)
        elif req_type == "PATCH":
            body_json = data
            response = http.patch(data_transfer_api_url, json=body_json, headers=headers)
            logging.info("Data to data transfer: " + json.dumps(body_json, ensure_ascii=False))
        elif req_type == "POST":
            response = http.post(data_transfer_api_url.format(data), json={}, headers=headers)
        else:
            raise Exception("Unknown request type")

        logging.info("RESPONSE STATUS: " + str(response.status_code))
        logging.info("RESPONSE HEADERS: {}".format(response.headers))
        logging.info("RESPONSE: {}".format(response.content))
        if (response.status_code == 200):
            response.raise_for_status()
            return json.loads(response.content)

        raise TaskFailure("Request to {} is fail".format(data_transfer_api_url))

    def activate(self):
        response = self.sendRequest(DATA_TRANSFER_ACTIVATE_API, "POST", self.Parameters.transfer_id)

        if "id" in response:
            self.Context.activation_id = response["id"]

    def wait_transfer_finish(self):
        api_for_check = DATA_TRANSFER_CHECK_OPERATION_API + "/{}".format(self.Context.activation_id)
        while (self.Context.attempts_count > 0):
            self.Context.attempts_count = self.Context.attempts_count - 1
            response = self.sendRequest(api_for_check, "GET")
            if "done" not in response:
                if (self.Context.attempts_count == 0):
                    raise TaskFailure("Operation {} not finished yet".format(self.Context.activation_id))
                raise sdk2.WaitTime(self.Parameters.wait_sessions_check_interval)
            else:
                if "error" in response:
                    raise TaskFailure("Operation {} failed".format(self.Context.activation_id))

    def on_create(self):
        self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={"Name": "DataTransferActivator", "release": ctt.ReleaseStatus.STABLE},
        ).first()

    def on_execute(self):
        with self.memoize_stage.activation:
            self.activate()
        if self.Parameters.wait_transfer_finish:
            self.wait_transfer_finish()
