# -*- coding: utf-8 -*-

import time
import logging
from datetime import date
from collections import namedtuple

import requests

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.common import task_env


ST_EMPTY = 'empty()'
ST_NOT_EMPTY = 'notEmpty()'

BACKEND_COMPONENT_ID = 72985
FRONTEND_COMPONENT_ID = 72986
STATFACE_COMPONENT_ID = 90166
PRODUCT_COMPONENT_ID = 77497

TG_CHAT_ID = -1001222491961

CRITICAL_TICKET_SIGN = '💀'
CRITICAL_TAG = 'escalate'

_log = logging.getLogger(__name__)


class TGClient(object):
    TG_API = 'https://api.telegram.org/bot{token}'

    def __init__(self, token):
        self.token = token

    @staticmethod
    def escape(string):
        chars_to_escape = ['_', '*', '[', ']', '(', ')', '~', '`', '>', '#', '+', '-', '=', '|', '{', '}', '.', '!']
        for ch in chars_to_escape:
            string = string.replace(ch, '\\' + ch)
        return string

    def _ask_api(self, method, path, json_data=None):
        url = '{}/{}'.format(self.TG_API.format(token=self.token), path)
        resp = requests.request(method, url, json=json_data)
        if resp.status_code >= 400:
            _log.error('returned %s: %s', resp.status_code, resp.content)
        resp.raise_for_status()

    def send_message(self, message, parse_mode='MarkdownV2'):
        self._ask_api(
            'POST',
            'sendMessage',
            json_data={'text': message, 'chat_id': TG_CHAT_ID, 'parse_mode': parse_mode}
        )


def get_tg_login(staff_login, staff_token):
    r = requests.get(
        'https://staff-api.yandex-team.ru/v3/persons?login={}&_one=1&_fields=accounts'.format(staff_login),
        headers={'Authorization': 'OAuth {}'.format(staff_token)}
    )
    r.raise_for_status()
    return [d['value'] for d in r.json()['accounts'] if d['type'] == 'telegram'][0]


def send_event_to_juggler(host, service, status, description, tags=None):
    _log.info('Going to send juggler event: host=%s, service=%s, status=%s', host, service, status)
    tags = tags or []
    reply = requests.post(
        'http://juggler-push.search.yandex.net/events',
        json={
            'source': 'datalens',
            'events': [
                {
                    'host': host,
                    'service': service,
                    'status': status,
                    'description': description,
                    'tags': tags,
                }
            ]
        },
        timeout=10,
    )
    reply.raise_for_status()
    event_status = reply.json()['events'][0]
    if event_status['code'] != 200:
        raise requests.exceptions.HTTPError(event_status['error'])


class DLHELPHelper(sdk2.Task):

    class Requirements(task_env.TinyRequirements):
        environments = [
            # List of required packages
            environments.PipEnvironment(
                'startrek_client', version='2.5',
                custom_parameters=['requests==2.18.4'], use_wheel=False
            ),
        ]

    def on_execute(self):
        PersonsOnDuty = namedtuple('PersonsOnDuty', ['primary', 'backup'])

        def get_persons_on_duty_from_gore_api(duty_type, gore_token):
            service = {
                'product': 'datalens-product',
                'back': 'datalens-back',
                'front': 'datalens-front'
            }[duty_type]
            r = requests.get(
                'https://resps-api.cloud.yandex.net/api/v0/duty/{}'.format(service),
                headers={'Authorization': 'OAuth {}'.format(gore_token)}
            )
            if r.status_code != 200:
                _log.error('Gore api returned %s: %s', r.status_code, r.content)
            r.raise_for_status()
            data = r.json()
            tstamp = int(time.time())
            primary = None
            backup = None

            for item in data:
                if item['datestart'] <= tstamp < item['dateend']:
                    if item['resp']['order'] == 0:
                        primary = item['resp']['username']
                    elif item['resp']['order'] == 1:
                        backup = item['resp']['username']
                    else:
                        _log.error('Unknown resp order: %s', item['resp'])

            return PersonsOnDuty(primary=primary, backup=backup)

        def get_persons_on_duty_from_abc_api(duty_type, abc_token):
            duty_type_to_abc_id_and_slug = {
                'product': {'abc_id': 2170, 'primary_slug': 'datalens-product', 'backup_slug': None},
                'back': {'abc_id': 30824, 'primary_slug': 'cloud-primary', 'backup_slug': 'cloud-backup'},
                'front': {'abc_id': 1501, 'primary_slug': 'primary', 'backup_slug': 'backup'},
            }
            abc_info = duty_type_to_abc_id_and_slug[duty_type]

            res = []
            for prback in ('primary_slug', 'backup_slug'):
                slug = abc_info[prback]
                if slug is None:
                    continue
                r = requests.get(
                    'https://abc-back.yandex-team.ru/api/v4/duty/on_duty/',
                    params={
                        'service': abc_info['abc_id'],
                        'schedule__slug': abc_info[prback]
                    },
                    headers={'Authorization': 'OAuth {}'.format(abc_token)}
                )
                r.raise_for_status()
                res.append(r.json()[0]['person']['login'])

            return PersonsOnDuty(primary=res[0], backup=res[1] if len(res) > 1 else None)

        def get_persons_on_duty(duty_type, gore_token):
            assert duty_type in ('back', 'front', 'common', 'product')

            if duty_type == 'common':
                today = date.today()
                if (today.day + today.month) % 2 == 0:
                    common_duty = 'back'
                else:
                    common_duty = 'front'
                return get_persons_on_duty_from_abc_api(common_duty, gore_token)
            else:
                return get_persons_on_duty_from_abc_api(duty_type, gore_token)

        def ticket_is_critical(ticket):
            return CRITICAL_TAG in ticket.tags

        def main():
            from startrek_client import Startrek

            secret = sdk2.yav.Secret('sec-01d4jkj89g0gac6c3nm2nqq5q3')
            st_token = secret.data()['st_oauth']
            gore_token = secret.data()['gore_oauth']
            tg_token = sdk2.yav.Secret('sec-01e98kzw308tqn04k48mf7hmnr').data()['token']

            tg_client = TGClient(token=tg_token)
            st_client = Startrek(useragent='DLHELP_HELPER', token=st_token)

            def assign_ticket(st_ticket, assignee):
                _log.info('Assigning ticket %s to %s', st_ticket.key, assignee)
                ticket.update(assignee=assignee)

                message = 'New ticket: [{summ}](https://st.yandex-team.ru/{key}) for @{asgn}'.format(
                    summ=tg_client.escape(st_ticket.summary), key=st_ticket.key,
                    asgn=tg_client.escape(get_tg_login(assignee, st_token)),
                )
                if ticket_is_critical(st_ticket):
                    message = '{} *CRIT*: {}'.format(CRITICAL_TICKET_SIGN, message)

                tg_client.send_message(
                    message=message
                )

            # notify about blockers without a component
            _log.info('Going no notify about untriaged blockers')
            new_important_tickets = st_client.issues.find(
                filter={
                    'queue': 'DLHELP', 'resolution': ST_EMPTY, 'status': 'new',
                    'type': 'task', 'tags': [CRITICAL_TAG], 'assignee': ST_NOT_EMPTY
                }
            )

            for ticket in new_important_tickets:
                _log.info('Notifying about ticket %s', ticket.key)
                tg_client.send_message(
                    '❗ @{asgn}, critical ticket [{summ}](https://st.yandex-team.ru/{key}) is waiting for you'.format(
                        asgn=tg_client.escape(get_tg_login(ticket.assignee.login, st_token)),
                        summ=tg_client.escape(ticket.summary), key=ticket.key,
                    )
                )
            _log.info('Untriaged blockers notifying is done')

            # handle tickets with set components
            _log.info('Going to process tickets with set components (triaged or new-bug-created)')
            for person_on_duty, component_id in (
                (get_persons_on_duty('back', gore_token=gore_token).primary, BACKEND_COMPONENT_ID),
                (get_persons_on_duty('front', gore_token=gore_token).primary, FRONTEND_COMPONENT_ID),
                (get_persons_on_duty('product', gore_token=gore_token).primary, PRODUCT_COMPONENT_ID),
                (get_persons_on_duty('back', gore_token=gore_token).primary, STATFACE_COMPONENT_ID),
            ):
                _log.info('Working with component_id %s (%s on duty)', component_id, person_on_duty)
                tickets = st_client.issues.find(
                    filter={
                        'queue': 'DLHELP', 'resolution': ST_EMPTY, 'type': 'task',
                        'components': component_id, 'assignee': ST_EMPTY
                    }
                )
                for ticket in tickets:
                    _log.info('Assigning ticket %s', ticket.key)
                    assign_ticket(ticket, person_on_duty)

            _log.info('Tickets with components have been processed')

            # handle brand new tickets
            _log.info('Going to process new tickets')
            common_primary = get_persons_on_duty('common', gore_token=gore_token).primary
            new_tickets = st_client.issues.find(
                filter={
                    'queue': 'DLHELP', 'resolution': ST_EMPTY, 'status': 'new',
                    'type': 'task', 'assignee': ST_EMPTY,
                }
            )
            for ticket in new_tickets:
                if 'Editor' in [c.name for c in ticket.components]:
                    continue
                _log.info('Assigning ticket %s', ticket.key)
                assign_ticket(ticket, common_primary)

            _log.info('New tickets have been processed')

            send_event_to_juggler(
                host='new_tickets',
                service='DLHELP',
                status='OK',
                description='Ok'
            )

        main()
