import logging
import time

from sandbox import sdk2
from sandbox.projects.common import task_env
from sandbox.sandboxsdk import environments


_log = logging.getLogger(__name__)


class DatalensDutyJugglerSetter(sdk2.Task):

    class Requirements(task_env.TinyRequirements):
        environments = [
            # List of required packages
            environments.PipEnvironment('pyyaml'),
            environments.PipEnvironment('requests'),
        ]

    class Parameters(sdk2.Parameters):
        juggler_config_url = sdk2.parameters.String(
            'juggler_config_url',
            required=False,
            default=(
                'https://a.yandex-team.ru/api/tree/blob/'
                'trunk/arcadia/datalens/backend/tools/environments/monitoring_config.yaml'
            )
        )

    def on_execute(self):
        import yaml
        import requests

        def get_persons_on_duty_from_gore_api(_gore_token):  # kind of copypaste from dlhelp_helper
            r = requests.get(
                'https://resps-api.cloud.yandex.net/api/v0/duty/datalens-back',
                headers={'Authorization': 'OAuth {}'.format(_gore_token)}
            )
            if r.status_code != 200:
                _log.error('Gore api returned %s: %s', r.status_code, r.content)
            r.raise_for_status()
            data = r.json()
            tstamp = int(time.time())
            primary = None
            backup = None

            for item in data:
                if item['datestart'] <= tstamp < item['dateend']:
                    if item['resp']['order'] == 0:
                        primary = item['resp']['username']
                    elif item['resp']['order'] == 1:
                        backup = item['resp']['username']
                    else:
                        _log.error('Unknown resp order: %s', item['resp'])

            return [primary, backup]

        secret = sdk2.yav.Secret('sec-01ekmt2r5p50hp5qw875dewfgw')
        juggler_token = secret.data()['juggler_token']
        arcanum_token = secret.data()['arcanum_token']
        gore_token = sdk2.yav.Secret('sec-01d4jkj89g0gac6c3nm2nqq5q3').data()['bb_oauth']

        r = requests.get(
            self.Parameters.juggler_config_url,
            headers={'Authorization': 'OAuth {}'.format(arcanum_token)}
        )
        r.raise_for_status()
        config = yaml.load(r.content)
        logins = get_persons_on_duty_from_gore_api(_gore_token=gore_token)

        for rule in config['rules']:
            template_kwargs = rule['template_kwargs'].copy()
            if 'login' in template_kwargs:
                template_kwargs.update({
                    'login': logins + [lg for lg in template_kwargs['login'] if lg not in logins]
                })
            elif 'logins' in template_kwargs:
                template_kwargs.update({
                    'logins': logins + [lg for lg in template_kwargs['logins'] if lg not in logins]
                })
            else:
                raise Exception('Do not understand this template_kwargs: %s' % template_kwargs)

            r = requests.post(
                'https://juggler-api.yandex-team.ru/api/notify_rules/add_or_update_notify_rule',
                json={
                    'rule_id': rule['rule_id'],
                    'selector': rule['selector'],
                    'template_name': rule['template_name'],
                    'namespace': rule['namespace'],
                    'template_kwargs': template_kwargs
                },
                params={'do': '1'},
                headers={'Authorization': 'OAuth {}'.format(juggler_token)}
            )
            if r.status_code >= 400:
                logging.warning('Juggler api fail: %s', r.content)

            r.raise_for_status()
