import logging
import time

import requests
import collections


class HitmanFrontApi:
    def __init__(
            self,
            oauth_token,
            backend_url='https://hitman.yandex-team.ru/front/',
    ):
        self.oauth_token = oauth_token
        self.backend_url = backend_url
        self._headers = {
            'Authorization': 'OAuth {key}'.format(key=oauth_token)
        }

    def get(
            self,
            path,
            params=collections.defaultdict(),
            headers=collections.defaultdict(),
    ):
        final_headers = self._headers.copy()
        final_headers.update(headers)

        return requests.get(
            self.backend_url + path,
            params=params,
            headers=final_headers,
            verify=False
        ).json()


class HitmanException(Exception):
    pass


class HitmanApiHelper:
    def __init__(
            self,
            oauth_token,
            cache_path,
            yt_client,
    ):
        self.oauth_token = oauth_token
        self.cache_path = cache_path
        self.yt_client = yt_client
        self.hitman_api = HitmanFrontApi(oauth_token)

    def get_status_batch(
            self,
            job_ids,
    ):
        statuses = self._get_cached_status_batch(job_ids)
        updates = {}
        for job_id in job_ids:
            if job_id and job_id not in statuses:
                status = self._get_status_from_hitman(job_id)
                statuses[job_id] = status
                updates[job_id] = status
        self._update_cache(updates)
        return statuses

    def _get_cached_status_batch(
            self,
            job_ids,
    ):
        statuses = {}
        query = [{'hitman_job': job_id} for job_id in job_ids]
        res = list(self.yt_client.lookup_rows(self.cache_path, query))
        for row in res:
            if self._is_terminal(row["status"]) or not self._is_invalidated(row["timestamp"]):
                statuses[row["hitman_job"]] = {
                    "status": row["status"],
                    "finish_ts": row["finish_ts"]
                }

        return statuses

    def _update_cache(self, data):
        ts = time.time()
        items = [
            {
                "hitman_job": job_id,
                "status": status["status"],
                "finish_ts": status["finish_ts"],
                "timestamp": ts,
            } for job_id, status in data.iteritems()
        ]
        self.yt_client.insert_rows(self.cache_path, items, format="yson", raw=False, update=True)

    def _get_status_from_hitman(self, job_id):
        delay = 1.0
        multiplier = 1.5
        for i in xrange(15):
            try:
                response = self.hitman_api.get("jobs/{}".format(job_id))
                return {
                    "status": response["status"],
                    "finish_ts": response["finishTs"]
                }
            except HitmanException as e:
                logging.info("Hitman API error: {}, sleep for %s sec", e.message, delay)
                time.sleep(delay)
                delay *= multiplier
        raise HitmanException("Can't get Hitman status for job {}".format(job_id))

    @staticmethod
    def _is_terminal(status):
        return status and status in TERMINAL_STATUSES

    @staticmethod
    def _is_invalidated(timestamp):
        return time.time() - timestamp > INVALIDATE_TIMEOUT


CACHE_TABLE_SCHEMA = [
    {"name": "hitman_job", "type": "string"},
    {"name": "status", "type": "string"},
    {"name": "finish_ts", "type": "string"},
    {"name": "timestamp", "type": "double"},
]

CACHE_TABLE_KEYS = ["hitman_job"]

TERMINAL_STATUSES = ["SUCCEEDED"]

INVALIDATE_TIMEOUT = 5 * 60
