# coding=utf-8

import os
import sys

import sandbox.projects.dbd.commons.evaluate_bt_score_job as bt_score_job
import sandbox.projects.dbd.commons.hitman_api as hitman_api
import sandbox.projects.dbd.commons.utils as utils
import sandbox.sandboxsdk.svn as sdk_svn
import sandbox.sandboxsdk.environments as sdk_environments
from sandbox import sdk2


class ProcessDbdRawLogs(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            sdk_environments.PipEnvironment('yandex-yt', '0.8.17-0'),
            sdk_environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

    class Parameters(sdk2.Task.Parameters):
        yt_token = sdk2.parameters.String(
            "Vault with YT Token",
            required=True,
        )

        yt_cluster = sdk2.parameters.String(
            "YT Cluster",
            required=True,
            default="hahn",
        )

        hitman_token = sdk2.parameters.String(
            "Vault with Hitman Token",
            required=True,
        )

        hitman_status_cache = sdk2.parameters.String(
            "Path to Hitman cache",
            required=True,
            default='//home/fivecg/dyntables/cache/dbd_hitman_cache',
        )

        raw_logs_path = sdk2.parameters.String(
            "Path to raw logs table",
            required=True,
        )

        result_path = sdk2.parameters.String(
            "Path to result table",
            required=True,
        )

    @staticmethod
    def read_table(table_name, yt_client):
        import yt.wrapper as yt
        if yt.exists(table_name, client=yt_client):
            return [line for line in yt.read_table(table_name, client=yt_client)]
        else:
            return []

    def on_execute(self):

        sys.path.append(
            sdk_svn.Arcadia.get_arcadia_src_dir(
                "arcadia:/arc/trunk/arcadia/quality/mstand_metrics/offline_production/infra/dbd/nirvana_ops"
            )
        )

        import ytlib.yt_io_helpers as ytio
        import yt.wrapper as yt
        import pandas

        hitman_token = sdk2.Vault.data(self.Parameters.hitman_token)
        cache_path = self.Parameters.hitman_status_cache

        os.environ['YT_TOKEN'] = str(sdk2.Vault.data(self.Parameters.yt_token))
        yt_client = ytio.create_client(str(self.Parameters.yt_cluster))

        hitman_api_helper = hitman_api.HitmanApiHelper(
            oauth_token=hitman_token,
            cache_path=cache_path,
            yt_client=yt_client,
        )

        raw_logs = ProcessDbdRawLogs.read_table(
            table_name=self.Parameters.raw_logs_path,
            yt_client=yt_client,
        )
        df = pandas.DataFrame(raw_logs)

        grouped = df.groupby(["hitman-project", "hitman-process", "hitman-job"])
        jobs = [bt_score_job.Job.from_df(group) for key, group in grouped]

        utils.batch_update_job_statuses(
            jobs=jobs,
            batch_size=500,
            hitman_api_helper=hitman_api_helper,
        )

        yt.remove(
            self.Parameters.result_path,
            client=yt_client,
        )

        yt.create_table(
            path=self.Parameters.result_path,
            recursive=True,
            attributes={
                "external": False,
                "dynamic": False,
            },
            client=yt_client,
        )
        ytio.save_items_to_static_table(
            items=[job.to_dyn_table_item() for job in jobs],
            table_name=str(self.Parameters.result_path),
            yt_client=yt_client,
            append=True,
        )
