# coding=utf-8

import os
import stat
import logging

from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.dc3.dc_release.common import dc_release_from_branch
from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.sdk2.helpers.process import subprocess
from sandbox.sdk2.environments import Xcode
import sandbox.common.types.resource as ctr
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc


class DcReleaseMac(dc_release_from_branch.DCReleaseFromBranchTaskMixin, sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 2
        ram = 3000
        disk_space = 20 * 1024
        dns = ctm.DnsType.DNS64
        # IPV4 необходим из-за того, что нотифаер в altool не работает с ipv6, даже под nat64 (возможно, обновление XCode может помочь)
        client_tags = (ctc.Tag.OSX_CATALINA | ctc.Tag.OSX_MOJAVE | ctc.Tag.OSX_BIG_SUR | ctc.Tag.OSX_MONTEREY) & ctc.Tag.USER_MSSNGR_ELECTRON_BUILDER & ctc.Tag.IPV4
        environments = (
            sdk2.environments.NodeJS('12.13.0', 'osx'),
        )

    class Parameters(sdk2.Task.Parameters):
        _branch_params = dc_release_from_branch.dc_release_from_branch_task_parameters()

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

    @property
    def binary_executor_query(self):
        return {
            "attrs": {
                "task_type": "DC_RELEASE",
                "released": self.Parameters.binary_executor_release_type,
                "platform": "osx"
            },
            "state": [ctr.State.READY]
        }

    def get_env(self):
        logging.debug('Initializing environments: {}'.format(os.environ))

        command_env = os.environ.copy()

        command_env["CSC_LINK"] = self.Parameters.tokens.data()["developer-id-cert"]
        command_env["CSC_KEY_PASSWORD"] = self.Parameters.tokens.data()["developer-id-cert-password"]

        return command_env

    def on_execute(self):
        logging.info("Start build dc_release_mac_task")

        with arcadia_sdk.mount_arc_path(self.Parameters.arcadia_url, arc_oauth_token=self.Parameters.tokens.data()["arc-token"]) as arcadia_path:
            logging.info('Mount path: {}'.format(arcadia_path))

            repo_path = os.path.join(arcadia_path, 'adv', 'frontend', 'services', 'commander')

            self.save_secret("notarizeAppleId.js", os.path.join(repo_path, "scripts/notarizeAppleId.js"))
            self.save_secret("notarizeApplePassword.js", os.path.join(repo_path, "scripts/notarizeApplePassword.js"))
            self.save_secret("signerToken.js", os.path.join(repo_path, "scripts/signerToken.js"))
            self.save_secret("mds-access-key", os.path.join(repo_path, "scripts/mdsAccessKey"), False)
            self.save_secret("mds-secret-key", os.path.join(repo_path, "scripts/mdsSecretKey"), False)
            self.save_secret("developer-id-cert", os.path.join(repo_path, "scripts/developerIdCert.js"))

            prepare_path = os.path.join(repo_path, "scripts/sandbox/prepare_mac.sh")
            release_path = os.path.join(repo_path, "scripts/sandbox/release_mac.sh")

            for script_path in (prepare_path, release_path):
                os.chmod(script_path, os.stat(script_path).st_mode | stat.S_IEXEC)

            with sdk2.helpers.ProcessLog(self, logger='build') as pl:
                subprocess.check_call(
                    prepare_path,
                    env=self.get_env(),
                    cwd=repo_path,
                    shell=True,
                    stdout=pl.stdout,
                    stderr=pl.stderr
                )

            Xcode("11.2.1").prepare()

            with sdk2.helpers.ProcessLog(self, logger='build') as pl:
                subprocess.check_call(
                    release_path,
                    env=self.get_env(),
                    cwd=repo_path,
                    shell=True,
                    stdout=pl.stdout,
                    stderr=pl.stderr
                )

    def save_secret(self, secret_name, secret_path, is_file_secret=True):
        logging.debug('Save secret {}'.format(secret_name))
        file = open(secret_path, 'w')

        secret = self.Parameters.tokens.data()[secret_name]
        file.write(secret.decode("base64") if is_file_secret else secret)

        file.close()
