# coding=utf-8

import os
import logging

from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.dc3.dc_release.common import dc_release_from_branch
from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.sdk2.helpers.process import subprocess
import sandbox.common.types.resource as ctr
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc


class DcReleaseWin(dc_release_from_branch.DCReleaseFromBranchTaskMixin, sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 2
        ram = 3000
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.WINDOWS
        environments = (
            sdk2.environments.NodeJS('12.13.0', 'win_nt'),
        )

    class Parameters(sdk2.Task.Parameters):
        _branch_params = dc_release_from_branch.dc_release_from_branch_task_parameters()

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

    @property
    def binary_executor_query(self):
        return {
            "attrs": {
                "task_type": "DC_RELEASE",
                "released": self.Parameters.binary_executor_release_type,
                "platform": "win"
            },
            "state": [ctr.State.READY]
        }

    def get_env(self):
        logging.debug('Initializing environments: {}'.format(os.environ))

        command_env = os.environ.copy()

        # Omit /bin dir, as NodeJS for Windows does not have it
        node_bindir = os.path.dirname(self.Requirements.environments[0].bindir)
        command_env["PATH"] = node_bindir + ";" + command_env["PATH"]

        return command_env

    def on_execute(self):
        logging.info("Start build...")

        with arcadia_sdk.mount_arc_path(self.Parameters.arcadia_url, arc_oauth_token=self.Parameters.tokens.data()["arc-token"]) as arcadia_path:
            logging.info('Mount path: {}'.format(arcadia_path))

            repo_path = os.path.join(arcadia_path, 'adv', 'frontend', 'services', 'commander')

            self.save_secret("notarizeAppleId.js", os.path.join(repo_path, "scripts/notarizeAppleId.js"))
            self.save_secret("notarizeApplePassword.js", os.path.join(repo_path, "scripts/notarizeApplePassword.js"))
            self.save_secret("signerToken.js", os.path.join(repo_path, "scripts/signerToken.js"))
            self.save_secret("mds-access-key", os.path.join(repo_path, "scripts/mdsAccessKey"), False)
            self.save_secret("mds-secret-key", os.path.join(repo_path, "scripts/mdsSecretKey"), False)
            self.save_secret("developer-id-cert", os.path.join(repo_path, "scripts/developerIdCert.js"))

            win_path = os.path.join(repo_path, "scripts/sandbox/release_win.bat")

            with sdk2.helpers.ProcessLog(self, logger='build') as pl:
                subprocess.check_call(
                    win_path,
                    env=self.get_env(),
                    cwd=repo_path,
                    shell=True,
                    stdout=pl.stdout,
                    stderr=pl.stderr
                )

    def save_secret(self, secret_name, secret_path, is_file_secret=True):
        logging.debug('Save secret {}'.format(secret_name))
        file = open(secret_path, 'w')

        secret = self.Parameters.tokens.data()[secret_name]
        file.write(secret.decode("base64") if is_file_secret else secret)

        file.close()
