import logging
import time
from datetime import datetime
from sandbox.projects.release_machine.helpers.deploy.ya_deploy_releaser import select_deploy_tickets

DEPLOY_API_HOST_DEFAULT = "xdc.yp.yandex.net"
YP_API_PORT = '8090'
YP_API_HOST_KEY = 'host'
YP_API_TOKEN_KEY = 'token'

NO_TICKET_TIMEOUT = 5 * 60
WAIT_TICKETS_TOTAL_TIMEOUT = 75 * 60
TICKET_REQUEST_INTERVAL = 15


def is_all_closed(tickets):
    for ticket in tickets:
        if isinstance(ticket.progress, dict) and ticket.progress.get("closed"):
            logging.info("Ticket %s (stage %s) is closed", ticket.ticket_id, ticket.stage_id)
            continue
        logging.info("Ticket %s (stage %s) is not closed", ticket.ticket_id, ticket.stage_id)
        return False
    return True


class DeployTicketWatcher:

    yp_client = None

    def __init__(self, params=None):
        from yp import client
        if isinstance(params, client.YpClient):
            self.yp_client = params
        elif isinstance(params, dict) and YP_API_TOKEN_KEY in params:
            host=params.get(YP_API_HOST_KEY, DEPLOY_API_HOST_DEFAULT)
            self.yp_client = client.YpClient(
                '{}:{}'.format(host, YP_API_PORT),
                config={"token": params.get(YP_API_TOKEN_KEY)}
            )
        elif isinstance(params, str):
            self.yp_client = client.YpClient(
                '{}:{}'.format(DEPLOY_API_HOST_DEFAULT, YP_API_PORT),
                config={"token": params}
            )
        else:
            raise Exception('Error init yp client')

    def wait_release_tickets(self, release_id, stages):
        start_time = datetime.now()
        logging.info("Wait tickets in stages: %s", ", ".join(map(str, stages)))
        while True:
            all_tickets = select_deploy_tickets(release_id, self.yp_client)
            tickets = filter(lambda ticket : ticket.stage_id in stages, all_tickets)
            logging.info("Filter (%s) tickets by stages", len(tickets))
            now = datetime.now()
            delta = (now - start_time).total_seconds()

            if not tickets and delta > NO_TICKET_TIMEOUT:
                raise Exception('No deploy tickets found for release {} after {} sec'.format(release_id, NO_TICKET_TIMEOUT))

            if delta > WAIT_TICKETS_TOTAL_TIMEOUT:
                raise Exception('Wait tickets timeout for release {} after {} sec'.format(release_id, WAIT_TICKETS_TOTAL_TIMEOUT))

            if tickets and is_all_closed(tickets):
                return True

            time.sleep(TICKET_REQUEST_INTERVAL)
