# -*- coding: utf-8 -*-
import logging
import json
import datetime
import time
import requests

from sandbox import sdk2
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.common.startrek_utils import ST


class BotStatistic(sdk2.Task):
    """
    Various notification for devops vanguard.

    * List of incidents without "Sotrudniki->Dezhurniy" field set
    * List of incidents that has no actions for too long
    * To assign next guardians of the realm
    """

    class Requirements(sdk2.Requirements):
        environments = (
            sdk2.environments.PipEnvironment(
                'startrek_client', version="2.5.0", custom_parameters=["--upgrade-strategy only-if-needed"]
            ),
            sdk2.environments.PipEnvironment('yandex-yt', use_wheel=True),
            sdk2.environments.PipEnvironment('yandex-yt-yson-bindings', use_wheel=True),
            sdk2.environments.PipEnvironment('yandex-yt-yson-bindings-skynet', use_wheel=True),
        )
        client_tags = task_env.TaskTags.startrek_client

    class Parameters(sdk2.Task.Parameters):

        override_receiver = sdk2.parameters.Bool(
            "Send telegram messages to other person, instead of devops chat (for debugging)",
            default=False,
        )
        with override_receiver.value[True]:
            receiver = sdk2.parameters.String(
                "login of receiver",
                required=True,
                default='lebedev-aa',
            )
        created = sdk2.parameters.String(
            "Time created ticket",
            default='2021-01-01',
        )
        with sdk2.parameters.Group('Tokens') as token:
            solomon_token = sdk2.parameters.String(
                'SOLOMON token',
                default='solomon-token-robot-gencfg',
                required=True
            )
            yt_token = sdk2.parameters.String(
                'YT token',
                default='yt-token-robot-gencfg2',
                required=True
            )

    class Context(sdk2.Task.Context):
        forsaken_spi = []
        forsaken_action_items = []
        details_issues = []

    def on_execute(self):
        import yt.wrapper as yt
        self.yt = yt
        self.yt_client = self.yt.YtClient('hahn', sdk2.Vault.data('natapolienko', self.Parameters.yt_token))
        self.st_token = sdk2.Vault.data('SEARCH-RELEASERS', 'robot-search-devops_st_token')
        self.st_helper = STHelper(self.st_token, 'robot-search-devops')
        self.solomon_token = sdk2.Vault.data('natapolienko', self.Parameters.solomon_token)
        self.all_statistic_ping()

    def all_statistic_ping(self):
        spi_count = {}
        query = ST.Query(
            not_closed=False,
            created=self.Parameters.created
        ).build_query()
        issues = ST.find_tickets_by_query(self.st_helper, query)
        # for debug
        # issues = [self.st_helper.st_client.issues['SPI-11135']]
        for issue in issues:
            count_ping = ST.count_ping_comments_before(issue, 'robot-search-devops')
            if count_ping:
                spi_count[issue.key] = {'count_ping': count_ping, 'closed': False, 'ydt': False}
                spi_count[issue.key]['link_spi'] = 'https://st.yandex-team.ru/{}'.format(issue.key)
                if issue.status.key == 'closed':
                    spi_count[issue.key]['closed'] = True
                if issue.ydt >= 0:
                    spi_count[issue.key]['ydt'] = True
                    for update in list(issue.changelog.get_all()):
                        if update.updatedBy.login == 'robot-search-devops':
                            first_ping_parsed = datetime.datetime.strptime(update.updatedAt[:-9], '%Y-%m-%dT%H:%M:%S')
                            break
                    logging.info('first ping %s', first_ping_parsed)
                    change_ydt = list(issue.changelog.get_all(field='ydt'))
                    if change_ydt:
                        update_time_ydt = datetime.datetime.strptime(change_ydt[0].updatedAt[:-9], '%Y-%m-%dT%H:%M:%S')
                        logging.info('ydt time %s', update_time_ydt)
                        if update_time_ydt > first_ping_parsed:
                            spi_count[issue.key]['ydt_with_bot'] = True
                        else:
                            spi_count[issue.key]['ydt_with_bot'] = False
                spi_count[issue.key]['impact'] = issue.impact if issue.impact else ""
        self.push_yt_table(spi_count)
        self.push_solomon_data(spi_count)

    def push_yt_table(self, input_data):
        rows = []
        for key, value in input_data.items():
            row = {'spi': key}
            row.update(value)
            rows.append(row)
        logging.info(rows)
        if not self.yt_client.exists("//home/searchmon/stats/bot_stat"):
            schema = [
                {"name": "spi", "type": "string"},
                {"name": "link_spi", "type": "string"},
                {"name": "count_ping", "type": "int64"},
                {"name": "impact", "type": "string"},
                {"name": "ydt", "type": "boolean"},
                {"name": "ydt_with_bot", "type": "boolean"},
                {"name": "closed", "type": "boolean"},
            ]
            self.yt_client.create("table", "//home/searchmon/stats/bot_stat", attributes={"schema": schema, "primary_medium": "ssd_blobs"})
        self.yt_client.write_table(self.yt_client.TablePath("//home/searchmon/stats/bot_stat", append=False), rows, format=self.yt.JsonFormat(attributes={"encode_utf8": True}))

    def push_solomon_data(self, input_table):
        solomon_data_api_url = 'https://solomon.yandex.net/api/v2/push?project={}&cluster={}&service={}'
        project = 'robot_search_devops'
        cluster = 'bot_stat'
        service = 'count_ping'
        headers = {
            'Authorization': 'OAuth {}'.format(self.solomon_token),
            'Content-Type': 'application/json',
            'Accept': 'application/json',
        }
        data = {
            'sensors': []
        }
        ind_row = 0
        for input_key, input_value in input_table.items():
            ind_row += 1
            sensor = {
                'labels': {
                    'spi': input_key,
                    'link_spi': input_value['link_spi'],
                    'closed': input_value['closed'],
                    'ydt': input_value['ydt'],
                    'impact': input_value['impact'],
                    'ydt_with_bot': input_value['ydt_with_bot'] if input_value.get('ydt_with_bot') else ""
                },
                'value': input_value['count_ping'],
            }
            data['sensors'].append(sensor)
            if ind_row % 1000 == 0:
                logging.info('data lens {}, ind_row {}'.format(len(data['sensors']), ind_row))
                for delay in (0, 3, 6, 9, -1):
                    try:
                        logging.info('Start push solomon data for ({} sensors)'.format(len(data['sensors'])))
                        logging.info(json.dumps(data))
                        response = requests.post(solomon_data_api_url.format(project, cluster, service), headers=headers, data=json.dumps(data))
                        logging.info('request : url {}', response.request.url)
                        if response.status_code != 200:
                            raise ValueError('HTTP Error {} != 200 ({})'.format(response.status_code, response.text))
                    except Exception as e:
                        if delay >= 0:
                            logging.error(' {}: {}, sleep {}'.format(type(e), e, delay))
                            time.sleep(delay)
                        else:
                            logging.error('{}: {}, FAIL'.format(type(e), e))
                            raise
                data = {
                    'sensors': []
                }
        for delay in (0, 3, 6, 9, -1):
            try:
                logging.info('Start push solomon data for ({} sensors)'.format(len(data['sensors'])))
                response = requests.post(solomon_data_api_url.format(project, cluster, service), headers=headers, data=json.dumps(data))
                if response.status_code != 200:
                    raise ValueError('HTTP Error {} != 200 ({})'.format(response.status_code, response.text))
            except Exception as e:
                if delay >= 0:
                    logging.error(' {}: {}, sleep {}'.format(type(e), e, delay))
                    time.sleep(delay)
                else:
                    logging.error('{}: {}, FAIL'.format(type(e), e))
                    raise
