# -*- coding: utf-8 -*-
import logging
import json
import datetime
import time
import re
import requests

from sandbox import sdk2

import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as rm_task_env
from sandbox.projects.release_machine import rm_notify
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.common.startrek_utils import ST
from sandbox.projects.common.staff import StaffClient
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import task_env
from sandbox.projects.common import requests_wrapper

from sandbox.projects.devops.DevOpsAlert.main_config import (
    ACTION_ITEM_DELIMS,
    IGNORE_QUEUE,
    BANNER_SYSTEM_DEPARTMENT_ID
)

from sandbox.projects.devops.SupportAlert.templates import (
    RESPONSIBLE,
    BASE_TG_TEXT_TEMPLATED,
    TEXT_WITHOUT_RESOLUTION,
    TEXT_EXISTS_SPPROBLEM,
    TEXT_MAYBE_EXISTS_SPPROBLEM,
    TEXT_EXISTS_ACTION_ITEMS,
    ACTION_ITEM_PING_TEMPLATE,
    BASE_RTCSUPPORT_TEXT,
    BASE_RESPONSIBLE_TEXT
)

BASE_VIDEO_TEXT = (
    "Привет всем! Прошло много дней после инцидента. Пожалуйста:\n"
    "1) опишите влияние на пользователя - как проявилось, какой процент затронуто. "
    "Посчитайте даунтайм, если не работала какая-то функциональность.\n"

    "2) опишите причину инцидента, с подкреплением нужными графиками\n"

    "3) опишите шаги для недопущения повторения в будущем: "
    "нужно описать в тикете, завести и прилинковать action item "
    "(тикет, в котором будет проставлен тег spi:actionitem) **(ВАЖНО)** "
    "(например VIDEOPOISK13894)\n"

    "4) после 1-3 можно закрыть тикет как разобранный.\n"

    "5) если инцидент повторяющийся - нужно завести st/SPPROBLEM "
    "или к существующему SPP прилинковать инцидент (обязательно SPI должен быть в секции 'Blocker to' для SPP); "
    "для таких случаев SPI можно сразу закрывать. Например, SPPROBLEM262\n"

    "6) если SPI является дубликатом - можно закрыть как дубликат (не забыть закрыть "
    "именно как дубликат чтобы прилинковать корневой тикет)"
)

BLOCKED_QUEUE = (
    'SPI-',
    'RTCSUPPORT-',
    'QLOUDSUPPORT-',
    'INFRAINCIDENT-',
)

FILTER_QUEUE = (
    'RELEASE-',
    'SPPROBLEM-',
    'LSR-',
    'SPI-',
    'CORPMAIL-',
    'DISKSUP-',
    'CLOUDSUPPORT-',
    'GENCFG-',  # paranoid owner here
    'CAPTCHA-',  # restricted access
)

FIELD_POSSIBLE_VALUES = {
    'Impact': [
        'internal',
        'external',
    ],
    'Support_line': [
        'marty',
        'devops',
        'service',
    ],
    'SRE Area': [
        'human',
        'operations',
        'release',
        'bug',
        'tools',
        'service',
        'net',
        'hardware',
        'platform',
        'monitoring',
        'mds',
        'apphost',
        'balancer',
        'saas',
        'external',
        'its',
        'config',
    ]
}

STANDARD_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    # first list for l1,l2 and l3; second for l2 and l3; third for l3
    'extra_summonees': [
        ["natapolienko"],
        ["mvel"],
        ["dmitryno"],
    ],
    'st_product': "",
    'st_queue': "SPI",
    'incident_silence_time': 14,
    'action_item_silence_time': 60,
    'tags_to_check': ['area', 'impact', 'support_line'],
    'fields_to_check': ['SRE Area', 'Impact', 'Support_line'],
    'sre_times_filter': '("Notification Time": empty() OR "Begin Time": empty() OR "End Time": empty())',
    'ignore_tags': [
        "product:PROD-WEB",
        "product:PROD-COMMON",
        "product:PROD-IMAGES",
        "product:PROD-COLLECTIONS",
        "product:PROD-SEARCHAPP",
        "product:PROD-VIDEO",
        "product:PROD-ALICE",
        "product: PROD-BANNER",
        "product:PROD-DIRECT",
        "product:PROD-METRIKA",
        "product:PROD-PCODE",
        "product:PROD-MODADVERT",
        "product:PROD-RMP",
    ],
    'ping_action_items': True,
    'ping_forsaken_incidents': False,
    'check_forsaken_ticket': False,
    'tg_text': BASE_RESPONSIBLE_TEXT,
    'forsaken_incidents_extra_filter': 'Status: !"Проведён анализ"',
    'skip_weekends': True,
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'ping_development': True,
    'reopen_closed_inc_without_ydt': True,
    'ping_incident_with_empty_downtime': True,
    'change_weight_hwproblems': True,
    'relevant_people_closed_incident': True,
}

WEB_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    # first list for l1,l2 and l3; second for l2 and l3; third for l3
    'extra_summonees': [
        ["natapolienko"],
        ["mvel"],
        ["dmitryno"],
    ],
    'st_product': "PROD-WEB",
    'st_queue': "SPI",
    'incident_silence_time': 14,
    'action_item_silence_time': 60,
    'tags_to_check': ['area', 'impact', 'support_line'],
    'fields_to_check': ['SRE Area', 'Impact', 'Support_line'],
    'sre_times_filter': '("Notification Time": empty() OR "Begin Time": empty() OR "End Time": empty())',
    'ignore_tags': [],
    'ping_action_items': True,
    'ping_forsaken_incidents': True,
    'check_forsaken_ticket': True,
    'tg_text': BASE_RESPONSIBLE_TEXT,
    'forsaken_incidents_extra_filter': 'Status: !"Проведён анализ"',
    'skip_weekends': True,
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'day_for_ping_tg': [0],
    'ping_development': True,
    'relevant_people_closed_incident': True,
    'ping_incident_with_empty_downtime': True,
}

COMMON_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    # first list for l1,l2 and l3; second for l2 and l3; third for l3
    'extra_summonees': [
        ["natapolienko"],
        ["mvel"],
        ["dmitryno"],
    ],
    'st_product': "PROD-COMMON",
    'st_queue': "SPI",
    'incident_silence_time': 14,
    'action_item_silence_time': 60,
    'tags_to_check': ['area', 'impact', 'support_line'],
    'fields_to_check': ['SRE Area', 'Impact', 'Support_line'],
    'sre_times_filter': '("Notification Time": empty() OR "Begin Time": empty() OR "End Time": empty())',
    'ignore_tags': [],
    'ping_action_items': True,
    'ping_forsaken_incidents': True,
    'check_forsaken_ticket': True,
    'tg_text': BASE_RESPONSIBLE_TEXT,
    'forsaken_incidents_extra_filter': 'Status: !"Проведён анализ"',
    'skip_weekends': True,
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'day_for_ping_tg': [0],
    'ping_development': True,
    'relevant_people_closed_incident': True,
    'ping_incident_with_empty_downtime': True,
}

IMAGES_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'extra_summonees': [["ulgen4a"]],  # first list for l1,l2 and l3; second for l2 and l3; third for l3
    'st_product': "PROD-IMAGES",
    'st_queue': "SPI",
    'incident_silence_time': 3,
    'action_item_silence_time': 14,
    'ignore_tags': [],
    'ping_action_items': True,
    'ping_forsaken_incidents': True,
    'check_forsaken_ticket': True,
    'tg_text': RESPONSIBLE.format(
        devops_doc_link='https://nda.ya.ru/3UVFny',
        alt_devops_master='ulgen4a',
    ),
    'forsaken_incidents_extra_filter': 'Status: !"Проведён анализ"',
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'ping_development': True,
    'relevant_people_closed_incident': True,
    'ping_incident_with_empty_downtime': True,
}

COLLECTIONS_CONFIG = {
    'st': {
        'token_name': 'robot-pdbdevops-startrek-token',
        'token_owner': 'YASAP',
        'useragent': 'robot-pdbdevops',
    },
    'extra_summonees': [],  # first list for l1,l2 and l3; second for l2 and l3; third for l3
    'st_product': "PROD-COLLECTIONS",
    'st_queue': "SPI",
    'incident_silence_time': 3,
    'action_item_silence_time': 14,
    'ignore_tags': [],
    'ping_action_items': True,
    'ping_forsaken_incidents': False,
    'check_forsaken_ticket': False,
    'tg_text': BASE_RESPONSIBLE_TEXT,
    'forsaken_incidents_extra_filter': 'Status: !"Проведён анализ"',
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'ping_development': True,
    'relevant_people_closed_incident': True,
    'ping_incident_with_empty_downtime': True,
}

MARTY_CONFIG = {
    'st': {
        'token_name': rm_const.COMMON_TOKEN_NAME,
        'token_owner': rm_const.COMMON_TOKEN_OWNER,
        'useragent': rm_const.ROBOT_RELEASER_USER_NAME,
    },
    'marties': [
        'damoka',
        'svagalakov',
        'agavelya',
        'gvdozd',
        'tedkon',
    ],
    'persmarties': [
        'groovik',
        'epsilond1',
        'harond1',
        'mickxolotl',
    ],
    'efirmarties': [
        'redheadbonk',
        'alexunix',
        'mrt0rtikize',
    ],
    'st_queue': "SPI",
    'st_product': None,
    'ignore_tags': ['product:web'],
    'issues_ping_error_threshold': 100,
    'tg_text': BASE_RESPONSIBLE_TEXT,
    'forsaken_incidents_extra_filter': 'Status: !"Проведён анализ"',

    'sro_times_filter': (
        'Status: "Закрыт" AND '
        '("SRO Notification Time": empty() OR "SRO Begin Time": empty() OR "SRO End Time": empty())'
    ),
    'sre_times_filter': (
        'Status: "Закрыт" AND '
        '("Notification Time": empty() OR "Begin Time": empty() OR "End Time": empty())'
    ),
    'ping_people': True,
    'ping_development': True,
}

RM_CONFIG = {
    'st': {
        'token_name': rm_const.COMMON_TOKEN_NAME,
        'token_owner': rm_const.COMMON_TOKEN_OWNER,
        'useragent': rm_const.ROBOT_RELEASER_USER_NAME,
    },

    # first list for l1,l2 and l3; second for l2 and l3; third for l3
    'extra_summonees': [["mvel"]],
    'st_product': None,
    'st_queue': "RMINCIDENTS",
    'incident_silence_time': 7,
    'action_item_silence_time': 14,
    'ignore_tags': [],
    'ping_action_items': True,
    'ping_forsaken_incidents': True,
    'check_forsaken_ticket': True,
    'tg_text': BASE_TG_TEXT_TEMPLATED + RESPONSIBLE.format(
        devops_doc_link='https://nda.ya.ru/3Tvds7',
        alt_devops_master='ilyaturuntaev',
    ),
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'ping_development': True,
    'relevant_people_closed_incident': False,
}

LSR_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'extra_summonees': [["natapolienko"]],
    'st_product': "",
    'st_queue': "LSR",
    'incident_silence_time': 14,
    'action_item_silence_time': 14,
    'ignore_tags': [],
    'ping_action_items': True,
    'ping_forsaken_incidents': True,
    'check_forsaken_ticket': False,
    'tg_text': (
        "В этом LSR **нет активности** более {silence_time} дней.\n"
        "Закройте LSR или продолжите его разбор.\n"
        "Если у этого тикета нет исполнителя или исполнителем является уволенный сотрудник,"
        " пожалуйста, проставьте или смените исполнителя.\n"
        "Если не знаете, что делать, обратитесь к @mvel.\n"
        "Если вам не нравятся эти сообщения (или вы считаете их неуместными), "
        "пожалуйста, напишите @natapolienko. Спасибо!\n"
    ),
    'forsaken_incidents_extra_filter': '"Last Comment": <=now() -60d',
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'ping_development': True,
    'relevant_people_closed_incident': False,
}

SEARCHAPP_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    # first list for l1,l2 and l3; second for l2 and l3; third for l3
    'extra_summonees': [
        ["natapolienko"],
        ["mvel"],
        ["dmitryno"],
    ],
    'st_product': "PROD-SEARCHAPP",
    'st_queue': "SPI",
    'incident_silence_time': 14,
    'action_item_silence_time': 60,
    'tags_to_check': ['area', 'impact', 'support_line'],
    'fields_to_check': ['SRE Area', 'Impact', 'Support_line'],
    'sre_times_filter': '("Notification Time": empty() OR "Begin Time": empty() OR "End Time": empty())',
    'ignore_tags': [],
    'ping_action_items': True,
    'ping_forsaken_incidents': True,
    'check_forsaken_ticket': True,
    'tg_text': BASE_RESPONSIBLE_TEXT,
    'forsaken_incidents_extra_filter': 'Status: !"Проведён анализ"',
    'skip_weekends': True,
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'day_for_ping_tg': [0],
    'ping_development': True,
    'relevant_people_closed_incident': False,
    'ping_incident_with_empty_downtime': True,
}

VIDEO_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    # first list for l1,l2 and l3; second for l2 and l3; third for l3
    'extra_summonees': [],
    'st_product': "PROD-VIDEO",
    'st_queue': "SPI",
    'incident_silence_time': 7,
    'action_item_silence_time': 60,
    'tags_to_check': ['area', 'impact', 'support_line'],
    'fields_to_check': ['SRE Area', 'Impact', 'Support_line'],
    'sre_times_filter': '("Notification Time": empty() OR "Begin Time": empty() OR "End Time": empty())',
    'ignore_tags': [],
    'ping_action_items': True,
    'ping_forsaken_incidents': True,
    'check_forsaken_ticket': True,
    'tg_text': BASE_VIDEO_TEXT,
    'forsaken_incidents_extra_filter': 'Status: !"Проведён анализ"',
    'skip_weekends': True,
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'day_for_ping_tg': [0],
    'ping_development': True,
    'relevant_people_closed_incident': True,
    'ping_incident_with_empty_downtime': True,
}

ALICE_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    # first list for l1,l2 and l3; second for l2 and l3; third for l3
    'extra_summonees': [
        ["natapolienko"],
        ["mvel"],
        ["dmitryno"],
    ],
    'st_product': "PROD-ALICE",
    'st_queue': "SPI",
    'incident_silence_time': 14,
    'action_item_silence_time': 60,
    'tags_to_check': ['area', 'impact', 'support_line'],
    'fields_to_check': ['SRE Area', 'Impact', 'Support_line'],
    'sre_times_filter': '("Notification Time": empty() OR "Begin Time": empty() OR "End Time": empty())',
    'ignore_tags': [
        "product:PROD-WEB",
        "product:PROD-COMMON",
        "product:PROD-IMAGES",
        "product:PROD-COLLECTIONS",
        "product:PROD-SEARCHAPP",
        "product:PROD-VIDEO",
    ],
    'ping_action_items': True,
    'ping_forsaken_incidents': False,
    'check_forsaken_ticket': False,
    'tg_text': BASE_RESPONSIBLE_TEXT,
    'forsaken_incidents_extra_filter': 'Status: !"Проведён анализ"',
    'skip_weekends': True,
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'ping_development': True,
    'reopen_closed_inc_without_ydt': True,
    'ping_incident_with_empty_downtime': True,
    'change_weight_hwproblems': False,
    'relevant_people_closed_incident': False,
}

MDBSUPPORT_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    # first list for l1,l2 and l3; second for l2 and l3; third for l3
    'extra_summonees': ["amatol", "yurovniki"],
    'st_product': "",
    'st_queue': "MDBSUPPORT",
    'incident_silence_time': 2,
    'action_item_silence_time': 60,
    'tags_to_check': ['area', 'impact', 'support_line'],
    'fields_to_check': ['SRE Area', 'Impact', 'Support_line'],
    'sre_times_filter': '("Notification Time": empty() OR "Begin Time": empty() OR "End Time": empty())',
    'ignore_tags': [],
    'ping_action_items': False,
    'ping_forsaken_incidents': True,
    'check_forsaken_ticket': True,
    'tg_text': BASE_RESPONSIBLE_TEXT,
    'forsaken_incidents_extra_filter': (
        '((Status: "В работе" AND Updated: < now() - 2d) OR '
        '(Status: "Требуется информация" AND (Deadline: < today() OR Deadline: empty()) AND Updated: < now() - 4d))'
    ),
    'skip_weekends': True,
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'day_for_ping_tg': [0],
    'ping_development': True,
    'relevant_people_closed_incident': False,
}

RTCSUPPORT_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    # first list for l1,l2 and l3; second for l2 and l3; third for l3
    'extra_summonees': [],
    'st_product': "",
    'st_queue': "RTCSUPPORT",
    'incident_silence_time': 1,
    'action_item_silence_time': 60,
    'tags_to_check': ['area', 'impact', 'support_line'],
    'fields_to_check': ['SRE Area', 'Impact', 'Support_line'],
    'sre_times_filter': '("Notification Time": empty() OR "Begin Time": empty() OR "End Time": empty())',
    'ignore_tags': [],
    'ping_action_items': False,
    'ping_forsaken_incidents': True,
    'check_forsaken_ticket': True,
    'tg_text': BASE_RTCSUPPORT_TEXT,
    'forsaken_incidents_extra_filter': (
        '((Status: "В работе" AND Updated: < now() - 2d) OR Status: "Требуется информация" OR Status: "Линия 2")'
    ),
    'skip_weekends': True,
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'day_for_ping_tg': [0],
    'ping_development': True,
    'relevant_people_closed_incident': False,
}

MARKETING_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    'extra_summonees': [],
    'ignore_tags': [],
    'st_product': ["PROD-BANNER", "PROD-DIRECT", "PROD-METRIKA", "PROD-PCODE", "PROD-MODADVERT", "PROD-RMP"],
    'st_queue': "SPI",
    'incident_silence_time': 3,
    'ping_forsaken_incidents': True,
    'ping_action_items': False,
    'ping_people': True,
    'ping_development': True,
    'relevant_people_closed_incident': True,
}

BSAUDIT_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    'extra_summonees': [],
    'ignore_tags': [],
    'st_product': '',
    'st_queue': "BSAUDIT",
    'incident_silence_time': 2,
    'ping_forsaken_incidents': True,
    'ping_action_items': False,
    'ping_people': True,
    'ping_development': True,
    'relevant_people_closed_incident': True,
}

DIRECT_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    'extra_summonees': [],
    'ignore_tags': [],
    'st_product': '',
    'st_queue': "DIRECT, UC",
    'forsaken_incidents_extra_filter': (
        'Filter: 53044 Priority: Критичный Stage: Production'
    ),
    'incident_silence_time': 4,
    'ping_forsaken_incidents': True,
    'ping_action_items': False,
    'ping_people': True,
    'ping_development': True,
    'relevant_people_closed_incident': True,
}


BILLING_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    # first list for l1,l2 and l3; second for l2 and l3; third for l3
    'extra_summonees': [],
    'st_product': "PROD-BILLING",
    'st_queue': "SPI",
    'incident_silence_time': 14,
    'action_item_silence_time': 60,
    'tags_to_check': ['area', 'impact', 'support_line'],
    'fields_to_check': ['SRE Area', 'Impact', 'Support_line'],
    'sre_times_filter': '("Notification Time": empty() OR "Begin Time": empty() OR "End Time": empty())',
    'ignore_tags': [],
    'ping_action_items': True,
    'ping_forsaken_incidents': True,
    'check_forsaken_ticket': True,
    'tg_text': BASE_RESPONSIBLE_TEXT,
    'forsaken_incidents_extra_filter': 'Status: !"Проведён анализ"',
    'skip_weekends': True,
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'ping_development': True,
    'reopen_closed_inc_without_ydt': True,
    'ping_incident_with_empty_downtime': True,
    'change_weight_hwproblems': True,
    'relevant_people_closed_incident': False,
}


POINTS_WEIGHTS = {
    'SUPPORT': 1,
    'SPI': {
        'trivial': 1,
        'minor': 2,
        'normal': 3,
        'critical': 5,
        'blocker': 5
    },
    'LSR': 10,
    'default': 1
}

IMPACT_WEIGHTS = {
    'internal': 1,
    'external': 2,
    None: 1
}


def get_config(component):
    return {
        'web': WEB_CONFIG,
        'common': COMMON_CONFIG,
        'alice': ALICE_CONFIG,
        'images': IMAGES_CONFIG,
        'collections': COLLECTIONS_CONFIG,
        'marty': MARTY_CONFIG,
        'rm': RM_CONFIG,
        'lsr': LSR_CONFIG,
        'searchapp': SEARCHAPP_CONFIG,
        'standard': STANDARD_CONFIG,
        'video': VIDEO_CONFIG,
        'mdbsupport': MDBSUPPORT_CONFIG,
        'rtcsupport': RTCSUPPORT_CONFIG,
        'marketing': MARKETING_CONFIG,
        'bsaudit': BSAUDIT_CONFIG,
        'direct': DIRECT_CONFIG,
        'billing': BILLING_CONFIG
    }[component]


@rm_notify.notify(["rm_maintainers"], message=rm_notify.StandardMessage.some_trouble)
class DevOpsAlert(sdk2.Task):
    """
    Various notification for devops vanguard.

    * List of incidents without "Sotrudniki->Dezhurniy" field set
    * List of incidents that has no actions for too long
    * To assign next guardians of the realm
    """

    class Requirements(task_env.TinyRequirements):
        disk_space = 1000  # 1G
        ram = 4096  # 4G
        environments = [
        #    environments.PipEnvironment("yandex_tracker_client"),
            rm_task_env.TaskRequirements.startrek_client
        ]
        client_tags = rm_task_env.TaskTags.startrek_client

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.RadioGroup('Component') as component:
            component.values['web'] = component.Value(value='web', default=True)
            component.values['common'] = component.Value(value='common')
            component.values['images'] = component.Value(value='images')
            component.values['collections'] = component.Value(value='collections')
            component.values['marty'] = component.Value(value='marty')
            component.values['rm'] = component.Value(value='rm')
            component.values['lsr'] = component.Value(value='lsr')
            component.values['searchapp'] = component.Value(value='searchapp')
            component.values['standard'] = component.Value(value='standard')
            component.values['video'] = component.Value(value='video')
            component.values['alice'] = component.Value(value='alice')
            component.values['mdbsupport'] = component.Value(value='mdbsupport')
            component.values['rtcsupport'] = component.Value(value='rtcsupport')
            component.values['marketing'] = component.Value(value='marketing')
            component.values['bsaudit'] = component.Value(value='bsaudit')
            component.values['direct'] = component.Value(value='direct')

        override_receiver = sdk2.parameters.Bool(
            "Send telegram messages to other person, instead of devops chat (for debugging)",
            default=False,
        )
        with override_receiver.value[True]:
            receiver = sdk2.parameters.String(
                "login of receiver",
                required=True,
                default='lebedev-aa',
            )
        dry_run = sdk2.parameters.Bool(
            "Do not post comments in startrek (for debugging)",
            default=False,
        )
        sleep_in_the_beginning = sdk2.parameters.Bool(
            "Sleep for one minute in the beginning (for debugging)",
            default=False,
        )

    class Context(sdk2.Task.Context):
        forsaken_spi = []
        forsaken_action_items = []
        details_issues = []

    def on_execute(self):
        self.login = None
        self.start_date = 'today() - 6M'
        self.forbidden_issues = set()

        if self.Parameters.sleep_in_the_beginning:
            time.sleep(60)
        self.config = get_config(self.Parameters.component)

        # Startrek
        if self.config.get('st'):
            if 'secret_id' in self.config['st'] and 'key' in self.config['st']:
                if 'version' in self.config['st']:
                    secret = sdk2.yav.Secret(self.config['st']['secret_id'], self.config['st']['version'])
                else:
                    secret = sdk2.yav.Secret(self.config['st']['secret_id'])
                self.st_token = secret.data()[self.config['st']['key']]
            elif 'token_owner' in self.config['st'] and 'token_name' in self.config['st']:
                self.st_token = sdk2.Vault.data(self.config['st']['token_owner'], self.config['st']['token_name'])
            else:
                raise ValueError('Invalid st config')
            self.st_helper = STHelper(self.st_token, 'DevOpsAlert Bot')
            self.login = self.st_helper.st_client.myself.login
            if self.config.get('ping_forsaken_incidents'):
                self.ping_forsaken_incidents()
            if self.config.get('ping_action_items'):
                self.ping_action_items()
            if self.config.get('reopen_closed_inc_without_ydt'):
                self.reopen_ping_closed_inc_without_ydt()
            if self.config.get('ping_incident_with_empty_downtime'):
                self.ping_incident_with_empty_downtime()
            if self.config.get('change_weight_hwproblems'):
                self.change_weight_hwproblems()
            if self.config.get('relevant_people_closed_incident'):
                self.reopen_closed_inc_allowed_to_close_spi_logins()

            pinged_issues = self.Context.forsaken_spi + self.Context.forsaken_action_items + self.Context.details_issues
            self.set_info(
                "Pinged issues: {}".format(
                    ' '.join(
                        '<a href="{0}" target="_blank">{1}</a>'.format(
                            ST.issue_link(issue_key),
                            issue_key,
                        ) for issue_key in pinged_issues
                    )
                ),
                do_escape=False,
            )
        logging.info('Calculate YDT for SPPROBLEMs and LSRs')
        # SPPROBLEM calc,  PROD-WEB for now
        if self.config.get('st_product') == 'PROD-WEB':
            self.ydt_weight_importance_update('SPPROBLEM')
        if self.config.get('st_queue') == 'LSR':
            self.ydt_weight_importance_update('LSR')

    def get_head(self, whom):
        headers = {'Authorization': 'OAuth {}'.format(self.st_token)}
        response = requests.get(
            'https://staff-api.yandex-team.ru/v3/persons',
            headers=headers,
            params={
                'login': whom,
                '_fields': 'chief.login'
            }
        ).json()
        return response['result'][0]['chief']['login']

    def get_head_of_department(self, id):
        headers = {'Authorization': 'OAuth {}'.format(self.st_token)}
        response = requests.get(
            'https://staff-api.yandex-team.ru/v3/groups',
            headers=headers,
            params={
                'id': str(id),
                '_fields': 'department.heads.person.login'
            }
        ).json()
        return response['result'][0]['department']['heads'][0]['person']['login']

    def check_forsaken_ticket(self):
        """
        Find incidents which are not resolved and not updating) for a long time.

        :return message informing about forsaken incidents
        or `None` if no forsaken issues detected.
        """
        silence_time = self.config["incident_silence_time"]
        query, short_url = ST.forsaken_incidents_queries(
            silence_time,
            self.config['st_product'],
            self.config['st_queue'],
            self.config['ignore_tags'],
            self.config.get('forsaken_incidents_extra_filter', '')
        )
        issues = ST.find_tickets_by_query(self.st_helper, query)
        message = None
        if not issues:
            logging.info("There are no forsaken issues, we are awesome!")
            return None
        elif len(issues) > 4:
            message = (
                'У нас {} инцидентов, которые не обновлялись больше {} дней. '
                'Вот первые три: \n{}. <a href="{}">Полный список</a>'
            ).format(
                len(issues),
                silence_time,
                ', '.join([ST.issue_link(found_issue.key) for found_issue in issues[:3]]),
                short_url,
            )
        else:
            message = "Инциденты, которые не обновлялись больше {} дней: \n{}.".format(
                silence_time,
                ', '.join([ST.issue_link(found_issue.key) for found_issue in issues])
            )
        return message

    def check_the_text_for_similarity(self, text, all_comments):
        """Check the last message is from the robot and the text of this message matches the ping text."""
        part_template_comment_text = {
            'text_common': 'В этом инциденте **нет активности** более',
            'text_exists_spproblem': 'Данный тикет является частным случаем проблемы',
            'text_maybe_exists_spproblem': 'Предполагается, что данный тикет является частным случаем проблемы',
            'text_exists_actionitems': 'Для данного инцидента выделено',
            'text_without_resolution': 'В этом инциденте не проставлена резолюция.',
            'text_without_udt_vdt_umbrella': 'Не забудьте расчитать и заполнить поля',
            'text_lsr': 'В этом LSR **нет активности** более',
            'text_actionitems': 'Данная задача поможет избежать',
            'text_without_sre': 'В этом инциденте не выставлены SRE времена.',
            'text_video': 'Привет всем! Прошло много дней после инцидента. Пожалуйста:',
            'text_rtcsupport': 'В этой задаче не было никаких изменений уже',
        }

        if all_comments and all_comments[-1].updatedBy.login == self.login:
            concatenation_text = text + all_comments[-1].text
            if any([
                concatenation_text.count(template_text) == 2
                for template_text in part_template_comment_text.values()
            ]):
                return True
        return False

    def add_followers(self, issue, follower):
        """Add followers to issue."""
        import startrek_client

        logging.info('followers %s', issue.followers)
        for link_follower in issue.followers:
            logging.info(link_follower)
            if follower in link_follower.id:
                logging.info('return %s', link_follower.id)
                return

        if self._forbidden_issue(issue.key, 'Update tags'):
            return

        try:
            self.st_helper.st_client.issues[issue.key].update(
                followers={'add': [follower]},
                params={'notify': False},
            )
        except startrek_client.exceptions.Forbidden as e:
            self._add_forbidden_issue(issue.key, 'Update tags')
        except Exception as e:
            eh.log_exception("Failed to update tags on {}".format(issue.key), e)

    def set_attribute_minusDc(self, issue):
        if self._forbidden_issue(issue.key, 'Set tags'):
            return

        drills_links = [
            linkobject for linkobject in issue.links
            if "DRILLS" in linkobject.object.key
        ]
        if not drills_links:
            return
        try:
            self.st_helper.st_client.issues[issue.key].update(
                minusDc="Да",
                params={'notify': False},
            )
        except startrek_client.exceptions.Forbidden as e:
            self._add_forbidden_issue(issue.key, 'Set tags')
        except Exception as e:
            eh.log_exception("Failed to update tags on {}".format(issue.key), e)

    def set_tag(self, issue, tag):
        """Set tag to issue without notification."""
        import startrek_client

        if self.Parameters.dry_run:
            return

        if tag in issue.tags:
            return

        if 'victim' in tag:
            self.add_followers(issue, 'rmmagomedov')

        if self._forbidden_issue(issue.key, 'Set tags'):
            return

        try:
            self.st_helper.st_client.issues[issue.key].update(
                tags={'add': [tag]},
                params={'notify': False},
            )
        except startrek_client.exceptions.Forbidden as e:
            self._add_forbidden_issue(issue.key, 'Set tags')
        except Exception as e:
            eh.log_exception("Failed to update tags on {}".format(issue.key), e)

    def change_link(self, issue, victims, type_link):
        """Change link to Depend"""
        import startrek_client
        if self.Parameters.dry_run:
            return

        for victim in victims:
            for linkobject in issue.links:
                logging.info('%s, %s', linkobject.object.key, victim)
                if linkobject.object.key != victim:
                    continue
                # victim found

                if self._forbidden_issue(issue.key, 'Update link'):
                    continue

                try:
                    self.st_helper.st_client.issues[linkobject.object.key].update(
                        links={"relationship": type_link, "issue": issue.key},
                        params={'notify': False},
                    )
                except startrek_client.exceptions.Forbidden as e:
                    self._add_forbidden_issue(issue.key, 'Update link')
                except Exception as e:
                    eh.log_exception("Failed to update link on {}".format(issue.key), e)

    def get_victim(self, issue):
        logging.info("Try to find victim for %s", issue.key)
        items = []
        if issue.description:
            items_place = issue.description.split('=== Пострадавшие ===')
            if len(items_place) > 1:
                items_place = items_place[-1].split('=== Методички ===')[0]
                items = re.findall("[A-Z]+-[0-9]+", items_place)
        logging.info('victims: %s', items)
        return items

    def collect_related_tickets(self, issues, relate, direction, tag_main, tag_link, type_ticket):
        main_issue_with_link = {}
        for issue in issues:
            if not issue.links:
                continue
            depend_links_all = [
                linkobject for linkobject in issue.links
                if linkobject.type.id == relate and linkobject.direction == direction
            ]
            if type_ticket:
                depend_links = [
                    linkobject for linkobject in depend_links_all
                    if linkobject.object.key.startswith(type_ticket)
                ]
            else:
                depend_links = depend_links_all
            if depend_links:
                main_issue_with_link[issue] = []
                if tag_main:
                    self.set_tag(issue, tag_main)
            for link in depend_links:
                if link.object.key.startswith('LSR-') or link.object.key.startswith('SPPROBLEM-'):
                    logging.info('LSR or SPPROBLEM in depend %s', link.object.key)
                    self.change_link(issue, [link.object.key], "is dependent by")
                    continue
                try:
                    item_obj = self.st_helper.st_client.issues[link.object.key]
                except Exception as e:
                    eh.log_exception("Failed to get action item from ST. Probably too simple regexp.", e, task=self)
                    continue
                if tag_link:
                    self.set_tag(item_obj, tag_link)
                main_issue_with_link[issue].append(link)
        return main_issue_with_link

    def update_sum_ydt(self, issue, links):
        """Calculate ydt for all related SPI(links) and set counted ydt to issue."""
        if self.Parameters.dry_run:
            return
        sum_ydt = 0
        sp_update_dict = {
            'params': {'notify': False}
        }
        for link in links:
            item_obj = self.st_helper.st_client.issues[link.object.key]
            if item_obj.ydt:
                sum_ydt += item_obj.ydt
        logging.info('summ_ydt = %s', sum_ydt)
        if sum_ydt > 0:
            sp_update_dict['ydt'] = round(sum_ydt, 3)
        sp_issue = self.st_helper.st_client.issues[issue.key]
        try:
            sp_issue.update(**sp_update_dict)
        except Exception as e:
            eh.log_exception('Cannot update issue {}'.format(sp_issue.key), e)

    def ping_incident_with_empty_downtime(self):
        query = ST.Query(
            product=self.config['st_product'],
            not_closed=False,
            queue=self.config['st_queue'],
            tags_to_ignore=self.config['ignore_tags'],
            created=self.start_date
        ).build_query()
        issues = ST.find_tickets_by_query(self.st_helper, query)

        # for debug
        # issues = [self.st_helper.st_client.issues['SPI-14522'], self.st_helper.st_client.issues['SPI-14475']]
        for issue in issues:
            add_victims = self.get_victim(issue)
            logging.info('%s: %s', issue.key, add_victims)
            self.change_link(issue, add_victims, "depends on")
        text = ("Не забудьте расчитать и заполнить поля 'Даунтайм Яндекса' и 'Даунтайм Вертикали'.\n")
        main_issue_with_link = self.collect_related_tickets(
            issues, 'depends', 'inward', 'spi:umbrella', 'spi:victim', ""
        )
        logging.info('main_issue_with_link %s', main_issue_with_link)
        for issue, links in main_issue_with_link.items():
            self.update_sum_ydt(issue, links)
            for link in links:
                if link.object.key.startswith("SPI-"):
                    logging.info('SPI %s', link)
                    item_obj = self.st_helper.st_client.issues[link.object.key]
                    time_last_ping_robot = ST.time_since_last_bot_message(item_obj, self.st_token)
                    if item_obj.ydt is None or item_obj.vdt is None:
                        count_ping_comments_robot = ST.count_ping_comments_before(
                            item_obj, self.login, text
                        )
                        if time_last_ping_robot >= 7 or (time_last_ping_robot == 0 and count_ping_comments_robot == 0):
                            people = (
                                ST.get_assignee(item_obj, self.st_token)
                                or ST.get_issue_devops(item_obj, self.st_token)
                            )
                            logging.info(
                                'ping people incidents %s %s %s %s',
                                people, text, time_last_ping_robot, count_ping_comments_robot
                            )
                            all_comments = list(item_obj.comments.get_all())
                            if self.check_the_text_for_similarity(text, all_comments):
                                ST.update_comment(self, item_obj, all_comments[-1].id, text, people)
                            else:
                                ST.ping(
                                    self,
                                    item_obj,
                                    text,
                                    people
                                )

    def reopen_ping_closed_inc_without_ydt(self):
        if self.Parameters.dry_run:
            return
        if not self.config['st_queue'] == 'SPI':
            return
        query = ST.Query(
            product=self.config['st_product'],
            not_closed=False,
            queue=self.config['st_queue'],
            tags_to_ignore=self.config['ignore_tags'],
            created=self.start_date
        ).build_query()
        issues = ST.find_tickets_by_query(self.st_helper, query)
        spi_not_ydt = []
        for issue in issues:
            logging.info("ydt %s %s", issue.ydt, issue.priority.key)
            if ST.is_issue_closed(issue) and issue.impact == 'external' and issue.ydt is None:
                if issue.resolution and (issue.resolution.key == 'duplicate' or issue.resolution.key == "can'tReproduce"):
                    logging.info(issue.resolution.key)
                    continue
                # DEVOPS-617 incident with component == 'net' reopen if priority > normal
                if (
                    any([component.name == 'net' for component in issue.components])
                    and (issue.priority.key == 'trivial' or issue.priority.key == 'normal')
                ):
                    continue
                spi_not_ydt.append(issue)
        text = ("В этом тикете проставлен impact=external, но не указан YDT.\n"
                "Пожалуйста, задайте YDT, если пострадали внешние пользователи, "
                "или измените impact на internal.\n"
                "Если значение YDT очень мало, проставьте 0.\n"
                "Если посчитать не представляется возможным, проставьте его равным 0.066\n")
        for issue in spi_not_ydt[:3]:
            if issue.transitions.get_all():
                logging.info("Transitions %s", issue.transitions.get_all()[0].id)
                issue.transitions[issue.transitions.get_all()[0].id].execute()
                logging.info("Reopen spi %s new status %s", issue.key, issue.status.key)
                people = ST.get_assignee(issue, self.st_token) or ST.get_issue_devops(issue, self.st_token)
                logging.info('ping people incidents %s', people)
                ST.ping(
                    self,
                    issue,
                    text,
                    people
                )

    def who_closed(self, issue):
        change_status = list(issue.changelog.get_all(field='status'))
        if change_status and issue.status.key == 'closed':
            logging.info('change status %s, %s', change_status, change_status[-1].updatedBy)
            return change_status[-1].updatedBy.id

    def reopen_closed_inc_allowed_to_close_spi_logins(self):
        if self.Parameters.dry_run:
            return
        if not self.config['st_queue'] == 'SPI':
            return
        logging.info('Started reopening closed incidents')
        text = (
            "Переоткрыли тикет и призвали вас, потому что инцидент был закрыт человеком не из списка допущенных.\n"
        )
        text_many = "Данный список можно найти на страницах"
        text_one = "Данный список можно найти на странице"
        end_text_many = ' соответствующих каждой компоненте, в поле allowedToCloseSpiLogins'
        end_text_one = ' соответствующей компоненте, в поле allowedToCloseSpiLogins'
        warden_component = ' https://warden.z.yandex-team.ru/components/{},'
        issues = []
        if type(self.config["st_product"]) == list:
            for product in self.config['st_product']:
                query = ST.Query(
                    product=product,
                    created=self.start_date,
                    not_closed=False,
                ).build_query()
                issues.extend(ST.find_tickets_by_query(self.st_helper, query))
        else:
            query = ST.Query(
                product=self.config['st_product'],
                not_closed=False,
                queue=self.config['st_queue'],
                tags_to_ignore=self.config['ignore_tags'],
                created=self.start_date,
            ).build_query()
            issues = ST.find_tickets_by_query(self.st_helper, query)
        # issues = [self.st_helper.st_client.issues['SPI-14922']]
        for issue in issues:
            if issue.resolution and issue.resolution.key == 'duplicate':
                logging.info(issue.resolution.key)
                continue
            people_responsible = self.allowed_to_close_spi_logins(issue.components)
            logging.info('People allowed for %s: %s', issue.key, people_responsible)
            if not people_responsible:
                continue
            who_closed = self.who_closed(issue)
            logging.info('Who closed %s: %s', issue.key, who_closed)
            if who_closed and who_closed not in people_responsible:
                logging.info('Ping SPI: %s', issue.key)
                all_text = text
                if len(issue.components) == 1:
                    all_text += text_one
                    all_text += warden_component.format(issue.components[0].name)
                    all_text += end_text_one
                    if issue.transitions.get_all():
                        logging.info("Transitions %s", issue.transitions.get_all()[0].id)
                        issue.transitions[issue.transitions.get_all()[0].id].execute()
                        logging.info("Reopen incident %s, new status %s", issue.key, issue.status.key)
                else:
                    all_text += text_many
                    for component in issue.components:
                        all_text += warden_component.format(component.name)
                    all_text += end_text_many
                ST.ping(
                    self,
                    issue,
                    all_text,
                    people_responsible
                )

    def find_owners(self, name):
        logging.info('component %s', name)
        data = requests_wrapper.post(
            'https://warden.z.yandex-team.ru/api/warden.Warden/getComponent',
            json={'name': name, 'parent_component_name': ''},
        )
        if data.status_code >= 400:
            resp = {}
        else:
            resp = data.json()
        logging.info('resp %s', resp)
        if resp:
            if resp.get('component'):
                return resp['component']
        logging.info('find owners null for %s', name)
        return []

    def allowed_to_close_spi_logins(self, components):
        people = []
        for component in components:
            owners_data = self.find_owners(component.name)
            logging.info('Owners data for component `%s`: %s', component.name, owners_data)
            if owners_data:
                logging.info('Owners data %s', owners_data)
                if owners_data.get('protocolSettings'):
                    if owners_data['protocolSettings'].get('allowedToCloseSpiLogins'):
                        people.extend(owners_data['protocolSettings']['allowedToCloseSpiLogins']['logins'])
                logging.info('name component: %s responsible %s', component.name, people)
        return people

    def responsible_vertical(self, components):
        people = []
        for component in components:
            data_for_owners = self.find_owners(component.name)
            if data_for_owners and data_for_owners.get('owners'):
                people.extend(data_for_owners['owners']['logins'])
            logging.info('name component: %s responsible %s', component.name, people)
        return people

    def change_weight_hwproblems(self):
        if self.Parameters.dry_run:
            return
        query = ST.Query(
            not_closed=False,
            queue='HWPROBLEMS',
        ).build_query()
        issues = ST.find_tickets_by_query(self.st_helper, query)

        query = ST.Query(
            not_closed=False,
            queue='DEPLOY',
        ).build_query()
        issues_deploy = ST.find_tickets_by_query(self.st_helper, query)
        # issues = [self.st_helper.st_client.issues['HWPROBLEMS-50']]
        logging.info('HWPROBLEMS: %s', issues)
        logging.info('DEPLOY: %s', issues_deploy)
        issues.extend(issues_deploy)
        for issue in issues:
            linkobjects = [
                linkobject for linkobject in issue.links
                if (linkobject.object.key.startswith("SPI-") or linkobject.object.key.startswith("RTCSUPPORT-"))
            ]
            logging.info('len %s links %s', len(linkobjects), linkobjects)
            sp = {
                'params': {'notify': False},
                'weight': len(linkobjects),
            }
            issue.update(**sp)

    def get_author(self, issue):
        try:
            author = issue.createdBy.id
            if not ST.get_person_is_dismissed(issue.createdBy.id, self.st_token):
                logging.debug("get_issue_devops returns: %s", author)
                return [author]
            else:
                return []
        except Exception as e:
            eh.log_exception("failed to get issue devops", e)
            return []

    def check_range_from_chief(self, people, chief):
        head_people = ST.get_head_for_person(people, self.st_token)
        if not head_people or head_people[0] == chief:
            return False
        return True

    def ping_marketing_spis(self):
        issues = []
        # issues = [self.st_helper.st_client.issues['SPI-26763'], self.st_helper.st_client.issues['SPI-27793']]
        for product in self.config['st_product']:
            query = ST.Query(
                product=product,
                created=self.start_date
            ).build_query()
            issues.extend(ST.find_tickets_by_query(self.st_helper, query))
        if not issues:
            logging.info("There are no issues not closed for this time")
        text = (
            "В инциденте не заполнены следующие обязательные поля:\n"
            "{list_place}\n"
            "Пожалуйста, заполните данные поля до {day_before_next_date_ping}!\n"
            "В противном случае {next_date_ping} в задачу будут призваны: {list_people}!"
        )
        text_crit = (
            "В инциденте не заполнены следующие обязательные поля:\n"
            "{list_place}\n"
            "Пожалуйста, заполните данные поля, необходимые для разбора инцидента!"
        )
        logging.info(issues)
        for issue in issues:
            try:
                self._ping_marketing_spi(issue, text, text_crit)
            except Exception as _e:
                logging.exception('Failed to ping %s: %s', issue.key, _e)

    def _ping_marketing_spi(self, issue, text, text_crit):
        tags = issue.tags

        if "banner:noncrit" in tags:
            logging.info("%s has tag banner:noncrit because not ping", issue.key)
            return

        list_place = []
        head_assignee = []
        head_head = []
        author = self.get_author(issue)
        assignee = ST.get_assignee(issue, self.st_token)
        if assignee and not ST.check_department_for_person(assignee[0], "yandex_dep79124", self.st_token):
            logging.info("%s has assignee without need department - not ping", issue.key)
            return
        if assignee and self.check_range_from_chief(assignee[0], "ironpeter"):
            head_assignee = ST.get_head_for_person(assignee[0], self.st_token)
        if head_assignee and self.check_range_from_chief(head_assignee[0], "ironpeter"):
            head_head = ST.get_head_for_person(head_assignee[0], self.st_token)
            if not self.check_range_from_chief(head_head[0], "ironpeter"):
                head_head = []
        logging.info('author: "%s", assignee: "%s", head_assignee: "%s", head_head: "%s"', author, assignee, head_assignee, head_head)
        description = issue.description

        # что видели пользователи
        intime_description = description.split(u"=== Что во время инцидента видели пользователи")
        if len(intime_description) > 1:
            intime_description = intime_description[1]
            intime_description = intime_description.split(u"=== Описание")[0]
            if "===" in intime_description:
                intime_description = intime_description.replace("===", "")
            if "&nbsp;" in intime_description:
                intime_description = intime_description.replace("&nbsp;", "")
            if u"Как проявилось у пользователей и сколько их пострадало" in intime_description:
                intime_description = intime_description.replace(u"Как проявилось у пользователей и сколько их пострадало", "")
                intime_description = intime_description.replace("++", "").replace("//", "").replace("!!", "").replace("(grey)", "")
            intime_description = intime_description.strip()
            if intime_description == "":
                list_place.append(u"Что во время инцидента видели пользователи")
        else:
            list_place.append(u"Что во время инцидента видели пользователи")
        # logging.info("intime {}".format(intime_description))

        # описание
        descr_description = description.split(u"=== Описание")
        if len(descr_description) > 1:
            descr_description = descr_description[1]
            descr_description = descr_description.split(u"=== Действия дежурного")[0]
            if "===" in descr_description:
                descr_description = descr_description.replace("===", "")
            if "&nbsp;" in descr_description:
                descr_description = descr_description.replace("&nbsp;", "")
            if u"Здесь нужно кратко и понятно" in descr_description:
                descr_description = descr_description.replace(u"Здесь нужно кратко и понятно написать что произошло.", "")
                descr_description = descr_description.replace(u"Здесь нужно кратко и понятно написать что же произошло.", "")
                descr_description = descr_description.replace(u"И не забыть указать причину, по которой случилось то, что случилось", "")
                descr_description = descr_description.replace("++", "").replace("//", "").replace("!!", "").replace("(grey)", "")
            descr_description = descr_description.strip()
            if descr_description == "":
                list_place.append(u"Описание")
        else:
            list_place.append(u"Описание")
        # logging.info("descr {}".format(descr_description))

        # хронология событий
        chron_description = description.split(u"=== Хронология событий")
        if len(chron_description) > 1:
            chron_description = chron_description[1]
            chron_description = chron_description.split(u"=== Анализ")[0]
            if "===" in chron_description:
                chron_description = chron_description.replace("===", "")
            if "&nbsp;" in chron_description:
                chron_description = chron_description.replace("&nbsp;", "")
            if u"Что именно вызвало, как обнаружили, основные этапы починки, как и когда уведомили смежников и" in chron_description:
                chron_description = chron_description.replace(u"Что именно вызвало, как обнаружили, основные этапы починки,", "")
                chron_description = chron_description.replace(u"как и когда уведомили смежников и анонсировали во внутренних", "")
                chron_description = chron_description.replace(u"координационных чатах, когда пользователи перестали", "")
                chron_description = chron_description.replace(u"замечать проявление проблемы / починка полностью закончилась", "")
                chron_description = chron_description.replace("++", "").replace("//", "").replace("!!", "").replace("(grey)", "")
            chron_description = chron_description.strip()
            if chron_description == "":
                list_place.append(u"Хронология событий")
        else:
            list_place.append(u"Хронология событий")
        # logging.info("chron {}".format(chron_description))

        # потери
        lost_description = description.split(u"=== Потери")
        if len(lost_description) > 1:
            lost_description = lost_description[1]
            lost_description = lost_description.split(u"=== Меры предотвращения")[0]
            if "===" in lost_description:
                lost_description = lost_description.replace("===", "")
            if "&nbsp;" in lost_description:
                lost_description = lost_description.replace("&nbsp;", "")
            if u"В данном разделе помимо общего описания потерь должен присутствовать расчет метрики YDT" in lost_description:
                lost_description = lost_description.replace(u"В данном разделе помимо общего описания потерь должен", "")
                lost_description = lost_description.replace(u"присутствовать расчет метрики YDT, если она рассчитывалась не автоматически.", "")
                lost_description = lost_description.replace(u"Рассчитанный YDT/VDT необходимо записать в поля тикета.", "")
                lost_description = lost_description.replace("++", "").replace("//", "").replace("!!", "").replace("(grey)", "")
            lost_description = lost_description.strip()
            if lost_description == "":
                list_place.append(u"Потери")
        else:
            list_place.append(u"Потери")
        # logging.info(u"lost {}".format(lost_description))

        logging.info(u"list place {}".format(", ".join(list_place)))
        if not list_place:
            logging.info("All required fields are filled in")
            return
        createdAt = datetime.datetime.strptime(issue.createdAt[:-9], "%Y-%m-%dT%H:%M:%S")
        delta_days = (datetime.datetime.utcnow() - createdAt).days
        logging.info(u"createdAt {} delta_days {}".format(createdAt, delta_days))
        end_date = createdAt.date() + datetime.timedelta(days=14)
        people = []
        next_people = ""
        next_date = ""
        res_text = ""
        if delta_days == 15:
            people = ["veral"]
            people.extend(author)
            people.extend(assignee)
            people.extend(head_assignee)
            people.extend(head_head)
            if "fedotovkir" not in people:
                people.append("fedotovkir")
            if "sankear" not in people:
                people.append("sankear")
            res_text = text_crit.format(list_place=", ".join(list_place))
        if delta_days == 10:
            people = []
            people.extend(author)
            people.extend(assignee)
            people.extend(head_assignee)
            people.extend(head_head)
            if "fedotovkir" not in people:
                people.append("fedotovkir")
            if "sankear" not in people:
                people.append("sankear")
            next_date = createdAt.date() + datetime.timedelta(days=15)
            next_people = 'staff:' + ", staff:".join(people) + ", staff:veral"
        if delta_days == 7:
            people = []
            people.extend(author)
            people.extend(assignee)
            people.extend(head_assignee)
            people.extend(head_head)
            if "fedotovkir" not in people:
                people.append("fedotovkir")
            next_date = createdAt.date() + datetime.timedelta(days=10)
            next_people = 'staff:' + ', staff:'.join(people)
            if "sankear" not in people:
                next_people += ', staff:sankear'
        if delta_days == 5:
            people.extend(author)
            people.extend(assignee)
            people.extend(head_assignee)
            next_date = createdAt.date() + datetime.timedelta(days=7)
            next_people = 'staff:' + ', staff:'.join(people)
            if head_head:
                next_people += ', staff:' + head_head[0]
            next_people += ", staff:fedotovkir"
        if delta_days == 3:
            people.extend(author)
            people.extend(assignee)
            next_date = createdAt.date() + datetime.timedelta(days=5)
            next_people = 'staff:' + ', staff:'.join(people)
            if head_assignee:
                next_people += ', staff:' + head_assignee[0]
        logging.info("people {}".format(people))
        logging.info("next_people {}".format(next_people))
        if people:
            if not res_text:
                day_before_next_date_ping = next_date - datetime.timedelta(days=1)
                res_text = text.format(
                    list_place=", ".join(list_place),
                    date=end_date,
                    next_date_ping=next_date,
                    list_people=next_people,
                    day_before_next_date_ping=day_before_next_date_ping
                )
            logging.info(u"text {}".format(res_text))
            if self.Parameters.dry_run:
                return
            ST.ping(
                self,
                issue,
                res_text,
                people
            )

    def get_qaEngineer(self, issue):
        try:
            qaEngineer = issue.qaEngineer.id
            if not ST.get_person_is_dismissed(qaEngineer, self.st_token):
                logging.debug("get_issue_devops returns: %s", qaEngineer)
                return [qaEngineer]
            else:
                return []
        except Exception as e:
            eh.log_exception("failed to get issue devops", e)
            return []

    def get_dutywork_from_abc(self, service, duty_slug):
        url = 'https://abc-back.yandex-team.ru/api/v4/duty/on_duty/?service={}'.format(service)
        headers = {'Authorization': 'OAuth {}'.format(self.st_token)}
        data = requests_wrapper.get(url, headers=headers)
        all_duty = data.json()
        for duty in all_duty:
            if duty['schedule']['slug'] == duty_slug:
                return [duty['person']['login']]
        return []

    def direct_ping(self):
        query = ST.Query(
            queue=self.config['st_queue'],
            created=self.start_date,
            extra_filter=self.config.get('forsaken_incidents_extra_filter', ''),
            not_closed=False
        ).build_query()
        issues = ST.find_tickets_by_query(self.st_helper, query)
        if not issues:
            logging.info("There are no issues not closed for this time")
        logging.info(issues)
        # for debug
        # issues = [self.st_helper.st_client.issues['UC-2041']]
        # logging.debug(issues)
        issues = [issue for issue in issues if issue.status.key != 'closed']
        logging.info('issues not closed')
        logging.info(issues)
        for issue in issues:
            logging.info(u'issue {}'.format(issue.key))
            head_assignee = []
            head_head = []
            author = self.get_author(issue)
            assignee = ST.get_assignee(issue, self.st_token)
            if assignee:
                head_assignee = ST.get_head_for_person(assignee[0], self.st_token)
            qaEngineer = self.get_qaEngineer(issue)
            logging.info(author)
            logging.info(assignee)
            logging.info(head_assignee)
            status = issue.status.key
            logging.info(status)

            begin_work_time = datetime.time(8, 0, 0)
            end_work_time = datetime.time(17, 0, 0)

            createdAt = datetime.datetime.strptime(issue.createdAt[:-9], "%Y-%m-%dT%H:%M:%S")
            logging.info(u"createdAt {}".format(createdAt))
            created_date = createdAt.date()

            now = datetime.datetime.utcnow()
            logging.info(u"now {}".format(now))
            now_date = now.date()

            sla_hour_res = 0
            if now_date == created_date:
                begin_work_time = datetime.datetime.combine(now_date, begin_work_time)
                end_work_time = datetime.datetime.combine(now_date, end_work_time)
                sla_hour = (now - createdAt).total_seconds() // 3600
                if sla_hour > 8:
                    sla_hour_res = 8
                if createdAt < begin_work_time and now < end_work_time:
                    sla_hour_res = (now - begin_work_time).total_seconds() // 3600
                if createdAt > begin_work_time and createdAt < end_work_time:
                    if now > end_work_time:
                        sla_hour_res = (end_work_time - createdAt).total_seconds() // 3600
                    if now < end_work_time:
                        sla_hour_res = sla_hour
            else:
                begin_work_time_create = datetime.datetime.combine(created_date, begin_work_time)
                end_work_time_create = datetime.datetime.combine(created_date, end_work_time)

                begin_work_time_now = datetime.datetime.combine(now_date, begin_work_time)
                end_work_time_now = datetime.datetime.combine(now_date, end_work_time)

                day_between = (now_date - created_date).days
                sla_hour_res = (day_between - 1)*8

                if createdAt < begin_work_time_create:
                    sla_hour_res += 8
                if createdAt > begin_work_time_create and createdAt < end_work_time_create:
                    sla_hour_res += (end_work_time_create - createdAt).total_seconds() // 3600

                if now > begin_work_time_now and now < end_work_time_now:
                    sla_hour_res += (now - begin_work_time_now).total_seconds() // 3600
                if now > end_work_time_now:
                    sla_hour_res += 8

            logging.info(u"sla_hour_res {}".format(sla_hour_res))

            people = []
            text = ''
            if sla_hour_res > 16:
                text = 'SLA истек'
                people = assignee
                people.extend(head_assignee)
                if 'alkaline' not in people:
                    people.extend(['alkaline'])
            else:
                if sla_hour_res > 8 and (not assignee or (status == 'Open' or status == 'needInfo')):
                    text = 'Ожидалось что задача будет в работе. Обратите, пожалуйста, внимание.'
                    people = author
                    people.extend(assignee)
                    people.extend(head_assignee)
                    if 'alkaline' not in people:
                        people.extend(['alkaline'])
                elif status == 'open':
                    if not assignee:
                        text = (
                            'SLA на исправление критичной проблемы в продакшне - 16 рабочих часов.\n'
                            'Пожалуйста, найди задаче исполнителя в нужном зонтике.\n'
                            'Смотри https://wiki.yandex-team.ru/direct/development/zbp/teams/#otvetstvennyepozontam'
                        )
                        people = author
                    else:
                        text = (
                            'Обрати, пожалуйста, внимание на таймер SLA. До его истечения исправление ошибки должно оказаться в продакшне.'
                        )
                        people = assignee
                if status == 'inProgress' and assignee and sla_hour_res > 8:
                    text = (
                        'Осталось 8 рабочих часов до истечения таймера SLA. '
                        'Убедись, что успеешь исправить проблему и довезти исправление до продакшна. Если сомневаешься - зови на помощь руководителя.'
                    )
                    people = assignee
                    people.extend(head_assignee)
                if status == 'readyForTest' and not qaEngineer:
                    text = 'Пожалуйста, найдите задаче QA-инженера.'
                    people = self.get_dutywork_from_abc('169', 'direct-directsup-duty')
                    if 'sudar' not in people:
                        people.extend(['sudar'])
                    if 'sonick' not in people:
                        people.extend(['sonick'])
                if status == 'betaTested':
                    text = (
                        'Убедись что правка успеет выехать в продакшн. '
                        'Скорее всего, тебе понадобится сделать хотфикс в тестируемый релиз или в продакшн.'
                    )
                    people = assignee
            all_comments = list(issue.comments.get_all())
            already_ping = False
            for comment in all_comments:
                if text == comment.text:
                    logging.info('Already ping %s', text)
                    already_ping = True
                    break
            logging.info("people {}".format(people))
            logging.info(u"text {}".format(text))
            if self.Parameters.dry_run:
                continue
            if people and not already_ping:
                ST.ping(
                    self,
                    issue,
                    text,
                    people
                )

    def ping_forsaken_incidents(self):
        silence_time = self.config["incident_silence_time"]
        if type(self.config['st_product']) == list:
            logging.info("marketing_spi")
            self.ping_marketing_spis()
            return
        elif self.config['st_queue'] == 'DIRECT, UC':
            logging.info("direct queue")
            self.direct_ping()
            return
        else:
            query, _ = ST.forsaken_incidents_queries(
                silence_time,
                self.config['st_product'],
                self.config['st_queue'],
                self.config['ignore_tags'],
                self.config.get('forsaken_incidents_extra_filter', '')
            )
            issues = ST.find_tickets_by_query(self.st_helper, query)
        # for debug
        # issues = [self.st_helper.st_client.issues['MARTY-3998']]
        if not issues:
            logging.info("There are no issues with empty devops field")
        # limit on the number of pinged LSR tickets to check the functionality, ping 3 old tickets
        elif self.config['st_queue'] == 'LSR' and len(issues) > 2:
            issues = issues[-3:]
        for issue in issues:
            logging.info("status issue %s", issue.status.key)
            if not self.config['ping_development'] and issue.status.key == 'inDevelopment':
                continue
            logging.info('Checking issue `%s` for ping', issue.key)
            if self.config['st_product'] == 'PROD-VIDEO':
                text = self.config['tg_text']
                people = ST.get_issue_devops(issue, self.st_token)
            elif self.config['st_queue'] == 'MDBSUPPORT':
                today = datetime.datetime.today()
                if self.config.get('skip_weekends'):
                    if today.weekday() in [5, 6]:
                        # skip Saturday and Sunday
                        return

                    if today.month == 1 and today.day <= 7:
                        # skip 1-7th of January
                        return
                if today.hour < 10 and today.hour > 20:
                    return
                logging.info('update %s', issue.updatedAt)
                last_ping = datetime.datetime.strptime(issue.updatedAt[:-9], '%Y-%m-%dT%H:%M:%S')
                between = today - last_ping
                between_hours = between.total_seconds() // 3600
                between_hours_not_work = between.days * 14
                between_hour_work = between_hours - between_hours_not_work
                logging.info(
                    'between hour %s not work %s work %s',
                    between_hours, between_hours_not_work, between_hour_work
                )
                text = ''
                if issue.status.key == 'inProgress' and between_hour_work > 20:
                    text = 'Тикет в работе уже больше двух дней. Обновите статус тикета, пожалуйста.'
                if issue.status.key == 'needInfo' and between_hour_work > 40:
                    text = (
                        'В тикете нет движения уже больше 4х дней, '
                        'а информация все еще требуется. Предоставьте информацию, пожалуйста.'
                    )
                people = (ST.get_assignee(issue, self.st_token) + self.config['extra_summonees'])
                people = list(set(people))
                logging.info("status %s text %s people %s", issue.status.key, text, people)
            elif self.config['st_queue'] == 'RTCSUPPORT':
                time_since_last_update = ST.time_since_last_not_ping_update(issue, self.login)
                logging.info("status %s last_update %s", issue.status.key, time_since_last_update)
                text = self.config['tg_text'].format(silence_time=time_since_last_update)
                people = ST.get_assignee(issue, self.st_token)
                logging.info("status %s text %s people %s", issue.status.key, text, people)
            elif self.config['st_queue'] == 'BSAUDIT':
                if ST.get_assignee(issue, self.st_token) is None or 'AuditRelease' not in issue.summary:
                    continue
                importance = ST.count_ping_comments_before(issue, self.login)
                all_people = ST.get_assignee(issue, self.st_token) + [self.get_head(ST.get_assignee(issue, self.st_token)), self.get_head_of_department(BANNER_SYSTEM_DEPARTMENT_ID)]
                text = ''
                people = all_people[:min(3, importance + 1)]
            else:
                action_items = self.get_action_items(issue)
                action_items = self.add_action_items_from_links(issue, action_items)
                spproblem_keys = [
                    linkobject for linkobject in issue.links if linkobject.object.key.startswith("SPPROBLEM-")
                ]
                if self.config['st_queue'] == 'SPI':
                    if ST.is_issue_closed(issue):
                        logging.info('Issue %s closed, but without resolution', issue.key)
                        text = TEXT_WITHOUT_RESOLUTION
                    elif spproblem_keys:
                        logging.info('SPI depends SPProblem %s', spproblem_keys)
                        text = ''
                        for linkobject in spproblem_keys:
                            logging.info('SPI links type %s', linkobject.type.id)
                            if linkobject.type.id == 'depends':
                                text += TEXT_EXISTS_SPPROBLEM.format(spproblem_key=linkobject.object.key)
                            else:
                                text += TEXT_MAYBE_EXISTS_SPPROBLEM.format(
                                    spproblem_key=linkobject.object.key,
                                    spproblem_type_id=linkobject.type.id,
                                )
                    elif action_items:
                        logging.info("count action items links to SPI %s", len(action_items))
                        text = TEXT_EXISTS_ACTION_ITEMS.format(ai_count=str(len(action_items)))
                    else:
                        if silence_time:
                            text = BASE_TG_TEXT_TEMPLATED.format(
                                silence_time=ST.time_since_last_not_ping_update(issue, self.login)
                            )
                    text += self.config['tg_text']
                else:
                    text = self.config['tg_text']
                    if silence_time:
                        text = text.format(
                            silence_time=ST.time_since_last_not_ping_update(issue, self.login)
                        )

                text += 'Выполнено {}'.format(lb.task_wiki_link(self.id, 'SB:{}'.format(self.id)))
                importance = ST.count_ping_comments_before(issue, self.login) + 1
                if issue.key not in self.Context.forsaken_spi:
                    self.Context.forsaken_spi.append(issue.key)
                if self.config.get('ping_people'):
                    people = ST.get_summonees(
                        issue,
                        importance=importance,
                        extra_summonees=self.config.get("extra_summonees", [[], [], []]),
                        token=self.st_token
                    )
                    if self.config['st_queue'] == 'LSR' and 'talion' not in people:
                        people.append('talion')
                else:
                    people = []
            logging.info('ping people forsaken incidents %s', people)
            if self.Parameters.dry_run:
                continue
            all_comments = list(issue.comments.get_all())
            if self.check_the_text_for_similarity(text, all_comments):
                ST.update_comment(self, issue, all_comments[-1].id, text, people)
            else:
                ST.ping(
                    self,
                    issue,
                    text,
                    people
                )

    def get_action_items(self, issue):
        logging.info("Try to find action items for %s", issue.key)
        # MARTY-1978 правки разделителей для AI
        if issue.description:
            items_place = ''
            res_delim = self.find_delimiters(issue.description)
            logging.debug("AI tags in description: %s", res_delim)
            if res_delim:
                items_place = issue.description.split(res_delim[0])[-1].split(res_delim[1])[0].strip()
        else:
            logging.debug("%s empty descripton", issue.key)
            items_place = ''
        # items from description
        items = re.findall("[A-Z]+-[0-9]+", items_place)
        logging.info("AI found in description: %s", items)
        return items

    def add_action_items_from_links(self, issue, items):
        import startrek_client

        for link in issue.links:
            link_id = link.object.key
            logging.info('[add_action_items_from_links] Processing %s for %s', link_id, issue.key)

            if self._forbidden_issue(link_id, 'Get issue'):
                continue

            try:
                link_ticket = self.st_helper.st_client.issues[link_id]
            except startrek_client.exceptions.Forbidden as e:
                self._add_forbidden_issue(link_id, 'Get issue')
                continue

            except Exception as e:
                eh.log_exception("Failed to get issue via {link}".format(link=link_id), e, task=self)
                continue
            link_tags = link_ticket.tags
            for link_tag in link_tags:
                if link_tag.find("spi:actionitem") != -1:
                    logging.info("%s has %s", link_ticket.key, link_tag)
                    if items.count(link_ticket.key) == 0:
                        logging.info("ticket %s with spi:actionitem skipped in descr", link_ticket.key)
                        items.append(link_ticket.key)
                else:
                    if items.count(link_ticket.key):
                        logging.info("ticket %s without spi:actionitem tag", link_ticket.key)
        return items

    def action_item_need_ping(self, item_obj, silence_time, skipped_priority):
        if any([link.object.key.split('-')[0] == "LSR" for link in item_obj.links]):
            logging.info("Skipping %s action_item, because LSR in links", item_obj.key)
            return False
        if item_obj.key.split('-')[0] == self.config['st_queue']:  # incident can't be action item
            logging.info("Skipping %s action item, because incident can't be action item", item_obj.key)
            return False
        if item_obj.key.split('-')[0] == "SPPROBLEM" or item_obj.key.split('-')[0] == "LSR":
            logging.info("Skipping %s action item, because spproblem or lsr can't be action item", item_obj.key)
            return False
        if item_obj.key.split('-')[0] == "YP" or item_obj.key.split('-')[0] == "YTORM" or item_obj.key.split('-')[0] == "YPADMIN":
            logging.info("Skipping %s action item, because YP-, YTORM-, YPADMIN-", item_obj.key)
            return False
        priority = str(item_obj.priority.key)
        if ST.is_issue_closed(item_obj):
            logging.info("Skipping %s action item, because a resolution is not set", item_obj.key)
            return False
        if not ST.is_issue_forsaken(item_obj, silence_time=silence_time):
            logging.info("Skipping %s action item, because action_item_silence_time didn't run out", item_obj.key)
            return False
        if skipped_priority and priority in skipped_priority:
            logging.info(
                "Skipping %s action item, because priority %s is from the skipped_priority",
                item_obj.key, priority,
            )
            return False
        if item_obj.key in self.Context.forsaken_action_items:
            logging.info("Skipping %s action item, because already pinged", item_obj.key)
            return False
        if ST.is_deadline_ok(item_obj) or ST.is_sprint_ok(item_obj):
            logging.info("Skipping %s action item, because deadline or sprint is ok", item_obj.key)
            return False
        # DEVOPS-625
        if item_obj.key.startswith("NOCLSR") and item_obj.status.key == "resolved":
            logging.info("Skipping %s action item, because queue=NOCLSR and status is resolved", item_obj.key)
            return False
        return True

    def ping_action_items(self):
        logging.info("Ping action items started")
        silence_time = self.config["action_item_silence_time"]
        skipped_priority = self.config["action_item_skipped_priority"]
        # query = ST.prod_incidents_query(self.config['st_product'], self.config['st_queue'])
        query = ST.Query(
            product=self.config['st_product'],
            not_closed=False,
            queue=self.config['st_queue'],
            tags_to_ignore=self.config['ignore_tags'],
            created=self.start_date,
        ).build_query()
        # if we need to read this, we have a problem in code, so log only in debug mode
        logging.debug("Incidents query: %s", query)
        incidents = ST.find_tickets_by_query(self.st_helper, query)
        # incidents = [self.st_helper.st_client.issues['SPI-20756']]
        for incident in incidents:
            self.set_attribute_minusDc(incident)
            action_items = self.get_action_items(incident)
            ai_link = [link.object.key for link in incident.links if link.type.inward == "Is subtask for"]
            logging.info(u"Подзадача %s", ai_link)
            action_items.extend(ai_link)
            logging.info('get action items %s', action_items)

            #  MARTY-1978  items from links
            action_items = self.add_action_items_from_links(incident, action_items)
            logging.info("Action items found: %s", action_items)
            for action_item in action_items:
                if action_item.startswith(IGNORE_QUEUE):
                    logging.info("item %s filtered, because matched incidents queue", action_item)
                    continue

                logging.info('Checking action item %s', action_item)
                try:
                    item_obj = self.st_helper.st_client.issues[action_item]
                except Exception as e:
                    eh.log_exception("Failed to get action item from ST. Probably too simple regexp.", e, task=self)
                    continue
                if not self.action_item_need_ping(item_obj, silence_time, skipped_priority):
                    continue
                self.Context.forsaken_action_items.append(item_obj.key)
                importance = ST.count_ping_comments_before(item_obj, self.login) + 1
                people = ST.get_summonees(
                    item_obj,
                    importance=importance,
                    extra_summonees=self.config.get("extra_summonees", [[], [], []]),
                    token=self.st_token,
                )
                human_silence_time = ST.time_since_last_not_ping_update(item_obj, self.login)
                text = ACTION_ITEM_PING_TEMPLATE.format(
                    incident=incident.key,
                    silence_time=human_silence_time,
                    sandbox_task_link=lb.task_wiki_link(self.id, self.id)
                )
                all_comments = list(item_obj.comments.get_all())
                if self.check_the_text_for_similarity(text, all_comments):
                    ST.update_comment(self, item_obj, all_comments[-1].id, text, people)
                else:
                    ST.ping(
                        self,
                        item_obj,
                        text,
                        people
                    )

    def ydt_weight_importance_update(self, queue):
        """
        Calculate summary ydt times for SPPROBLEM for each linked SPI and SPPROBLEM importance.

        :return: list of SPPROBLEMs (dict: problem key -> ydt's)
        """
        if self.Parameters.dry_run:
            return
        logging.info("Started count summary YDT")
        query = ST.Query(queue=queue, not_closed=False).build_query()

        # log query only in debug mode
        logging.debug("Incidents query: %s", query)
        # for debug
        # issues = [self.st_helper.st_client.issues['LSR-431']]
        issues = ST.find_tickets_by_query(self.st_helper, query)
        main_issues_with_link = self.collect_related_tickets(issues, 'depends', 'inward', '', '', "SPI-")
        for issue, links in main_issues_with_link.items():
            self.update_sum_ydt(issue, links)
            logging.info("Processing problem %s", issue.key)
            sp = {
                'params': {'notify': False},
                'weight': 0,
                'importance': 0
            }
            logging.info("original weight is %s", issue.weight)
            for link in links:
                if (queue == "LSR" and not link.object.key.startswith("SPI-")):
                    continue
                sp['weight'] += int(self.get_ticket_weight(link.object))
                # Значение поля importance - количество инцидентов, которые блокируют SPPROBLEM и
                # котроые относятся к очередям из словаря BLOCKED_QUEUE.
                if not link.object.key.startswith(BLOCKED_QUEUE):
                    continue
                logging.info("Blocked issue: %s", link.object.key)
                sp['importance'] += 1
            try:
                sp_issue = self.st_helper.st_client.issues[issue.key]
                sp_issue.update(**sp)
            except Exception as e:
                eh.log_exception('Failed to set new weight for isssue {}'.format(issue.key), e)

    def get_ticket_weight(self, issue):
        if 'SUPPORT' in issue.key:
            story_points = POINTS_WEIGHTS['SUPPORT']
        elif issue.key.startswith('SPI-'):
            story_points = POINTS_WEIGHTS['SPI'].get(issue.priority.key.lower(), '')
        elif issue.key.startswith('LSR-'):
            story_points = POINTS_WEIGHTS['LSR']
        else:
            story_points = POINTS_WEIGHTS['default']
        try:
            story_points *= IMPACT_WEIGHTS.get(issue.impact, 1)
        except Exception as e:
            # так как ошибка регулярная не логируем трейс
            logging.error('Access denied %s. Error: {}'.format(e), issue.key)
        return story_points

    def marty_sla_ping(self, marty_list):
        staff_client = StaffClient(token=self.st_token)
        logging.info("Started check MARTY SPI")
        start_date = (datetime.datetime.now() - datetime.timedelta(90)).strftime('%Y-%m-%d')
        sla_stats = []
        if marty_list:
            for martyr in marty_list:
                issues = self.st_helper.st_client.issues.find(
                    filter={
                        'queue': 'SPI',
                        'created': {'from': start_date},
                        'author': martyr,
                    }
                )
                pers_stat = {
                    'login': martyr,
                    'correct': [],
                    'no_duty_action': [],
                    'no_timeline': [],
                    'tg_login': staff_client.get_person_messenger(martyr),
                }
                if not issues:
                    logging.info("No tickets created by %s, nothing to process", martyr)
                    continue

                logging.info("Tickets created by %s", martyr)

                for issue in issues:
                    logging.debug('Issue %s created by %s', issue.key, issue.createdBy.id)
                    if not issue.description:
                        continue

                    if isinstance(issue.tags, list) and 'unmonitored'.startswith(tuple(issue.tags)):
                        continue

                    duty_actions = (
                        issue.description.split(u'== Действия дежурной смены ==')[-1].split(u'==Анализ==')[0].strip()
                    )
                    if not duty_actions:
                        logging.info('%s no duty actions', issue.key)
                        pers_stat['no_duty_action'].append(issue.key)
                        continue

                    duty_timeline = re.findall(r'\s*[0-9]?[0-9]:[0-9]{2}', duty_actions)
                    if not duty_timeline:
                        logging.info('%s no timeline', issue.key)
                        pers_stat['no_timeline'].append(issue.key)
                        continue

                    logging.info('%s ALL marty tags are here', issue.key)
                    pers_stat['correct'].append(issue.key)

                pers_stat['perc_correct'] = 100.0 * (
                    1.0 - float(len(pers_stat['no_duty_action']) + len(pers_stat['no_timeline'])) / len(issues)
                )
                pers_stat['created_by_marty'] = len(issues)
                sla_stats.append(pers_stat)

        return sla_stats

    def marty_sla_all(self):
        return {
            'marties': self.marty_sla_ping(self.config['marties']),
            'pers_marties': self.marty_sla_ping(self.config['persmarties']),
            'efir_marties': self.marty_sla_ping(self.config['efirmarties']),
        }

    @staticmethod
    def marty_sla_telegram_ping(sla_stats):
        message = []
        logging.debug(sla_stats)
        if isinstance(sla_stats, dict):
            for k in sla_stats.keys():
                message.append("Статистика для группы {}".format(k))
                if isinstance(sla_stats.get(k), list):
                    for sla_stat in sla_stats.get(k):
                        try:
                            message.append("Марти:{0} коррект:{1} всего:{2}".format(
                                sla_stat['login'],
                                "{:.2f}".format(sla_stat['perc_correct']),
                                sla_stat['created_by_marty'],
                            ))
                            if sla_stat['no_timeline']:
                                marty_err = ST.Query(extra_filter='Key:{}'.format(
                                    ','.join(sla_stat['no_timeline']))
                                ).build_short_query_url()
                                message.append("@{0} не заполнен таймлайн: {1}".format(sla_stat['tg_login'], marty_err))
                            if sla_stat['no_duty_action']:
                                marty_err = ST.Query(extra_filter='Key:{}'.format(
                                    ','.join(sla_stat['no_duty_action']))
                                ).build_short_query_url()
                                message.append("@{0} нет действий дежурной смены: {1}".format(
                                    sla_stat['tg_login'], marty_err
                                ))
                        except Exception as e:
                            logging.exception(e)
                            pass
                    message.append('#{}_timelines_report \n'.format(k))
        message = '\n'.join(message)
        return message

    @staticmethod
    def find_delimiters(descr):
        """
        Find delimiters in description.

        :param descr: ticket description (string)
        :return: list of delimiters in description
        """
        kword = []
        for delim in ACTION_ITEM_DELIMS:
            logging.debug('Checking delimiters {delimiters}'.format(delimiters=delim))
            delimiters = (
                re.findall(ur"[=]+[\s]?{}[\s]?[=]*".format(delim[0]), descr),
                re.findall(ur"[=]+[\s]?{}[\s]?[=]*".format(delim[1]), descr),
            )
            if delimiters[0] and delimiters[1]:
                kword = (delimiters[0][-1].strip(), delimiters[1][-1].strip())
        return kword

    @staticmethod
    def ydt_time_tosec(ydt_time):
        """
        Convert ydt time inteval to seconds.

        :param ydt_time  PT?([0-9]W)?([0-9]D)?([1-9]H)?([1-59]M)?([1-59]S)?
        :return overall_time in seconds
        """
        match = re.match('PT?(?:([0-9]+)W)?(?:([0-9]+)D)?(?:([0-9]+)H)?(?:([0-9]+)M)?(?:([0-9]+)S)?', ydt_time.upper())
        if not match:
            logging.debug('ydt format is incorrect %s', ydt_time)
            return 0
        weeks, days, hours, minutes, seconds = [int(item or 0) for item in match.groups()]
        return weeks * 60 * 60 * 24 * 7 + days * 60 * 60 * 24 + hours * 60 * 60 + minutes * 60 + seconds

    def _add_forbidden_issue(self, key, action):
        logging.error('No access to `%s`: %s failed', key, action)
        self.set_info('No access to `{}`: {} failed'.format(key, action))
        self.forbidden_issues.add(key)

    def _forbidden_issue(self, key, action):
        if key in self.forbidden_issues:
            logging.error('No access to `%s`: %s failed [cached]', key, action)
            return True
        return False


class Schedule(object):

    @staticmethod
    def get_schedule(arcadia_schedule_url, checkouted=None):
        checkouted = checkouted if checkouted else []
        if not checkouted:
            sdk2.svn.Arcadia.checkout(arcadia_schedule_url, 'schedule_data')
            checkouted.append(True)
        with open('schedule_data/websearch_schedule.json', 'r') as schedule_file:
            result = json.load(schedule_file)
        return result

    @staticmethod
    def get_current_duty_dict(arcadia_schedule_url):
        for week in reversed(Schedule.get_schedule(arcadia_schedule_url)):
            year, month, day = map(int, week['week'].split('.'))
            duty_start = datetime.datetime(year, month, day, 17)
            if duty_start > datetime.datetime.now():
                # They are future devops who are scheduled but not started yet, skip them
                continue
            return week

    @staticmethod
    def get_last_duty_dict(arcadia_schedule_url):
        return Schedule.get_schedule(arcadia_schedule_url)[-1]
