# -*- coding: utf-8 -*-
import datetime
import time
import logging
import json
import requests
from sandbox import sdk2

UPSTREAMS_FRAME = 100
NANNY_URL = 'http://nanny.yandex-team.ru'
NANNY_SECTION_URL = 'https://nanny.yandex-team.ru/ui/#/services/balancers/list/production/sections/list'
NANNY_SERVICE_URL = '((https://nanny.yandex-team.ru/ui/#/services/catalog/{}/ {}))'
YASM_PANEL_URL = 'https://yasm.yandex-team.ru/template/panel/dynbalancer_common_panel/section={}/?from=1583010000000&to=1585688340000'


class ResultQuotasResource(sdk2.Resource):
    """ Plaintext file resource """


class DynBalancerStats(sdk2.Task):
    """
    Various notification for devops vanguard.

    * List of incidents without "Sotrudniki->Dezhurniy" field set
    * List of incidents that has no actions for too long
    * To assign next guardians of the realm
    """

    class Requirements(sdk2.Requirements):
        """ Plaintext file resource """

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.RadioGroup('Format_print') as format_print:
            format_print.values['wiki'] = format_print.Value(value='wiki', default=True)
            format_print.values['csv'] = format_print.Value(value='csv')

        with sdk2.parameters.Group('Datetime') as datetime_between:
            start_time = sdk2.parameters.String(
                'Datetime start',
                default='2020-03-01 00:00:00',
                required=True
            )
            end_time = sdk2.parameters.String(
                'Datetime end',
                default='2020-03-31 00:00:00',
                required=True
            )
        categories_whitelist = sdk2.parameters.String(
            'Categories list',
            default='/samogon',
            required=True
        )
        with sdk2.parameters.Group('Tokens') as token:
            nanny_token = sdk2.parameters.String(
                'NANNY OAUTH token',
                default='nanny_token',
                required=True
            )
            awacs_token = sdk2.parameters.String(
                'AWACS OAUTH token',
                default='awacs_token',
                required=True
            )
            solomon_token = sdk2.parameters.String(
                'SOLOMON token',
                default='solomon_token',
                required=True
            )
            owner_token = sdk2.parameters.String(
                'Owner token',
                default='owner-token',
                required=True
            )

    def on_execute(self):
        logging.info('start')
        self.session = requests.Session()
        self.session.headers['Content-Type'] = 'application/json'
        self.solomon_token = sdk2.Vault.data(self.Parameters.owner_token, self.Parameters.solomon_token)
        self.awacs_token = sdk2.Vault.data(self.Parameters.owner_token, self.Parameters.awacs_token)
        self.nanny_token = sdk2.Vault.data(self.Parameters.owner_token, self.Parameters.nanny_token)

        resource = ResultQuotasResource(self, "Result file", "result_dir")
        output_file = sdk2.ResourceData(resource)
        output_file.path.mkdir(0o755, parents=True, exist_ok=True)

        logging.info('get_nanny_services')
        services = self.get_nanny_services()

        dt_start = datetime.datetime.strptime(self.Parameters.start_time, '%Y-%m-%d %H:%M:%S')
        dt_end = datetime.datetime.strptime(self.Parameters.end_time, '%Y-%m-%d %H:%M:%S')
        et = time.mktime(dt_end.timetuple())  # Интересуют последние гарантированно агрегированные точки
        st = time.mktime(dt_start.timetuple())  # Все часовые точки за недею

        upstreams = self.get_balancer_upstreams('s.yandex-team.ru')
        stats = self.get_upstreams_stats(upstreams, st, et)
        self.print_results(services, upstreams, stats, self.Parameters.categories_whitelist, output_file)

    def get_nanny_services(self):
        headers = {"content-type": "application/json",
                   "accept": "application/json",
                   "Authorization": "OAuth {0}".format(self.nanny_token)}

        r = requests.get('https://nanny.yandex-team.ru/v2/services/?limit=10&skip=10&exclude_runtime_attrs=1',
                         headers=headers,
                         timeout=100)
        if r.status_code == 200:
            response = r.json()
            services = response.get('result', [])
            service_info = {}
            for svc in services:
                abc = svc.get('info_attrs', {}).get('content', {}).get('abc_group', None)
                service_info[svc['_id']] = {'abc': abc,
                                            'owners': svc.get('auth_attrs', {}).get('content', {}).get('owners', {}),
                                            'category': svc.get('info_attrs', {}).get('content', {}).get('category', None)}
            return service_info
        else:
            return []

    def get_upstream_stats(self, upstream, tmstmp_st, tmstmp_end):

        base_url = "https://solomon.yandex-team.ru/api/v2/projects/yasm_{0}/sensors/data"
        expression_template = '''signal="balancer_report-report-{0}-succ_summ", cluster="group_*", service="yasm", prj="dynbalancer"'''
        hdrs = {
            "Authorization": "OAuth {}".format(self.solomon_token),
            "Content-Type": "application/json;charset=UTF-8",
            "Accept": "application/json"
        }

        fmt = "%Y-%m-%dT%H:%M:%S.%fZ"

        start, end = (datetime.datetime.fromtimestamp(tmstmp_st) - datetime.timedelta(hours=3)).strftime(fmt), \
                     datetime.datetime.fromtimestamp(tmstmp_end).strftime(fmt)
        sum_rps = 0
        try:
            expr = "summary_sum({" + expression_template.format(upstream) + "})"
            payload = {
                "program": expr,
                "from": start,
                "to": end,
                "downsampling": {
                    "gridMillis": 3600000,
                    "fill": "NONE",
                }
            }
            logging.info('post %s', requests.post(base_url.format('balancer'), data=json.dumps(payload), headers=hdrs))
            results = requests.post(base_url.format('balancer'), data=json.dumps(payload), headers=hdrs).json().get('vector', [])
            for res in results:
                sum_val = 0
                group = res.get('timeseries', {}).get('labels', {}).get('group')
                if group.startswith('SAS_KERNEL_TEST'):
                    continue
                values = res.get('timeseries', {}).get('values', [])
                for val in values:
                    sum_val += val
                sum_rps += sum_val

        except AttributeError:
            pass

        return sum_rps

    def get_upstreams_stats(self, upstreams, start_time, end_time):
        stats = {}
        for u, u_meta in upstreams.iteritems():
            logging.info('getting stats for %s', u)
            stats[u] = self.get_upstream_stats(u, start_time, end_time)
        return stats

    def get_balancer_upstreams(self, namespace):
        result = {}
        skip = 0
        self.session.headers['Authorization'] = 'OAuth {}'.format(self.awacs_token)
        while True:
            response = self.session.post(
                'https://awacs.yandex-team.ru/api/ListUpstreams/',
                json={'namespaceId': namespace,
                      'limit': UPSTREAMS_FRAME,
                      'skip': skip}
            )
            response_data = response.json()
            for u in response_data.get('upstreams'):
                if u.get('spec', {}).get('deleted'):
                    continue
                upstream = u.get('meta', {}).get('id', 'unknown')
                modules = u.get('spec', {}).get('yandexBalancer', {}).get('config', {}).get('regexpSection', {}).get('nested', {}).get('modules', {})
                nanny_services = []
                for m in modules:
                    if 'balancer2' not in m:
                        continue
                    nanny_services = [svc['serviceId'] for svc in m['balancer2'].get('generatedProxyBackends', {}).get('nannySnapshots', [])]
                result[upstream] = nanny_services
            if len(response_data.get('upstreams')) < UPSTREAMS_FRAME:
                break
            skip += UPSTREAMS_FRAME

        return result

    def print_results(self, services, upstreams, upstreams_stats, categories_whitelist, output_file):
        logging.info('services = %s', services)
        logging.info('upstreams = %s', upstreams)
        logging.info('upstreams_stats = %s', upstreams_stats)
        logging.info('categories_whitelist = %s', categories_whitelist)
        res_data = []
        if self.Parameters.format_print == 'wiki':
            output_file = output_file.path.joinpath("stats")
            res_data.append('#|')
            res_data.append('||**Section** |**Total requests** |**Service** |**Category** ||')

        elif self.Parameters.format_print == 'csv':
            output_file = output_file.path.joinpath("stats.csv")
            res_data.append('Section;Requests;Service;Category')
        else:
            output_file = output_file.path.joinpath("stats.txt")

        for u, rps in sorted(upstreams_stats.iteritems(), key=lambda (k, v): v, reverse=True):
            categories = [services.get(backend_svc, {}).get('category', 'unknown') for backend_svc in upstreams[u]]
            if categories_whitelist:
                categories = [c for c in categories if c.startswith(categories_whitelist)]
            if len(categories) == 0:
                continue
            logging.info('categories =  %s', categories)

            if self.Parameters.format_print == 'wiki':
                res_data.append('||(({}/{} {}))|(({} {})) |{} |{} ||'.format(
                    NANNY_SECTION_URL,
                    u,
                    u,
                    YASM_PANEL_URL.format(u),
                    int(rps),
                    ','.join([NANNY_SERVICE_URL.format(s, s) for s in upstreams[u]]),
                    ','.join(categories))
                )
            elif self.Parameters.format_print == 'csv':
                res_data.append('{};{};{};{}'.format(u, int(rps), ','.join(upstreams[u]), ','.join(categories)))
            else:
                res_data.append('{}\t{}\t{}\t{}'.format(u, int(rps), ','.join(upstreams[u]), ','.join(categories)))

        if self.Parameters.format_print == 'wiki':
            res_data.append('|#')
        output_file.write_bytes('\n'.join(res_data))
        return
