# -*- coding: utf-8 -*-
"""
Usage example.

>>> fixer = SandboxGroupDecomposer("SEARCH-RELEASERS", '<auth>')
>>> group_info = fixer.get_group_info()
>>> print(fixer.log_members_by_staff_groups(group_info, group_level=2))

"""


from sandbox.common.rest import Client


class SandboxGroupDecomposer(object):

    STAFF_URL = "http://api.staff.yandex-team.ru/v3/"

    def __init__(self, group_name, auth=None):
        """
        Construct SandboxGroupDecomposer.

        :param group_name: target group name
        :param auth: instance of `sandbox.common.proxy.Authentication`
        """
        self._group_name = group_name
        self._sandbox_client = None
        self._staff_client = None
        self._auth_token = auth

    @property
    def sandbox_client(self):
        """Return Sandbox client based on `sandbox.common.rest.Client`."""
        if self._sandbox_client is None:
            self._sandbox_client = Client(base_url=Client.DEFAULT_BASE_URL, auth=self._auth_token)
        return self._sandbox_client

    @property
    def staff_client(self):
        """Return Staff client based on `sandbox.common.rest.Client`."""
        if self._staff_client is None:
            self._staff_client = Client(base_url=self.STAFF_URL, auth=self._auth_token)
        return self._staff_client

    def get_group_info(self):
        """
        Retrieve and return sandbox group info.

        Original API method:
        [](https://sandbox.yandex-team.ru/media/swagger-ui/index.html#/group/group_get)
        """
        group_data = self.sandbox_client.group[self._group_name].read()
        return group_data

    def combine_members_by_staff_groups(
        self, group_info,
        group_search_key='type',
        group_search_value='department',
        group_level=1,
    ):
        """
        Combine members by staff groups.

        For each member in the given group searches for his/her first group with `group_search_key` value
        matching `group_search_value` and returns a dict with users combined by such groups.

        If `group_level` provided and `group_level` >= 2 then this would try to combine at the higher level
        groups. E.g., if group_level=2 groups' parents are taken.

        :param group_info: result of `group_info`
        :param group_search_key: search field
        :param group_search_value: value to search for
        :param group_level: integer, >= 1, group combine level
        :return:
        ```
        {
            '<group_url>': {
                'id': <group_id>,
                'name': <group_name>,
                'members': ['<username>', '<username>', ...]
            },
            ...
        }
        ```
        """
        members = group_info.get('members', [])
        result_group_dict = {}
        for member in members:
            member_groups = self.staff_client.persons.read(_one=1, login=member, _fields='groups').get('groups', [])
            for group_dict in member_groups:
                group = group_dict['group']
                if group.get(group_search_key, None) != group_search_value:
                    continue
                # todo: handle `is_deleted` flag
                level = group_level
                while level > 1:
                    level -= 1
                    if 'parent' not in group or group['parent'] is None:
                        break
                    group = group['parent']
                group_url = group['url']
                if group_url not in result_group_dict:
                    result_group_dict[group_url] = {
                        'name': group['name'],
                        'id': group['id'],
                        'members': []
                    }
                result_group_dict[group_url]['members'].append(member)
                break
        return result_group_dict

    def log_members_by_staff_groups(self, *args, **kwargs):
        """
        Combine members by staff groups and return output as string.

        :return: A log string
        """
        combined = self.combine_members_by_staff_groups(*args, **kwargs)
        result = ""
        for gr_url in sorted(combined):
            gr = combined[gr_url]
            result += "-= {group_name} ({group_url}) =-\nMembers: {mem_list}\n{sep}\n".format(
                group_name=gr['name'],
                group_url=gr_url,
                mem_list=", ".join(gr['members']),
                sep='-' * 80,
            )
        return result
