# -*- coding: utf-8 -*-
import logging
from datetime import datetime, time
import requests
import json
from sandbox import sdk2
from sandbox.sandboxsdk import environments
import yql_script
from dateutil.relativedelta import relativedelta


class HardwareGencfgGroups(sdk2.Task):
    """
    Various notification for devops vanguard.

    * List of incidents without "Sotrudniki->Dezhurniy" field set
    * List of incidents that has no actions for too long
    * To assign next guardians of the realm
    """

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt', use_wheel=True),
            environments.PipEnvironment('yandex-yt-yson-bindings', use_wheel=True),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet', use_wheel=True),
            environments.PipEnvironment('yql', use_wheel=True)
        )

    class Parameters(sdk2.Task.Parameters):

        proxy = sdk2.parameters.String(
            "Proxy",
            default='hahn',
        )
        time_window = sdk2.parameters.Integer(
            "Time window",
            default=60,
        )
        source_table = sdk2.parameters.String(
            "Table mapping",
            default='//home/metering/gencfg/mapping',
        )
        output_table = sdk2.parameters.String(
            "Output table",
            default='//home/metering/gencfg/groups_test',
        )
        yasm_table = sdk2.parameters.String(
            "Yasm table",
            default='//home/data_com/cubes/gencfg/gencfg_yasm_groups'
        )
        period = sdk2.parameters.Integer(
            "Period time",
            default=6
        )
        with sdk2.parameters.Group('Tokens') as token:
            solomon_token = sdk2.parameters.String(
                'SOLOMON token',
                default='solomon-token-robot-gencfg',
                required=True
            )
            yql_token = sdk2.parameters.String(
                'YQL token',
                default='yql-token-robot-gencfg',
                required=True
            )
            yt_token = sdk2.parameters.String(
                'YT token',
                default='yt-token-robot-gencfg2',
                required=True
            )
            abc_token = sdk2.parameters.String(
                'ABC token',
                default='abc-token-robot-gencfg',
                required=True
            )

    def on_execute(self):
        import yt.wrapper as yt
        self.yt = yt
        self.yt_client = self.yt.YtClient('hahn', sdk2.Vault.data(self.owner, self.Parameters.yt_token))

        solomon_token = sdk2.Vault.data(self.owner, self.Parameters.solomon_token)
        abc_token = sdk2.Vault.data(self.owner, self.Parameters.abc_token)

        self.get_data_from_solomon_to_yt(solomon_token)
        self.tree_abc = {}
        self.foliage = []
        self.all_path = {}
        self.max_path = 0
        self.get_parent_abc(self.Parameters.source_table, abc_token)
        self.push_solomon_data(self.Parameters.output_table, solomon_token)

    def get_data_from_solomon_to_yt(self, solomon_token):
        from yql.api.v1.client import YqlClient

        end = datetime.now()
        start = end + relativedelta(days=(-1)*self.Parameters.period)
        end = end.isoformat(sep='T') + 'Z'
        start = start.isoformat(sep='T') + 'Z'

        yql_requests = yql_script.script.replace('<start>', start).replace('<end>', end)
        yql_requests = yql_requests.replace('<time_window>', str(self.Parameters.time_window)).replace('<token>', solomon_token)
        yql_requests = yql_requests.replace('<output_table>', self.Parameters.output_table).replace('<mapping_table>', self.Parameters.source_table)
        yql_requests = yql_requests.replace('<yasm_table>', self.Parameters.yasm_table)

        def run_query(query):
            with YqlClient(db='hahn', token=sdk2.Vault.data(self.owner, self.Parameters.yql_token)) as yql_client:
                request = yql_client.query(query, syntax_version=1)
                request.run()
                request.wait_progress()
                query_result = request.get_results()
                if not query_result.is_success:
                    raise Exception('yql_query result is not successful')
                return request
        request = run_query(yql_requests)

    def dfs(self, root, maybe_foliage, id_nested):
        if root:
            if id_nested:
                root.update({'name': 'nested_{}_parent'.format(id_nested)})
            id_nested += 1
            for next_root in root.keys():
                if next_root != 'name':
                    self.dfs(root[next_root], next_root, id_nested)
        else:
            self.foliage.append(maybe_foliage)

    def get_parent_abc(self, mapping_table, abc_token):
        from yql.api.v1.client import YqlClient

        get_abc = """
            INSERT INTO hahn.`//home/metering/gencfg/abc`
            WITH TRUNCATE
                SELECT DISTINCT abc
            FROM hahn.`{}`
        """.format(mapping_table)

        def run_query(query):
            with YqlClient(db='hahn', token=sdk2.Vault.data(self.owner, self.Parameters.yql_token)) as yql_client:
                request = yql_client.query(query, syntax_version=1)
                request.run()
                request.wait_progress()
                query_result = request.get_results()
                if not query_result.is_success:
                    raise Exception('yql_query result is not successful')
                return request
        request = run_query(get_abc)

        url_abc_api = 'https://abc-back.yandex-team.ru/api/v4/services/?fields=path&slug={}'
        headers = {
            "Authorization": "OAuth {0}".format(abc_token),
            "Content-Type": "application/json;charset=UTF-8",
            "Accept": "application/json"
        }
        all_row = self.yt_client.read_table(self.yt_client.TablePath('//home/metering/gencfg/abc'))
        for input_row in all_row:
            if input_row['abc']:
                response = requests.get(url_abc_api.format(input_row['abc']), headers=headers).json()
                path = response['results'][0]['path']
                all_parents = path.split('/')
                all_parents = [parent for parent in all_parents if parent]
                self.all_path.update({input_row['abc']: all_parents})
                if len(all_parents) > self.max_path:
                    self.max_path = len(all_parents)
                if all_parents[0] not in self.tree_abc.keys():
                    self.tree_abc[all_parents[0]] = {'name': 'main_parent'}
                branch_abc = self.tree_abc[all_parents[0]]
                for parent in all_parents[1:]:
                    if parent not in branch_abc.keys():
                        branch_abc.update({parent: {}})
                    branch_abc = branch_abc[parent]
        for root in self.tree_abc.keys():
            id_nested = 0
            self.dfs(self.tree_abc[root], root, id_nested)

    def push_solomon_data(self, input_table, solomon_token):
        solomon_data_api_url = 'https://solomon.yandex.net/api/v2/push?project={}&cluster={}&service={}'
        project = 'gencfg'
        cluster = 'resource-usage'
        service = 'master_group-usage'
        headers = {
            'Authorization': 'OAuth {}'.format(solomon_token),
            'Content-Type': 'application/json',
            'Accept': 'application/json',
        }
        data = {
            'sensors': []
        }
        ind_row = 0
        all_row = self.yt_client.read_table(self.yt_client.TablePath(input_table))
        for input_row in all_row:
            ind_row += 1
            if not input_row['abc']:
                sensor = {
                    'labels': {
                        'group': 'unknown',
                        'sensor': input_row['sensor'] + '_own'
                    },
                    'ts': input_row['timestamp'],
                    'value': input_row['value'],
                }
            else:
                if input_row['abc'] in self.foliage:
                    sensor = {
                        'labels': {
                            'group': input_row['abc'],
                            'sensor': input_row['sensor']
                        },
                        'ts': input_row['timestamp'],
                        'value': input_row['value'],
                    }
                else:
                    sensor = {
                        'labels': {
                            'group': input_row['abc'],
                            'sensor': input_row['sensor'] + '_own'
                        },
                        'ts': input_row['timestamp'],
                        'value': input_row['value'],
                    }
                path = self.all_path[input_row['abc']]
                id_nested = 0
                branch_abc = self.tree_abc[path[id_nested]]
                while branch_abc.get('name'):
                    sensor['labels'].update({branch_abc['name']: path[id_nested]})
                    id_nested += 1
                    if id_nested < len(path):
                        branch_abc = branch_abc[path[id_nested]]
                    else:
                        break
                while id_nested < self.max_path - 1:
                    sensor['labels'].update({'nested_{}_parent'.format(id_nested): "no_{}".format(id_nested)})
                    id_nested += 1
            data['sensors'].append(sensor)
            if ind_row % 1000 == 0:
                logging.info('data lens {}, ind_row {}'.format(len(data['sensors']), ind_row))
                for delay in (0, 3, 6, 9, -1):
                    try:
                        logging.info('Start push solomon data for ({} sensors)'.format(len(data['sensors'])))
                        response = requests.post(solomon_data_api_url.format(project, cluster, service), headers=headers, data=json.dumps(data))
                        logging.info('request : url {}', response.request.url)
                        if response.status_code != 200:
                            raise ValueError('HTTP Error {} != 200 ({})'.format(response.status_code, response.text))
                    except Exception as e:
                        if delay >= 0:
                            logging.error(' {}: {}, sleep {}'.format(type(e), e, delay))
                            time.sleep(delay)
                        else:
                            logging.error('{}: {}, FAIL'.format(type(e), e))
                            raise
                data = {
                    'sensors': []
                }
        for delay in (0, 3, 6, 9, -1):
            try:
                logging.info('Start push solomon data for ({} sensors)'.format(len(data['sensors'])))
                response = requests.post(solomon_data_api_url.format(project, cluster, service), headers=headers, data=json.dumps(data))
                if response.status_code != 200:
                    raise ValueError('HTTP Error {} != 200 ({})'.format(response.status_code, response.text))
            except Exception as e:
                if delay >= 0:
                    logging.error(' {}: {}, sleep {}'.format(type(e), e, delay))
                    time.sleep(delay)
                else:
                    logging.error('{}: {}, FAIL'.format(type(e), e))
                    raise
