# -*- coding: utf-8 -*-
import logging
import json
import datetime
import time
import re
import requests

from sandbox import sdk2

import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as rm_task_env
from sandbox.projects.release_machine import rm_notify
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.common.startrek_utils import ST
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import task_env
from sandbox.projects.common import requests_wrapper

from sandbox.projects.devops.SupportAlert.main_config import (
    ACTION_ITEM_DELIMS,
    IGNORE_QUEUE,
    BANNER_SYSTEM_DEPARTMENT_ID
)

from sandbox.projects.devops.SupportAlert.templates import (
    RESPONSIBLE,
    BASE_TG_TEXT_TEMPLATED,
    TEXT_WITHOUT_RESOLUTION,
    TEXT_EXISTS_SPPROBLEM,
    TEXT_MAYBE_EXISTS_SPPROBLEM,
    TEXT_EXISTS_ACTION_ITEMS,
    ACTION_ITEM_PING_TEMPLATE,
    BASE_RTCSUPPORT_TEXT,
    BASE_RESPONSIBLE_TEXT
)

RM_CONFIG = {
    'st': {
        'token_name': rm_const.COMMON_TOKEN_NAME,
        'token_owner': rm_const.COMMON_TOKEN_OWNER,
    },

    # first list for l1,l2 and l3; second for l2 and l3; third for l3
    'extra_summonees': [["mvel"]],
    'st_product': None,
    'st_queue': "RMINCIDENTS",
    'incident_silence_time': 7,
    'action_item_silence_time': 14,
    'ignore_tags': [],
    'ping_action_items': True,
    'ping_forsaken_incidents': True,
    'check_forsaken_ticket': True,
    'tg_text': BASE_TG_TEXT_TEMPLATED + RESPONSIBLE.format(
        devops_doc_link='https://nda.ya.ru/3Tvds7',
        alt_devops_master='ilyaturuntaev',
    ),
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'ping_development': True,
}


MDBSUPPORT_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    # first list for l1,l2 and l3; second for l2 and l3; third for l3
    'extra_summonees': ["amatol", "yurovniki"],
    'st_product': "",
    'st_queue': "MDBSUPPORT",
    'incident_silence_time': 2,
    'action_item_silence_time': 60,
    'tags_to_check': ['area', 'impact', 'support_line'],
    'fields_to_check': ['SRE Area', 'Impact', 'Support_line'],
    'sre_times_filter': '("Notification Time": empty() OR "Begin Time": empty() OR "End Time": empty())',
    'ignore_tags': [],
    'ping_action_items': False,
    'ping_forsaken_incidents': True,
    'check_forsaken_ticket': True,
    'tg_text': BASE_RESPONSIBLE_TEXT,
    'forsaken_incidents_extra_filter': (
        '((Status: "В работе" AND Updated: < now() - 2d) OR '
        '(Status: "Требуется информация" AND (Deadline: < today() OR Deadline: empty()) AND Updated: < now() - 4d))'
    ),
    'skip_weekends': True,
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'day_for_ping_tg': [0],
    'ping_development': True,
}

RTCSUPPORT_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    # first list for l1,l2 and l3; second for l2 and l3; third for l3
    'extra_summonees': [],
    'st_product': "",
    'st_queue': "RTCSUPPORT",
    'incident_silence_time': 1,
    'action_item_silence_time': 60,
    'tags_to_check': ['area', 'impact', 'support_line'],
    'fields_to_check': ['SRE Area', 'Impact', 'Support_line'],
    'sre_times_filter': '("Notification Time": empty() OR "Begin Time": empty() OR "End Time": empty())',
    'ignore_tags': [],
    'ping_action_items': False,
    'ping_forsaken_incidents': True,
    'check_forsaken_ticket': True,
    'tg_text': BASE_RTCSUPPORT_TEXT,
    'forsaken_incidents_extra_filter': (
        '((Status: "В работе" AND Updated: < now() - 2d) OR Status: "Требуется информация" OR Status: "Линия 2")'
    ),
    'skip_weekends': True,
    'ping_people': True,
    'action_item_skipped_priority': ["minor"],
    'day_for_ping_tg': [0],
    'ping_development': True,
}

BSAUDIT_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    'extra_summonees': [],
    'ignore_tags': [],
    'st_product': '',
    'st_queue': "BSAUDIT",
    'incident_silence_time': 2,
    'ping_forsaken_incidents': True,
    'ping_action_items': False,
    'ping_people': True,
    'ping_development': True,
}

DIRECT_CONFIG = {
    'st': {
        'secret_id': 'sec-01dg28wrjccyay1995dby6jsbp',
        'key': 'robot-search-devops_token'
    },
    'arcadia_schedule_url': 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/tools/devops/schedule/data/',
    'extra_summonees': [],
    'ignore_tags': [],
    'st_product': '',
    'st_queue': "DIRECT, UC",
    'forsaken_incidents_extra_filter': (
        'Filter: 53044 Priority: Критичный Stage: Production'
    ),
    'incident_silence_time': 4,
    'ping_forsaken_incidents': True,
    'ping_action_items': False,
    'ping_people': True,
    'ping_development': True,
}


def get_config(component):
    return {
        'rm': RM_CONFIG,
        'mdbsupport': MDBSUPPORT_CONFIG,
        'rtcsupport': RTCSUPPORT_CONFIG,
        'bsaudit': BSAUDIT_CONFIG,
        'direct': DIRECT_CONFIG,
    }[component]


@rm_notify.notify(["rm_maintainers"], message=rm_notify.StandardMessage.some_trouble)
class SupportAlert(sdk2.Task):
    """
    Various notification for devops vanguard.

    * List of incidents without "Sotrudniki->Dezhurniy" field set
    * List of incidents that has no actions for too long
    * To assign next guardians of the realm
    """

    class Requirements(task_env.TinyRequirements):
        disk_space = 1000  # 1G
        ram = 4096  # 4G
        environments = [
            rm_task_env.TaskRequirements.startrek_client
        ]
        client_tags = rm_task_env.TaskTags.startrek_client

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.RadioGroup('Component') as component:
            component.values['rm'] = component.Value(value='rm')
            component.values['mdbsupport'] = component.Value(value='mdbsupport')
            component.values['rtcsupport'] = component.Value(value='rtcsupport')
            component.values['bsaudit'] = component.Value(value='bsaudit')
            component.values['direct'] = component.Value(value='direct')

        override_receiver = sdk2.parameters.Bool(
            "Send telegram messages to other person, instead of devops chat (for debugging)",
            default=False,
        )
        with override_receiver.value[True]:
            receiver = sdk2.parameters.String(
                "login of receiver",
                required=True,
                default='lebedev-aa',
            )
        dry_run = sdk2.parameters.Bool(
            "Do not post comments in startrek (for debugging)",
            default=False,
        )
        sleep_in_the_beginning = sdk2.parameters.Bool(
            "Sleep for one minute in the beginning (for debugging)",
            default=False,
        )

    class Context(sdk2.Task.Context):
        forsaken_spi = []
        forsaken_action_items = []
        details_issues = []

    def on_execute(self):
        self.login = None
        self.forbidden_issues = set()

        if self.Parameters.sleep_in_the_beginning:
            time.sleep(60)
        self.config = get_config(self.Parameters.component)

        # Startrek
        if self.config.get('st'):
            if 'secret_id' in self.config['st'] and 'key' in self.config['st']:
                if 'version' in self.config['st']:
                    secret = sdk2.yav.Secret(self.config['st']['secret_id'], self.config['st']['version'])
                else:
                    secret = sdk2.yav.Secret(self.config['st']['secret_id'])
                self.st_token = secret.data()[self.config['st']['key']]
            elif 'token_owner' in self.config['st'] and 'token_name' in self.config['st']:
                self.st_token = sdk2.Vault.data(self.config['st']['token_owner'], self.config['st']['token_name'])
            else:
                raise ValueError('Invalid st config')
            self.st_helper = STHelper(self.st_token, 'SupportAlert Bot')
            self.login = self.st_helper.st_client.myself.login
            if self.config.get('ping_forsaken_incidents'):
                self.ping_forsaken_incidents()
            if self.config.get('ping_action_items'):
                self.ping_action_items()

            pinged_issues = self.Context.forsaken_spi + self.Context.forsaken_action_items + self.Context.details_issues
            self.set_info(
                "Pinged issues: {}".format(
                    ' '.join(
                        '<a href="{0}" target="_blank">{1}</a>'.format(
                            ST.issue_link(issue_key),
                            issue_key,
                        ) for issue_key in pinged_issues
                    )
                ),
                do_escape=False,
            )

    def get_head(self, whom):
        headers = {'Authorization': 'OAuth {}'.format(self.st_token)}
        response = requests.get(
            'https://staff-api.yandex-team.ru/v3/persons',
            headers=headers,
            params={
                'login': whom,
                '_fields': 'chief.login'
            }
        ).json()
        return response['result'][0]['chief']['login']

    def get_head_of_department(self, id):
        headers = {'Authorization': 'OAuth {}'.format(self.st_token)}
        response = requests.get(
            'https://staff-api.yandex-team.ru/v3/groups',
            headers=headers,
            params={
                'id': str(id),
                '_fields': 'department.heads.person.login'
            }
        ).json()
        return response['result'][0]['department']['heads'][0]['person']['login']

    def check_forsaken_ticket(self):
        """
        Find incidents which are not resolved and not updating) for a long time.

        :return message informing about forsaken incidents
        or `None` if no forsaken issues detected.
        """
        silence_time = self.config["incident_silence_time"]
        query, short_url = ST.forsaken_incidents_queries(
            silence_time,
            self.config['st_product'],
            self.config['st_queue'],
            self.config['ignore_tags'],
            self.config.get('forsaken_incidents_extra_filter', '')
        )
        issues = ST.find_tickets_by_query(self.st_helper, query)
        message = None
        if not issues:
            logging.info("There are no forsaken issues, we are awesome!")
            return None
        elif len(issues) > 4:
            message = (
                'У нас {} инцидентов, которые не обновлялись больше {} дней. '
                'Вот первые три: \n{}. <a href="{}">Полный список</a>'
            ).format(
                len(issues),
                silence_time,
                ', '.join([ST.issue_link(found_issue.key) for found_issue in issues[:3]]),
                short_url,
            )
        else:
            message = "Инциденты, которые не обновлялись больше {} дней: \n{}.".format(
                silence_time,
                ', '.join([ST.issue_link(found_issue.key) for found_issue in issues])
            )
        return message

    def check_the_text_for_similarity(self, text, all_comments):
        """Check the last message is from the robot and the text of this message matches the ping text."""
        part_template_comment_text = {
            'text_common': 'В этом инциденте **нет активности** более',
            'text_exists_spproblem': 'Данный тикет является частным случаем проблемы',
            'text_maybe_exists_spproblem': 'Предполагается, что данный тикет является частным случаем проблемы',
            'text_exists_actionitems': 'Для данного инцидента выделено',
            'text_without_resolution': 'В этом инциденте не проставлена резолюция.',
            'text_without_udt_vdt_umbrella': 'Не забудьте расчитать и заполнить поля',
            'text_lsr': 'В этом LSR **нет активности** более',
            'text_actionitems': 'Данная задача поможет избежать',
            'text_without_sre': 'В этом инциденте не выставлены SRE времена.',
            'text_video': 'Привет всем! Прошло много дней после инцидента. Пожалуйста:',
            'text_rtcsupport': 'В этой задаче не было никаких изменений уже',
        }

        if all_comments and all_comments[-1].updatedBy.login == self.login:
            concatenation_text = text + all_comments[-1].text
            if any([
                concatenation_text.count(template_text) == 2
                for template_text in part_template_comment_text.values()
            ]):
                return True
        return False

    def add_followers(self, issue, follower):
        """Add followers to issue."""
        import startrek_client

        logging.info('followers %s', issue.followers)
        for link_follower in issue.followers:
            logging.info(link_follower)
            if follower in link_follower.id:
                logging.info('return %s', link_follower.id)
                return

        if self._forbidden_issue(issue.key, 'Update tags'):
            return

        try:
            self.st_helper.st_client.issues[issue.key].update(
                followers={'add': [follower]},
                params={'notify': False},
            )
        except startrek_client.exceptions.Forbidden as e:
            self._add_forbidden_issue(issue.key, 'Update tags')
        except Exception as e:
            eh.log_exception("Failed to update tags on {}".format(issue.key), e)

    def set_attribute_minusDc(self, issue):
        if self._forbidden_issue(issue.key, 'Set tags'):
            return

        drills_links = [
            linkobject for linkobject in issue.links
            if "DRILLS" in linkobject.object.key
        ]
        if not drills_links:
            return
        try:
            self.st_helper.st_client.issues[issue.key].update(
                minusDc="Да",
                params={'notify': False},
            )
        except startrek_client.exceptions.Forbidden as e:
            self._add_forbidden_issue(issue.key, 'Set tags')
        except Exception as e:
            eh.log_exception("Failed to update tags on {}".format(issue.key), e)

    def set_tag(self, issue, tag):
        """Set tag to issue without notification."""
        import startrek_client

        if self.Parameters.dry_run:
            return

        if tag in issue.tags:
            return

        if self._forbidden_issue(issue.key, 'Set tags'):
            return

        try:
            self.st_helper.st_client.issues[issue.key].update(
                tags={'add': [tag]},
                params={'notify': False},
            )
        except startrek_client.exceptions.Forbidden as e:
            self._add_forbidden_issue(issue.key, 'Set tags')
        except Exception as e:
            eh.log_exception("Failed to update tags on {}".format(issue.key), e)

    def get_author(self, issue):
        try:
            author = issue.createdBy.id
            if not ST.get_person_is_dismissed(issue.createdBy.id, self.st_token):
                logging.debug("get_issue_devops returns: %s", author)
                return [author]
            else:
                return []
        except Exception as e:
            eh.log_exception("failed to get issue devops", e)
            return []

    def get_qaEngineer(self, issue):
        try:
            qaEngineer = issue.qaEngineer.id
            if not ST.get_person_is_dismissed(qaEngineer, self.st_token):
                logging.debug("get_issue_devops returns: %s", qaEngineer)
                return [qaEngineer]
            else:
                return []
        except Exception as e:
            eh.log_exception("failed to get issue devops", e)
            return []

    def get_dutywork_from_abc(self, service, duty_slug):
        url = 'https://abc-back.yandex-team.ru/api/v4/duty/on_duty/?service={}'.format(service)
        headers = {'Authorization': 'OAuth {}'.format(self.st_token)}
        data = requests_wrapper.get(url, headers=headers)
        all_duty = data.json()
        for duty in all_duty:
            if duty['schedule']['slug'] == duty_slug:
                return [duty['person']['login']]
        return []

    def direct_ping(self):
        query = ST.Query(
            queue=self.config['st_queue'],
            created='2021-09-01',
            extra_filter=self.config.get('forsaken_incidents_extra_filter', ''),
            not_closed=False
        ).build_query()
        issues = ST.find_tickets_by_query(self.st_helper, query)
        if not issues:
            logging.info("There are no issues not closed for this time")
        logging.info(issues)
        # for debug
        # issues = [self.st_helper.st_client.issues['UC-2041']]
        # logging.debug(issues)
        issues = [issue for issue in issues if issue.status.key != 'closed']
        logging.info('issues not closed')
        logging.info(issues)
        for issue in issues:
            logging.info(u'issue {}'.format(issue.key))
            head_assignee = []
            head_head = []
            author = self.get_author(issue)
            assignee = ST.get_assignee(issue, self.st_token)
            if assignee:
                head_assignee = ST.get_head_for_person(assignee[0], self.st_token)
            qaEngineer = self.get_qaEngineer(issue)
            logging.info(author)
            logging.info(assignee)
            logging.info(head_assignee)
            status = issue.status.key
            logging.info(status)

            begin_work_time = datetime.time(8, 0, 0)
            end_work_time = datetime.time(17, 0, 0)

            createdAt = datetime.datetime.strptime(issue.createdAt[:-9], "%Y-%m-%dT%H:%M:%S")
            logging.info(u"createdAt {}".format(createdAt))
            created_date = createdAt.date()

            now = datetime.datetime.utcnow()
            logging.info(u"now {}".format(now))
            now_date = now.date()

            sla_hour_res = 0
            if now_date == created_date:
                begin_work_time = datetime.datetime.combine(now_date, begin_work_time)
                end_work_time = datetime.datetime.combine(now_date, end_work_time)
                sla_hour = (now - createdAt).total_seconds() // 3600
                if sla_hour > 8:
                    sla_hour_res = 8
                if createdAt < begin_work_time and now < end_work_time:
                    sla_hour_res = (now - begin_work_time).total_seconds() // 3600
                if createdAt > begin_work_time and createdAt < end_work_time:
                    if now > end_work_time:
                        sla_hour_res = (end_work_time - createdAt).total_seconds() // 3600
                    if now < end_work_time:
                        sla_hour_res = sla_hour
            else:
                begin_work_time_create = datetime.datetime.combine(created_date, begin_work_time)
                end_work_time_create = datetime.datetime.combine(created_date, end_work_time)

                begin_work_time_now = datetime.datetime.combine(now_date, begin_work_time)
                end_work_time_now = datetime.datetime.combine(now_date, end_work_time)

                day_between = (now_date - created_date).days
                sla_hour_res = (day_between - 1)*8

                if createdAt < begin_work_time_create:
                    sla_hour_res += 8
                if createdAt > begin_work_time_create and createdAt < end_work_time_create:
                    sla_hour_res += (end_work_time_create - createdAt).total_seconds() // 3600

                if now > begin_work_time_now and now < end_work_time_now:
                    sla_hour_res += (now - begin_work_time_now).total_seconds() // 3600
                if now > end_work_time_now:
                    sla_hour_res += 8

            logging.info(u"sla_hour_res {}".format(sla_hour_res))

            people = []
            text = ''
            if sla_hour_res > 16:
                text = 'SLA истек'
                people = assignee
                people.extend(head_assignee)
                if 'alkaline' not in people:
                    people.extend(['alkaline'])
                if 'greyevil' not in people:
                    people.extend(['greyevil'])
            else:
                if sla_hour_res > 8 and (not assignee or (status == 'Open' or status == 'needInfo')):
                    text = 'Ожидалось что задача будет в работе. Обратите, пожалуйста, внимание.'
                    people = author
                    people.extend(assignee)
                    people.extend(head_assignee)
                    if 'alkaline' not in people:
                        people.extend(['alkaline'])
                elif status == 'open':
                    if not assignee:
                        text = (
                            'SLA на исправление критичной проблемы в продакшне - 16 рабочих часов.\n'
                            'Пожалуйста, найди задаче исполнителя в нужном зонтике.\n'
                            'Смотри https://wiki.yandex-team.ru/direct/development/zbp/teams/#otvetstvennyepozontam'
                        )
                        people = author
                    else:
                        text = (
                            'Обрати, пожалуйста, внимание на таймер SLA. До его истечения исправление ошибки должно оказаться в продакшне.'
                        )
                        people = assignee
                if status == 'inProgress' and assignee and sla_hour_res > 8:
                    text = (
                        'Осталось 8 рабочих часов до истечения таймера SLA. '
                        'Убедись, что успеешь исправить проблему и довезти исправление до продакшна. Если сомневаешься - зови на помощь руководителя.'
                    )
                    people = assignee
                    people.extend(head_assignee)
                if status == 'readyForTest' and not qaEngineer:
                    text = 'Пожалуйста, найдите задаче QA-инженера.'
                    people = self.get_dutywork_from_abc('169', 'direct-directsup-duty')
                    if 'sudar' not in people:
                        people.extend(['sudar'])
                    if 'sonick' not in people:
                        people.extend(['sonick'])
                if status == 'betaTested':
                    text = (
                        'Убедись что правка успеет выехать в продакшн. '
                        'Скорее всего, тебе понадобится сделать хотфикс в тестируемый релиз или в продакшн.'
                    )
                    people = assignee
            all_comments = list(issue.comments.get_all())
            already_ping = False
            for comment in all_comments:
                if text == comment.text:
                    logging.info('Already ping %s', text)
                    already_ping = True
                    break
            logging.info("people {}".format(people))
            logging.info(u"text {}".format(text))
            if self.Parameters.dry_run:
                continue
            if people and not already_ping:
                ST.ping(
                    self,
                    issue,
                    text,
                    people
                )

    def ping_forsaken_incidents(self):
        silence_time = self.config["incident_silence_time"]
        if self.config['st_queue'] == 'DIRECT, UC':
            logging.info("direct queue")
            self.direct_ping()
            return
        else:
            query, _ = ST.forsaken_incidents_queries(
                silence_time,
                self.config['st_product'],
                self.config['st_queue'],
                self.config['ignore_tags'],
                self.config.get('forsaken_incidents_extra_filter', '')
            )
            issues = ST.find_tickets_by_query(self.st_helper, query)
        # for debug
        # issues = [self.st_helper.st_client.issues['MARTY-3998']]
        if not issues:
            logging.info("There are no issues with empty devops field")
        # limit on the number of pinged LSR tickets to check the functionality, ping 3 old tickets
        elif self.config['st_queue'] == 'LSR' and len(issues) > 2:
            issues = issues[-3:]
        for issue in issues:
            logging.info("status issue %s", issue.status.key)
            if not self.config['ping_development'] and issue.status.key == 'inDevelopment':
                continue
            logging.info('Checking issue `%s` for ping', issue.key)
            if self.config['st_product'] == 'PROD-VIDEO':
                text = self.config['tg_text']
                people = ST.get_issue_devops(issue, self.st_token)
            elif self.config['st_queue'] == 'MDBSUPPORT':
                today = datetime.datetime.today()
                if self.config.get('skip_weekends'):
                    if today.weekday() in [5, 6]:
                        # skip Saturday and Sunday
                        return

                    if today.month == 1 and today.day <= 7:
                        # skip 1-7th of January
                        return
                if today.hour < 10 and today.hour > 20:
                    return
                logging.info('update %s', issue.updatedAt)
                last_ping = datetime.datetime.strptime(issue.updatedAt[:-9], '%Y-%m-%dT%H:%M:%S')
                between = today - last_ping
                between_hours = between.total_seconds() // 3600
                between_hours_not_work = between.days * 14
                between_hour_work = between_hours - between_hours_not_work
                logging.info(
                    'between hour %s not work %s work %s',
                    between_hours, between_hours_not_work, between_hour_work
                )
                text = ''
                if issue.status.key == 'inProgress' and between_hour_work > 20:
                    text = 'Тикет в работе уже больше двух дней. Обновите статус тикета, пожалуйста.'
                if issue.status.key == 'needInfo' and between_hour_work > 40:
                    text = (
                        'В тикете нет движения уже больше 4х дней, '
                        'а информация все еще требуется. Предоставьте информацию, пожалуйста.'
                    )
                people = (ST.get_assignee(issue, self.st_token) + self.config['extra_summonees'])
                people = list(set(people))
                logging.info("status %s text %s people %s", issue.status.key, text, people)
            elif self.config['st_queue'] == 'RTCSUPPORT':
                time_since_last_update = ST.time_since_last_not_ping_update(issue, self.login)
                logging.info("status %s last_update %s", issue.status.key, time_since_last_update)
                text = self.config['tg_text'].format(silence_time=time_since_last_update)
                people = ST.get_assignee(issue, self.st_token)
                logging.info("status %s text %s people %s", issue.status.key, text, people)
            elif self.config['st_queue'] == 'BSAUDIT':
                if ST.get_assignee(issue, self.st_token) is None or 'AuditRelease' not in issue.summary:
                    continue
                importance = ST.count_ping_comments_before(issue, self.login)
                all_people = ST.get_assignee(issue, self.st_token) + [self.get_head(ST.get_assignee(issue, self.st_token)), self.get_head_of_department(BANNER_SYSTEM_DEPARTMENT_ID)]
                text = ''
                people = all_people[:min(3, importance + 1)]
            else:
                action_items = self.get_action_items(issue)
                action_items = self.add_action_items_from_links(issue, action_items)
                spproblem_keys = [
                    linkobject for linkobject in issue.links if linkobject.object.key.startswith("SPPROBLEM-")
                ]
                if self.config['st_queue'] == 'SPI':
                    if ST.is_issue_closed(issue):
                        logging.info('Issue %s closed, but without resolution', issue.key)
                        text = TEXT_WITHOUT_RESOLUTION
                    elif spproblem_keys:
                        logging.info('SPI depends SPProblem %s', spproblem_keys)
                        text = ''
                        for linkobject in spproblem_keys:
                            logging.info('SPI links type %s', linkobject.type.id)
                            if linkobject.type.id == 'depends':
                                text += TEXT_EXISTS_SPPROBLEM.format(spproblem_key=linkobject.object.key)
                            else:
                                text += TEXT_MAYBE_EXISTS_SPPROBLEM.format(
                                    spproblem_key=linkobject.object.key,
                                    spproblem_type_id=linkobject.type.id,
                                )
                    elif action_items:
                        logging.info("count action items links to SPI %s", len(action_items))
                        text = TEXT_EXISTS_ACTION_ITEMS.format(ai_count=str(len(action_items)))
                    else:
                        if silence_time:
                            text = BASE_TG_TEXT_TEMPLATED.format(
                                silence_time=ST.time_since_last_not_ping_update(issue, self.login)
                            )
                    text += self.config['tg_text']
                else:
                    text = self.config['tg_text']
                    if silence_time:
                        text = text.format(
                            silence_time=ST.time_since_last_not_ping_update(issue, self.login)
                        )

                text += 'Выполнено {}'.format(lb.task_wiki_link(self.id, 'SB:{}'.format(self.id)))
                importance = ST.count_ping_comments_before(issue, self.login) + 1
                if issue.key not in self.Context.forsaken_spi:
                    self.Context.forsaken_spi.append(issue.key)
                if self.config.get('ping_people'):
                    people = ST.get_summonees(
                        issue,
                        importance=importance,
                        extra_summonees=self.config.get("extra_summonees", [[], [], []]),
                        token=self.st_token
                    )
                    if self.config['st_queue'] == 'LSR' and 'talion' not in people:
                        people.append('talion')
                else:
                    people = []
            logging.info('ping people forsaken incidents %s', people)
            if self.Parameters.dry_run:
                continue
            all_comments = list(issue.comments.get_all())
            if self.check_the_text_for_similarity(text, all_comments):
                ST.update_comment(self, issue, all_comments[-1].id, text, people)
            else:
                ST.ping(
                    self,
                    issue,
                    text,
                    people
                )

    def get_action_items(self, issue):
        logging.info("Try to find action items for %s", issue.key)
        # MARTY-1978 правки разделителей для AI
        if issue.description:
            items_place = ''
            res_delim = self.find_delimiters(issue.description)
            logging.debug("AI tags in description: %s", res_delim)
            if res_delim:
                items_place = issue.description.split(res_delim[0])[-1].split(res_delim[1])[0].strip()
        else:
            logging.debug("%s empty descripton", issue.key)
            items_place = ''
        # items from description
        items = re.findall("[A-Z]+-[0-9]+", items_place)
        logging.info("AI found in description: %s", items)
        return items

    def add_action_items_from_links(self, issue, items):
        import startrek_client

        for link in issue.links:
            link_id = link.object.key
            logging.info('[add_action_items_from_links] Processing %s for %s', link_id, issue.key)

            if self._forbidden_issue(link_id, 'Get issue'):
                continue

            try:
                link_ticket = self.st_helper.st_client.issues[link_id]
            except startrek_client.exceptions.Forbidden as e:
                self._add_forbidden_issue(link_id, 'Get issue')
                continue

            except Exception as e:
                eh.log_exception("Failed to get issue via {link}".format(link=link_id), e, task=self)
                continue
            link_tags = link_ticket.tags
            for link_tag in link_tags:
                if link_tag.find("spi:actionitem") != -1:
                    logging.info("%s has %s", link_ticket.key, link_tag)
                    if items.count(link_ticket.key) == 0:
                        logging.info("ticket %s with spi:actionitem skipped in descr", link_ticket.key)
                        items.append(link_ticket.key)
                else:
                    if items.count(link_ticket.key):
                        logging.info("ticket %s without spi:actionitem tag", link_ticket.key)
        return items

    def action_item_need_ping(self, item_obj, silence_time, skipped_priority):
        if any([link.object.key.split('-')[0] == "LSR" for link in item_obj.links]):
            logging.info("Skipping %s action_item, because LSR in links", item_obj.key)
            return False
        if item_obj.key.split('-')[0] == self.config['st_queue']:  # incident can't be action item
            logging.info("Skipping %s action item, because incident can't be action item", item_obj.key)
            return False
        if item_obj.key.split('-')[0] == "SPPROBLEM" or item_obj.key.split('-')[0] == "LSR":
            logging.info("Skipping %s action item, because spproblem or lsr can't be action item", item_obj.key)
            return False
        if item_obj.key.split('-')[0] == "YP" or item_obj.key.split('-')[0] == "YTORM" or item_obj.key.split('-')[0] == "YPADMIN":
            logging.info("Skipping %s action item, because YP-, YTORM-, YPADMIN-", item_obj.key)
            return False
        priority = str(item_obj.priority.key)
        if ST.is_issue_closed(item_obj):
            logging.info("Skipping %s action item, because a resolution is not set", item_obj.key)
            return False
        if not ST.is_issue_forsaken(item_obj, silence_time=silence_time):
            logging.info("Skipping %s action item, because action_item_silence_time didn't run out", item_obj.key)
            return False
        if skipped_priority and priority in skipped_priority:
            logging.info(
                "Skipping %s action item, because priority %s is from the skipped_priority",
                item_obj.key, priority,
            )
            return False
        if item_obj.key in self.Context.forsaken_action_items:
            logging.info("Skipping %s action item, because already pinged", item_obj.key)
            return False
        if ST.is_deadline_ok(item_obj) or ST.is_sprint_ok(item_obj):
            logging.info("Skipping %s action item, because deadline or sprint is ok", item_obj.key)
            return False
        # DEVOPS-625
        if item_obj.key.startswith("NOCLSR") and item_obj.status.key == "resolved":
            logging.info("Skipping %s action item, because queue=NOCLSR and status is resolved", item_obj.key)
            return False
        return True

    def ping_action_items(self):
        logging.info("Ping action items started")
        silence_time = self.config["action_item_silence_time"]
        skipped_priority = self.config["action_item_skipped_priority"]
        # query = ST.prod_incidents_query(self.config['st_product'], self.config['st_queue'])
        query = ST.Query(
            product=self.config['st_product'],
            not_closed=False,
            queue=self.config['st_queue'],
            tags_to_ignore=self.config['ignore_tags'],
            created="2021-04-01"
        ).build_query()
        # if we need to read this, we have a problem in code, so log only in debug mode
        logging.debug("Incidents query: %s", query)
        incidents = ST.find_tickets_by_query(self.st_helper, query)
        # incidents = [self.st_helper.st_client.issues['SPI-20756']]
        for incident in incidents:
            self.set_attribute_minusDc(incident)
            action_items = self.get_action_items(incident)
            ai_link = [link.object.key for link in incident.links if link.type.inward == "Is subtask for"]
            logging.info(u"Подзадача %s", ai_link)
            action_items.extend(ai_link)
            logging.info('get action items %s', action_items)
            # DEVOPS-311 spi:actionitem tags from description
            for action_item in action_items:
                try:
                    item_obj = self.st_helper.st_client.issues[action_item]
                except Exception as e:
                    eh.log_exception("Failed to get action item from ST. Probably too simple regexp.", e, task=self)
                    continue
                if item_obj.key.startswith(IGNORE_QUEUE) or item_obj.key.startswith('SPI'):
                    logging.info("item %s filtered, because matched incidents or ignore queue", item_obj.key)
                    continue
                # DEVOPS-625
                if (
                    'spi:actionitem' not in item_obj.tags
                    and not (
                        ST.is_issue_closed(item_obj)
                        or (
                            item_obj.key.startswith("NOCLSR-")
                            and item_obj.status.key == "resolved"
                        )
                        or (
                            item_obj.key.startswith("YDODUTY-")
                            and 'duty:actionitem' not in item_obj.tags
                        )
                    )
                ):
                    text = 'Данный тикет назначен AI по инциденту {}'.format(incident.key)
                    logging.info('AI ping %s %s', item_obj, text)
                    people = []
                    ST.ping(
                        self,
                        item_obj,
                        'Данный тикет назначен AI по инциденту {}'.format(incident.key),
                        people
                    )

                self.set_tag(item_obj, 'spi:actionitem')

            #  MARTY-1978  items from links
            action_items = self.add_action_items_from_links(incident, action_items)
            logging.info("Action items found: %s", action_items)
            for action_item in action_items:
                if action_item.startswith(IGNORE_QUEUE):
                    logging.info("item %s filtered, because matched incidents queue", action_item)
                    continue

                logging.info('Checking action item %s', action_item)
                try:
                    item_obj = self.st_helper.st_client.issues[action_item]
                except Exception as e:
                    eh.log_exception("Failed to get action item from ST. Probably too simple regexp.", e, task=self)
                    continue
                if not self.action_item_need_ping(item_obj, silence_time, skipped_priority):
                    continue
                self.Context.forsaken_action_items.append(item_obj.key)
                importance = ST.count_ping_comments_before(item_obj, self.login) + 1
                people = ST.get_summonees(
                    item_obj,
                    importance=importance,
                    extra_summonees=self.config.get("extra_summonees", [[], [], []]),
                    token=self.st_token,
                )
                human_silence_time = ST.time_since_last_not_ping_update(item_obj, self.login)
                text = ACTION_ITEM_PING_TEMPLATE.format(
                    incident=incident.key,
                    silence_time=human_silence_time,
                    sandbox_task_link=lb.task_wiki_link(self.id, self.id)
                )
                all_comments = list(item_obj.comments.get_all())
                if self.check_the_text_for_similarity(text, all_comments):
                    ST.update_comment(self, item_obj, all_comments[-1].id, text, people)
                else:
                    ST.ping(
                        self,
                        item_obj,
                        text,
                        people
                    )

    @staticmethod
    def find_delimiters(descr):
        """
        Find delimiters in description.

        :param descr: ticket description (string)
        :return: list of delimiters in description
        """
        kword = []
        for delim in ACTION_ITEM_DELIMS:
            logging.debug('Checking delimiters {delimiters}'.format(delimiters=delim))
            delimiters = (
                re.findall(ur"[=]+[\s]?{}[\s]?[=]*".format(delim[0]), descr),
                re.findall(ur"[=]+[\s]?{}[\s]?[=]*".format(delim[1]), descr),
            )
            if delimiters[0] and delimiters[1]:
                kword = (delimiters[0][-1].strip(), delimiters[1][-1].strip())
        return kword

    def _add_forbidden_issue(self, key, action):
        logging.error('No access to `%s`: %s failed', key, action)
        self.set_info('No access to `{}`: {} failed'.format(key, action))
        self.forbidden_issues.add(key)

    def _forbidden_issue(self, key, action):
        if key in self.forbidden_issues:
            logging.error('No access to `%s`: %s failed [cached]', key, action)
            return True
        return False


class Schedule(object):

    @staticmethod
    def get_schedule(arcadia_schedule_url, checkouted=None):
        checkouted = checkouted if checkouted else []
        if not checkouted:
            sdk2.svn.Arcadia.checkout(arcadia_schedule_url, 'schedule_data')
            checkouted.append(True)
        with open('schedule_data/websearch_schedule.json', 'r') as schedule_file:
            result = json.load(schedule_file)
        return result

    @staticmethod
    def get_current_duty_dict(arcadia_schedule_url):
        for week in reversed(Schedule.get_schedule(arcadia_schedule_url)):
            year, month, day = map(int, week['week'].split('.'))
            duty_start = datetime.datetime(year, month, day, 17)
            if duty_start > datetime.datetime.now():
                # They are future devops who are scheduled but not started yet, skip them
                continue
            return week

    @staticmethod
    def get_last_duty_dict(arcadia_schedule_url):
        return Schedule.get_schedule(arcadia_schedule_url)[-1]
