# -*- coding: utf-8 -*
import os
import json
import logging
import requests

from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import decorators
from sandbox.projects.common import task_env
from sandbox.projects.release_machine import rm_notify


class Method(object):
    CREATE = 'create'
    UPDATE = 'update'
    UPSERT = 'upsert'
    REPLACE = 'replace'


class ObjectType(object):
    PANEL = 'panels'
    ALERT = 'alerts'


_OBJECT_NOT_FOUND = 'OBJECT_NOT_FOUND'


class TaskNotFinished(Exception):
    pass


class FailedToRunTask(Exception):
    pass


def _j(*args):
    return os.path.join(*args)


_DRIVE = 'drive/devops'
_DRIVE_ALERTS = _j(_DRIVE, 'alerts')
_DRIVE_PANELS = _j(_DRIVE, 'panels')

_IP = 'search/tools/iconostasis_panels'
_IPI = 'search/tools/iconostasis_panels/included_panels'
_WP = 'search/tools/devops/yasm_web_panels'
_RM = 'release_machine/tools/monitoring'
_SD = 'infra/yp_service_discovery/monitoring/yasm'
_BDC = 'infra/box_dns_controller/monitoring/yasm'

_RM_ALERTS = _j(_RM, 'alerts')
_RM_PANELS = _j(_RM, 'panels')
_L7_BALANCER_ALERTS_DIR = 'balancer/production/l7_alerts'
_L7_BALANCER_PANELS_DIR = 'balancer/production/yasm_panels'

# PLEASE ADD YOUR TEMPLATE ALERTS HERE
_TEMPLATE_ALERTS = {
    'accesslog': _j(_IP, 'accesslog'),
    'latency': _j(_IP, 'latency'),

    'marty_balancer_backend_fails': _j(_IP, 'marty_balancer_backend_fails'),

    # box_dns_controller
    'box_dns_controller': _j(_BDC, 'box_dns_controller_alerts.jinja'),

    # service_discovery
    'service_discovery': _j(_SD, 'service_discovery_alerts.jinja'),
    'service_discovery_alerts_logic': _j(_SD, 'service_discovery_alerts_logic.jinja'),
    'service_discovery_base_alerts': _j(_SD, 'service_discovery_base_alerts.inc.jinja'),

    # various upper services templates
    'web_sandwich': 'web/report/yasm/web_sandwich_alerts.jinja',
    'alice_upper': 'web/report/yasm/alice_alerts.jinja',
    'upper_alerts_logic': 'web/report/yasm/upper_alerts_logic.jinja',
    'upper_base_alerts': 'web/report/yasm/upper_base_alerts.inc.jinja',

    'wizards': _j(_IP, 'wizards.jinja2'),

    'apphost_critical_deps': _j(_IP, 'apphost_critical_deps'),
    'apphost_alerts_logic': _j(_IP, 'apphost_alerts_logic'),
    'apphost_alerts': 'apphost/tools/jinja_templates/apphost_template_alerts.jinja2',

    'balancer.yastatic.alerts': _j(_IP, 'balancer.yastatic.alerts'),
    'images_alerts': 'extsearch/images/tools/quentao/images_alerts',
    'l7_to_pumpkin_traffic_percent': _j(_IP, 'l7_to_pumpkin_traffic_percent'),

    'web_prefetch': _j(_IP, 'web_prefetch/alerts'),
    'web_runtime': 'search/tools/iconostasis_panels/web_runtime.jinja',

    'upper_crashes': 'search/tools/iconostasis_panels/upper_crashes_alerts',

    # drive
    'drive_b2b_taxi': _j(_DRIVE_ALERTS, 'b2b_taxi.jinja2'),
    'drive_chat': _j(_DRIVE_ALERTS, 'chats.jinja2'),
    'drivematics_prod_db': _j(_DRIVE_ALERTS, 'drivematics_database.jinja2'),
    'drive_external': _j(_DRIVE_ALERTS, 'external.jinja2'),
    'drive_health': _j(_DRIVE_ALERTS, 'health.jinja2'),
    'drive_robots': _j(_DRIVE_ALERTS, 'robots.jinja2'),
    'drive_service_routing': _j(_DRIVE_ALERTS, 'service_routing.jinja2'),
    'drive_telematics': _j(_DRIVE_ALERTS, 'telematics.jinja2'),
    'drive_user_app': _j(_DRIVE_ALERTS, 'user_app.jinja2'),
    'drivematics': _j(_DRIVE_ALERTS, 'drivematics.jinja2'),
    'robot_exports': _j(_DRIVE_ALERTS, 'robot_exports.jinja2'),
    'robot_registration': _j(_DRIVE_ALERTS, 'robot_registration.jinja2'),
    'robot_fines': _j(_DRIVE_ALERTS, 'robot_fines.jinja2'),

    # release_machine
    'rm_yp_ram_usage': _j(_RM_ALERTS, 'rm_yp_ram_usage.jinja2'),
    'rm_event_post_proto_events_alert': _j(_RM_ALERTS, 'rm_event_post_proto_events_alert.jinja2'),
    'rm_event_post_proto_events_error': _j(_RM_ALERTS, 'rm_event_post_proto_events_error.jinja2'),
    'rm_model_check_user_permissions_prerelease_alert':
        _j(_RM_ALERTS, 'rm_model_check_user_permissions_prerelease_alert.jinja2'),
    'rm_model_check_user_permissions_prerelease_error':
        _j(_RM_ALERTS, 'rm_model_check_user_permissions_prerelease_error.jinja2'),
    'rm_model_check_user_permissions_release_alert':
        _j(_RM_ALERTS, 'rm_model_check_user_permissions_release_alert.jinja2'),
    'rm_model_check_user_permissions_release_error':
        _j(_RM_ALERTS, 'rm_model_check_user_permissions_release_error.jinja2'),
    'rm_model_get_changelog_alert': _j(_RM_ALERTS, 'rm_model_get_changelog_alert.jinja2'),
    'rm_model_get_changelog_error': _j(_RM_ALERTS, 'rm_model_get_changelog_error.jinja2'),
    'rm_model_get_component_alert': _j(_RM_ALERTS, 'rm_model_get_component_alert.jinja2'),
    'rm_model_get_component_chats_alert': _j(_RM_ALERTS, 'rm_model_get_component_chats_alert.jinja2'),
    'rm_model_get_component_chats_error': _j(_RM_ALERTS, 'rm_model_get_component_chats_error.jinja2'),
    'rm_model_get_component_error': _j(_RM_ALERTS, 'rm_model_get_component_error.jinja2'),
    'rm_model_get_components_alert': _j(_RM_ALERTS, 'rm_model_get_components_alert.jinja2'),
    'rm_model_get_components_error': _j(_RM_ALERTS, 'rm_model_get_components_error.jinja2'),
    'rm_model_get_events_alert': _j(_RM_ALERTS, 'rm_model_get_events_alert.jinja2'),
    'rm_model_get_events_error': _j(_RM_ALERTS, 'rm_model_get_events_error.jinja2'),
    'rm_model_get_notification_config_alert': _j(_RM_ALERTS, 'rm_model_get_notification_config_alert.jinja2'),
    'rm_model_get_notification_config_error': _j(_RM_ALERTS, 'rm_model_get_notification_config_error.jinja2'),
    'rm_model_get_preferences_alert': _j(_RM_ALERTS, 'rm_model_get_preferences_alert.jinja2'),
    'rm_model_get_preferences_error': _j(_RM_ALERTS, 'rm_model_get_preferences_error.jinja2'),
    'rm_model_get_scopes_alert': _j(_RM_ALERTS, 'rm_model_get_scopes_alert.jinja2'),
    'rm_model_get_scopes_error': _j(_RM_ALERTS, 'rm_model_get_scopes_error.jinja2'),
    'rm_model_get_state_alert': _j(_RM_ALERTS, 'rm_model_get_state_alert.jinja2'),
    'rm_model_get_state_error': _j(_RM_ALERTS, 'rm_model_get_state_error.jinja2'),
    'rm_model_ok_to_release_alert': _j(_RM_ALERTS, 'rm_model_ok_to_release_alert.jinja2'),
    'rm_model_ok_to_release_error': _j(_RM_ALERTS, 'rm_model_ok_to_release_error.jinja2'),
    'rm_model_update_component_statistics_alert': _j(_RM_ALERTS, 'rm_model_update_component_statistics_alert.jinja2'),
    'rm_model_update_component_statistics_error': _j(_RM_ALERTS, 'rm_model_update_component_statistics_error.jinja2'),

    # experiments
    'uaas': 'quality/ab_testing/exp_daemon/monitoring/uaas_alerts.jinja2',
    'uaas2': 'quality/ab_testing/exp_daemon/monitoring/uaas2_alerts.jinja2',
    'ems': 'quality/ab_testing/scripts/ems/ems/monitoring/ems_alerts.jinja2',

    # morda
    'portal.morda.sources': 'portal/devops/monitoring/yasm/alert_templates_v2/auto_update/portal.morda.sources',
    'portal.morda.benchtimes': 'portal/devops/monitoring/yasm/alert_templates_v2/auto_update/portal.morda.benchtimes',
    'portal.morda.common.template': 'portal/devops/monitoring/yasm/alert_templates_v2/auto_update/portal.morda.common.template',
}

# PLEASE ADD YOUR TEMPLATE PANELS HERE
_TEMPLATE_PANELS = {
    # common search
    'search_load': _j(_IP, 'search_load'),
    'search_share_alerts': _j(_IP, 'search_share_alerts'),

    # balancer
    'yastatic_balancer': _j(_IP, 'yastatic_balancer'),
    'yandex_tld': _j(_IP, 'yandex_tld'),
    'yandex_tld_5xx': _j(_IP, 'yandex_tld_5xx'),
    'yandex_tld_fails': _j(_IP, 'yandex_tld_fails'),
    'yandex_tld_rps_and_fails': _j(_IP, 'yandex_tld_rps_and_fails'),
    'yandex_tld_fast_rps_and_fails': _j(_IP, 'yandex_tld_fast_rps_and_fails'),
    'balancer_load_panel': _j(_L7_BALANCER_PANELS_DIR, 'balancer_load_panel.jinja2'),

    # apphost
    'apphost_critical_deps_panel': _j(_IP, 'apphost_critical_deps_panel'),
    'apphost_duty_alerts': 'apphost/tools/jinja_templates/apphost_duty_alerts.jinja2',

    # web
    'sdch_panel': 'search/sdch/yasm-panel.jinja2',
    'web': _j(_IP, 'web'),
    'web_sandwich_iconostasis': 'web/report/yasm/web_sandwich_iconostasis.jinja',

    # some shared services
    'shared_services_iconostasis': 'web/report/yasm/shared_services_iconostasis.jinja',

    # upper search services crashes
    'upper_crashes': 'search/tools/iconostasis_panels/upper_crashes_panel',

    # release_machine
    'rm_panel': _j(_RM_PANELS, 'rm_panel.jinja2'),
    'rm_urls': _j(_RM_PANELS, 'rm_urls.jinja2'),

    # coredumps
    'coredumps': 'infra/cores/yasm/coredumps.jinja2',

    # multipanel is a panel with alerts from different services and even different verticals
    'multipanel': _j(_IP, 'multipanel'),
    'marty_ymsupport_content': _j(_IPI, 'marty_ymsupport_content'),
    'marty_suggest_content': _j(_IPI, 'marty_suggest_content'),
    'marty_q_content': _j(_IPI, 'marty_q_content'),
    'marty_weather_content': _j(_IPI, 'marty_weather_content'),
    'marty_ugc_content': _j(_IPI, 'marty_ugc_content'),
    'marty_entitysearch_content': _j(_IPI, 'marty_entitysearch_content'),
    'marty_ether_content': _j(_IPI, 'marty_ether_content'),
    'marty_front_maps_content': _j(_IPI, 'marty_front_maps_content'),
    'marty_smarttv_content': _j(_IPI, 'marty_smarttv_content'),
    'marty_geosuggest_content': _j(_IPI, 'marty_geosuggest_content'),
    'marty_zora_content': _j(_IPI, 'marty_zora_content'),
    'marty_ocsp_content': _j(_IPI, 'marty_ocsp_content'),
    'marty_infra_content': _j(_IPI, 'marty_infra_content'),
    'marty_horizon_content': _j(_IPI, 'marty_horizon_content'),
    'marty_telemost_content': _j(_IPI, 'marty_telemost_content'),
    'marty_favicon_content': _j(_IPI, 'marty_favicon_content'),
    'marty_lpc_content': _j(_IPI, 'marty_lpc_content'),

    # images
    'images_lights': 'extsearch/images/tools/quentao/images_lights',
    'images': _j(_IP, 'images'),

    # drive
    'DriveBackendServiceApp': _j(_DRIVE_PANELS, 'service_app.jinja2'),
    'DriveBilling': _j(_DRIVE_PANELS, 'billing.jinja2'),
    'DriveBillingNew': _j(_DRIVE_PANELS, 'billing.jinja2'),
    'DriveBillingWoW': _j(_DRIVE_PANELS, 'billing_wow.jinja2'),
    'DriveChatAdminHandles': _j(_DRIVE_PANELS, 'chat_admin.jinja2'),
    'DriveChatClientHandles': _j(_DRIVE_PANELS, 'chat_user_app.jinja2'),
    'DriveChatRegistrationHandles': _j(_DRIVE_PANELS, 'chat_registration.jinja2'),
    'DriveExternalApi': _j(_DRIVE_PANELS, 'external_api.jinja2'),
    'DriveFreshness': _j(_DRIVE_PANELS, 'freshness.jinja2'),
    'DriveFrontendAdminHandles': _j(_DRIVE_PANELS, 'admin.jinja2'),
    'DriveFrontendExternalHandles': _j(_DRIVE_PANELS, 'external.jinja2'),
    'DriveFrontendHandles': _j(_DRIVE_PANELS, 'user_app.jinja2'),
    'DriveHandlersInProgress': _j(_DRIVE_PANELS, 'handlers_in_progress.jinja2'),
    'DriveServerHealth': _j(_DRIVE_PANELS, 'health.jinja2'),
    'DriveServiceRouting': _j(_DRIVE_PANELS, 'service_routing.jinja2'),
    'DriveTelematicsConnections': _j(_DRIVE_PANELS, 'telematics_connections.jinja2'),
    'DriveTelematicsServerHealth': _j(_DRIVE_PANELS, 'telematics_health.jinja2'),
    'DriveTrust': _j(_DRIVE_PANELS, 'trust.jinja2'),
    'billing_WoW': _j(_DRIVE_PANELS, 'billing_wow.jinja2'),
    'DriveFueling': _j(_DRIVE_PANELS, 'fueling.jinja2'),
    'DriveFines': _j(_DRIVE_PANELS, 'fine.jinja2'),
    'DrivePDDFines': _j(_DRIVE_PANELS, 'fine_pdd.jinja2'),
    'DrivematicsHandles': _j(_DRIVE_PANELS, 'drivematics_handles.jinja2'),
    'DriveLPM': _j(_DRIVE_PANELS, 'lpm_user_updater.jinja2'),

    # YP
    'yp_alerts': _j(_IP, 'yp_alerts'),

    # uaas
    'uaas': 'quality/ab_testing/exp_daemon/monitoring/uaas_panel.jinja2',
    'uaas2': 'quality/ab_testing/exp_daemon/monitoring/uaas2_panel.jinja2',

    # marty
    'flow_tools_lights': _j(_IP, 'flow_tools_lights'),

    # L7 Balancer
    'L7_self': _j(_L7_BALANCER_ALERTS_DIR, 'L7_self.jinja'),
    'L7_sd_test': _j(_L7_BALANCER_ALERTS_DIR, 'L7_sd_test.jinja'),

    # morda
    'morda_subreqname': 'portal/devops/monitoring/yasm/jinja2/auto_update/morda_morda_subreqname.j2',
    'morda_bench_all': 'portal/devops/monitoring/yasm/jinja2/auto_update/morda_bench_all.j2',
}

_PANELS = {
    'web_prefetch': _j(_IP, 'web_prefetch/panel'),
    'web-hamster-lights': _j(_IP, 'web-hamster-lights'),
    'xml_health_web': 'search/daemons/noapacheupper/yasm/xml_health_web.json',
    'search-xml': _j(_IP, 'search-xml'),
    'sitesearch-l7': _j(_IP, 'sitesearch-l7'),
    'explogdaemon': _j(_IP, 'explogdaemon.json'),  # SEARCH-10671
    'wizard_sources_alerts': _j(_IP, 'wizard_sources_alerts.json'),
    'cores_monitoring': 'infra/cores/yasm/cores_monitoring.json',
}

_HAMSTER_ALERTS_DIR = _j(_IP, 'hamster_alerts')

_ALERTS = {
    'hamster_web5b_failures_percent': _j(_HAMSTER_ALERTS_DIR, 'hamster_web5b_failures_percent.json'),
    'hamster_web_middle_failures_percent': _j(_HAMSTER_ALERTS_DIR, 'hamster_web_middle_failures_percent.json'),
    'hamster_mmeta_crash': _j(_HAMSTER_ALERTS_DIR, 'hamster_mmeta_crash.json'),
    'hamster_sinsig_dcg_5_diff_prod': _j(_HAMSTER_ALERTS_DIR, 'hamster_sinsig_dcg_5_diff_prod.json'),
    'hamster_marty_monitoring_serp_failed': _j(_HAMSTER_ALERTS_DIR, 'hamster_marty_monitoring_serp_failed.json'),
    'hamster_apphost_begemot_failures_percent':
        _j(_HAMSTER_ALERTS_DIR, 'hamster_apphost_begemot_failures_percent.json'),
    'hamster-mmeta-platinum-unanswers': _j(_HAMSTER_ALERTS_DIR, 'hamster-mmeta-platinum-unanswers.json'),
    'hamster-apphost-quota-usage': _j(_HAMSTER_ALERTS_DIR, 'hamster-apphost-quota-usage.json'),
    'hamster-int_l1-ooms': _j(_HAMSTER_ALERTS_DIR, 'hamster-int_l1-ooms.json'),
    'hamster-market-alert': _j(_HAMSTER_ALERTS_DIR, 'hamster-market-alert.json'),
    'hamster-mmeta-banfilter-unanswers': _j(_HAMSTER_ALERTS_DIR, 'hamster-mmeta-banfilter-unanswers.json'),
    'hamster-mmeta-callisto-unanswers': _j(_HAMSTER_ALERTS_DIR, 'hamster-mmeta-callisto-unanswers.json'),
    'hamster-mmeta-samohod-unanswers': _j(_HAMSTER_ALERTS_DIR, 'hamster-mmeta-samohod-unanswers.json'),
    'hamster-mmeta-web-unanswers': _j(_HAMSTER_ALERTS_DIR, 'hamster-mmeta-web-unanswers.json'),
    'hamster-ranking_mid-saas-unanswer-percent-alert':
        _j(_HAMSTER_ALERTS_DIR, 'hamster-ranking_mid-saas-unanswer-percent-alert.json'),
    'hamster_apphost_blender_failures_percent':
        _j(_HAMSTER_ALERTS_DIR, 'hamster_apphost_blender_failures_percent.json'),
    'hamster_geov': _j(_HAMSTER_ALERTS_DIR, 'hamster_geov.json'),
    'hamster_quality_age': _j(_HAMSTER_ALERTS_DIR, 'hamster_quality_age.json'),
    'hamster_serpdata_saas_failures_percent': _j(_HAMSTER_ALERTS_DIR, 'hamster_serpdata_saas_failures_percent.json'),
    'hamster_quick_unanswers_percent': _j(_HAMSTER_ALERTS_DIR, 'hamster_quick_unanswers_percent.json'),
    'hamster-monitoring-images-cbir-age': _j(_HAMSTER_ALERTS_DIR, 'hamster-monitoring-images-cbir-age.json'),
    'hamster-monitoring-images-cbir-serp-failed':
        _j(_HAMSTER_ALERTS_DIR, 'hamster-monitoring-images-cbir-serp-failed.json'),
    'hamster-monitoring-images-desktop-age':
        _j(_HAMSTER_ALERTS_DIR, 'hamster-monitoring-images-desktop-age.json'),
    'hamster-monitoring-images-desktop-serp-failed':
        _j(_HAMSTER_ALERTS_DIR, 'hamster-monitoring-images-desktop-serp-failed.json'),
    'hamster-monitoring-video-desktop-age':
        _j(_HAMSTER_ALERTS_DIR, 'hamster-monitoring-video-desktop-age.json'),
    'hamster-monitoring-video-desktop-serp-failed':
        _j(_HAMSTER_ALERTS_DIR, 'hamster-monitoring-video-desktop-serp-failed.json'),
    'hamster-monitoring-video-touch-age':
        _j(_HAMSTER_ALERTS_DIR, 'hamster-monitoring-video-touch-age.json'),
    'hamster-monitoring-video-touch-serp-failed':
        _j(_HAMSTER_ALERTS_DIR, 'hamster-monitoring-video-touch-serp-failed.json'),
    'hamster-snippet-unanswer-platinum':
        _j(_HAMSTER_ALERTS_DIR, 'hamster-snippet-unanswer-platinum.json'),
    'hamster-snippet-unanswer-samohod':
        _j(_HAMSTER_ALERTS_DIR, 'hamster-snippet-unanswer-samohod.json'),
    'hamster-snippet-unanswer-web':
        _j(_HAMSTER_ALERTS_DIR, 'hamster-snippet-unanswer-web.json'),
    'hamster-snippet-unanswer-webfresh':
        _j(_HAMSTER_ALERTS_DIR, 'hamster-snippet-unanswer-webfresh.json'),

    # DEVOPS-507
    'empty_serps_numdocs_popular_sas': _j(_IP, 'empty_serps_numdocs_popular_sas.json'),
    'empty_serps_numdocs_popular_man': _j(_IP, 'empty_serps_numdocs_popular_man.json'),
    'empty_serps_numdocs_popular_vla': _j(_IP, 'empty_serps_numdocs_popular_vla.json'),

    'sas-web-prestable-error-21': _j(_IP, 'sandwich/sas-web-prestable-error-21.json'),
    'sas-web-prod-error-21': _j(_IP, 'sandwich/sas-web-prod-error-21.json'),
    'man-web-prod-error-21': _j(_IP, 'sandwich/man-web-prod-error-21.json'),
    'vla-web-prod-error-21': _j(_IP, 'sandwich/vla-web-prod-error-21.json'),

    # Search load (SEARCH-9918 etc)
    'search_load_web': _j(_IP, 'search_load_web.json'),

    # Basesearch
    'SAS_E_UNANSWERS': _j(_WP, 'SAS_E_UNANSWERS.json'),
    'MAN_E_UNANSWERS': _j(_WP, 'MAN_E_UNANSWERS.json'),
    'VLA_E_UNANSWERS': _j(_WP, 'VLA_E_UNANSWERS.json'),

    # release_machine
    'release_engine_prod_event_crawler_delay': _j(_RM_ALERTS, 'release_engine_prod_event_crawler_delay.jinja2'),

    # logos
    "logos.common.5xx": "logos/web/tools/alerts/logos.common.5xx.json",
}


_ALERTS_SERIES = {
    # L7 Balancer
    'l7_alerts': _j(_L7_BALANCER_ALERTS_DIR, 'L7_alerts.json'),
    # 'L7_yastatic_alerts': _j(_L7_BALANCER_ALERTS_DIR, 'L7_yastatic_alerts.json'),
    'L7_sd_test_alerts': _j(_L7_BALANCER_ALERTS_DIR, 'L7_sd_test_alerts.json'),
}


@rm_notify.notify2(people=["mvel"])
class UpdateMartyPanels(sdk2.Task):
    """Update all known panels used by Marty (to prevent disasters like SPI-8961)."""

    class Parameters(sdk2.task.Parameters):
        kill_timeout = 15 * 60  # 15 min

    class Requirements(task_env.TinyRequirements):
        pass

    def get_alert_comment(self, template_source_file):
        result = [
            '<# ################################################################################################ #',
            'This template is provided by sandbox task {task_id} from file {template_source_file}.'.format(
                task_id=self.id,
                template_source_file=template_source_file,
            ),
            'All changes will be removed on next task start. See {wiki_doc_page}'.format(
                wiki_doc_page='https://wiki.yandex-team.ru/devops/updatemartypanels/',
            ),
            ' # ################################################################################################ #>'
        ]
        return '\n'.join(result)

    class Context(sdk2.Task.Context):
        failed_keys = []

    def on_execute(self):
        for template_key, arcadia_path in _TEMPLATE_ALERTS.iteritems():
            try:
                contents = self._read_arcadia_file(arcadia_path)
                contents = '{}\n{}'.format(self.get_alert_comment(arcadia_path), contents)
                self._update_alert_template(template_key, contents)
            except Exception as e:
                self.Context.failed_keys.append(template_key)
                self.set_info('Cannot update/insert template alert `{}`'.format(template_key))
                eh.log_exception('Cannot update template alert', e, task=self)

        for template_key, arcadia_path in _TEMPLATE_PANELS.iteritems():
            try:
                contents = self._read_arcadia_file(arcadia_path)
                contents = '{}\n{}'.format(self.get_alert_comment(arcadia_path), contents)
                self._update_panel_template(template_key, contents)
            except Exception as e:
                self.Context.failed_keys.append(template_key)
                self.set_info('Cannot update/insert template panel `{}`'.format(template_key))
                eh.log_exception('Cannot update template panel', e, task=self)

        for key, arcadia_path in _ALERTS.iteritems():
            try:
                contents = self._read_arcadia_file(arcadia_path)
                result = self._update_alert(key, contents)
                if result == _OBJECT_NOT_FOUND:
                    self._create_alert(key, contents)
            except Exception as e:
                self.Context.failed_keys.append(key)
                self.set_info('Cannot update/insert non-template alert `{}`'.format(key))
                eh.log_exception('Cannot update alert', e, task=self)

        for key, arcadia_path in _ALERTS_SERIES.iteritems():
            try:
                contents = self._read_arcadia_file(arcadia_path)
                result = self._replace_alerts(key, contents)
            except Exception as e:
                self.Context.failed_keys.append(key)
                self.set_info('Cannot replace alerts from `{}`'.format(key))
                eh.log_exception('Cannot update alert', e, task=self)

        for key, arcadia_path in _PANELS.iteritems():
            try:
                contents = self._read_arcadia_file(arcadia_path)
                result = self.upsert(key, contents)
            except Exception as e:
                self.Context.failed_keys.append(key)
                self.set_info('Cannot update/insert non-template panel `{}`'.format(key))
                eh.log_exception('Cannot update panel', e, task=self)

        self.Context.failed_keys = sorted(list(set(self.Context.failed_keys)))

        if self.Context.failed_keys:
            eh.check_failed('Cannot update some panels or alerts, see task info/logs for details')

    def _read_arcadia_file(self, arcadia_path):
        """
        Read file from Arcadia repo.

        :param arcadia_path: path related to arcadia root, e.g. `search/tools/devops/whatever`
        :return file contents
        """
        checkout_path = Arcadia.export(
            'arcadia:/arc/trunk/arcadia/{}'.format(arcadia_path), os.path.basename(arcadia_path)
        )
        return fu.read_file(checkout_path)

    def _create_alert_template(self, template_key, content):
        self._operate_template(
            method=Method.CREATE,
            template_type=ObjectType.ALERT,
            template_key=template_key,
            content=content,
        )

    def _create_panel_template(self, template_key, content):
        self._operate_template(
            method=Method.CREATE,
            template_type=ObjectType.PANEL,
            template_key=template_key,
            content=content,
        )

    def _update_alert_template(self, template_key, content):
        result = self._operate_template(
            method=Method.UPDATE,
            template_type=ObjectType.ALERT,
            template_key=template_key,
            content=content,
        )
        if result == _OBJECT_NOT_FOUND:
            self._create_alert_template(template_key, content)

    def _update_panel_template(self, template_key, content):
        result = self._operate_template(
            method=Method.UPDATE,
            template_type=ObjectType.PANEL,
            template_key=template_key,
            content=content,
        )
        if result == _OBJECT_NOT_FOUND:
            self._create_panel_template(template_key, content)

    @decorators.retries(5, delay=2, backoff=2)
    def _upsert(self, key, update_url, update_data):
        update_response = requests.post(
            update_url,
            json=update_data,
        )
        return self._check_error(
            key=key,
            method=Method.UPSERT,
            response=update_response,
            check_non_existent=False,
        )

    def upsert(self, key, content):
        """
        Update existing or insert new panel. Named as combo of `update/insert`.

        :param key: panel/alert identifier (please do not mix with alias)
        :param content: panel/alert text (contents)
        """
        update_url = 'https://yasm.yandex-team.ru/srvambry/upsert'
        update_data = {
            'keys': {
                'user': 'robot-srch-releaser',
                'key': key,
            },
            'values': json.loads(content),
        }
        logging.info('Trying to update panel `%s`', key)
        return self._upsert(key, update_url, update_data)

    def _check_error(self, key, method, response, check_non_existent=True):
        """
        Check YASM response for errors and show them in task info.

        - Check YASM response.
        - Print some diag into task info
        - Store problematic key in task context.

        :param key: panel/alert identifier (please do not mix with alias)
        :param method: one of Method item (e.g. insert/update)
        :param response: valid `requests` module response object.
        :return `_OBJECT_NOT_FOUND` or response object or raise Exception on 5xx
        """
        if response is None:
            raise Exception('Invalid response passed, we need valid requests response')

        if response.status_code != 200:
            logging.info('[_check_error] Non-200 response: %s, code %s', response.text, response.status_code)

        if response.status_code < 400 or response.status_code >= 500:
            logging.info('[_check_error] Skip error processing, status code is %s', response.status_code)
            response.raise_for_status()
            return response

        if check_non_existent and response.status_code == 404:
            return _OBJECT_NOT_FOUND

        # Here we deal with 4xx codes except 404
        self.Context.failed_keys.append(key)

        try:
            response_json = json.loads(response.text)
            status = response_json.get('status')
            logging.info('YASM response status: `%s`', status)
            if status in ['error', 'failed']:
                """
                Response example:
                {
                    "error": "ExpectedDuplicateKeys: insert duplicate key error [...]",
                    "error_code": "duplicate_keys",
                    "fqdn": "vla1-0730.search.yandex.net",
                    "status": "error"
                }
                """
                self.set_info(
                    'Problem with {method} of `{key}`: {error}'.format(
                        method=method,
                        key=key,
                        error=response_json.get('error') or response_json.get('message'),
                    )
                )
                return response

            eh.check_failed('YASM reported something strange, see logs for details. ')
        except Exception as e:
            eh.log_exception('Cannot parse YASM response', e, task=self)
            eh.check_failed('Cannot parse YASM response. {}'.format(e))

        return response

    @decorators.retries(5, delay=2, backoff=2)
    def _create_alert(self, key, content):
        """
        Create alert by given key and with given content.

        :param key: panel/alert identifier (please do not mix with alias)
        :param content: panel/alert text (contents)
        """
        create_url = 'https://yasm.yandex-team.ru/srvambry/alerts/create'
        logging.info('Trying to create alert `%s`', key)
        try:
            create_response = requests.post(
                create_url,
                params={
                    'name': key,
                },
                data=content,
            )
            return self._check_error(
                key=key,
                method=Method.CREATE,
                response=create_response,
                check_non_existent=False,
            )

        except Exception as exc:
            eh.log_exception('[_create_alert] Cannot post to yasm', exc, task=self)
            raise

    @decorators.retries(5, delay=2, backoff=2)
    def _update_alert(self, key, content):
        """
        Update alert by given key and with given content.

        :param key: panel/alert identifier (please do not mix with alias)
        :param content: panel/alert text (contents)
        """
        update_url = 'https://yasm.yandex-team.ru/srvambry/alerts/update'
        logging.info('Trying to update alert `%s`', key)
        try:
            update_response = requests.post(
                update_url,
                params={
                    'name': key,
                },
                data=content,
            )
            return self._check_error(
                key=key,
                method=Method.UPDATE,
                response=update_response,
                check_non_existent=True,
            )

        except Exception as exc:
            eh.log_exception('[_update_alert] Cannot post to yasm', exc, task=self)
            raise

    @decorators.retries(5, delay=10, backoff=2)
    def _replace_alerts(self, key, content):
        """
        Replace alerts group with given content
        """
        replace_url = 'https://yasm.yandex-team.ru/srvambry/alerts/replace/background'
        content = json.loads(content)
        if 'prefix' not in content or 'alerts' not in content:
            raise ValueError('Invalid content for {key}'.format(key=key))
        logging.info('Trying to replace alerts with prefix `%s`', content.get('prefix'))
        try:
            replace_response = requests.post(
                replace_url,
                json=content
            )
            operation_id = replace_response.json().get('response', {}).get('result', {}).get('operation_id')
            if not operation_id:
                raise FailedToRunTask('Query failed')
            return self._get_task_status(key, operation_id)
        except Exception as exc:
            eh.log_exception('[_replace_alerts] Cannot post to yasm', exc, task=self)
            raise

    @decorators.retries(10, delay=10, backoff=2)
    def _get_task_status(self, key, operation_id):
        status = None
        task_status_url = 'https://yasm.yandex-team.ru/srvambry/alerts/replace/status'
        task_status_response = requests.post(
            task_status_url,
            json={"operation_id": operation_id}
        )
        status = task_status_response.json().get('response', {}).get('result', {}).get('status')
        if not status == 'finished':
            raise TaskNotFinished('Not finished yet: {status}'.format(status=status))
        return self._check_error(
            key=key,
            method=Method.REPLACE,
            response=task_status_response,
            check_non_existent=True,
        )

    @decorators.retries(5, delay=10, backoff=2)
    def _operate_template(self, method, template_type, template_key, content):
        """
        Operate template (insert or update).

        :param method: can be one of `Method`
        :param template_type: can be one of `ObjectType`
        :param template_key: template identifier (please do not mix with alias)
        :param content: template text (contents)
        :return:
            - `None` in case of success
            - Error description (string) in case of failure
            - Exception in case of unknown error.
        """
        url = 'https://yasm.yandex-team.ru/srvambry/tmpl/{template_type}/{method}'.format(
            template_type=template_type,
            method=method,
        )
        data = {
            'key': template_key,
            'content': content,
        }
        logging.info('[_operate_template] Trying to %s template %s %s', method, template_type, template_key)
        try:
            response = requests.post(
                url,
                params={'key': template_key},
                json=data,
            )
            result = self._check_error(
                key=template_key,
                method=method,
                response=response,
                check_non_existent=True,
            )
            if result == _OBJECT_NOT_FOUND:
                return _OBJECT_NOT_FOUND

            response.raise_for_status()

            if template_type == ObjectType.ALERT:
                apply_url = 'https://yasm.yandex-team.ru/srvambry/tmpl/alerts/apply/{key}'.format(key=template_key)
                logging.info('Applying alerts from panel %s', template_key)
                try:
                    apply_response = requests.post(apply_url)
                    logging.info('apply response: %s, code %s', apply_response.text, apply_response.status_code)
                    if apply_response.status_code >= 400 and apply_response.status_code < 500:
                        response_json = json.loads(apply_response.text)
                        error_desc = response_json.get('error')
                        return 'Cannot apply alerts: {}'.format(error_desc)

                except Exception as exc:
                    eh.log_exception('[_operate_template] Cannot apply alerts', exc, task=self)

        except Exception as exc:
            eh.log_exception('[_operate_template] Cannot post to yasm', exc, task=self)
            raise
