import os
import logging

from sandbox import sdk2

from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs import svn


ATD_REVISIONS_FILE = 'build/yandex_specific/atd/revisions.txt'


class ATDRevisionsResource(sdk2.Resource):
    """ ATD revisions file """
    revision = sdk2.parameters.String("Revision")


class ATDRevisionsUpdater(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        revision = sdk2.parameters.String("Revision")
        human_readable = sdk2.parameters.Bool("Add human readable comments", default=False)
        commit = sdk2.parameters.Bool("Commit", default=True)

    def on_execute(self):
        source_root = svn.Arcadia.get_arcadia_src_dir('arcadia:/arc/trunk/arcadia')
        ya_dump = [os.path.join(source_root, 'ya'), 'dump', 'atd-revisions']

        if self.Parameters.human_readable:
            ya_dump.extend(['--human-readable'])

        revision = self.Parameters.revision
        if revision:
            ya_dump.extend(['--revision', revision])
        else:
            revision = svn.Arcadia.get_revision(source_root)

        revs_file = os.path.join(source_root, ATD_REVISIONS_FILE)

        logging.debug('Update arcadia_tests_data revisions: source root is %s, commaid is %s, revision is %s', source_root, ya_dump, revision)
        with sdk2.helpers.ProcessLog(self, logger='updater') as pl:
            with open(revs_file, 'w') as rev_txt:
                sp.check_call(
                    ya_dump,
                    stdout=rev_txt, stderr=pl.stderr, cwd=source_root,
                )

        if self.Parameters.commit:
            self._commit_revs(revs_file, revision)
        else:
            self._publish_revs(revs_file, revision)

    def _commit_revs(self, revs_file, revision):
        commit_msg = 'Update arcadia_tests_data revisions for r{}'.format(revision)
        commit_msg += '\nSKIP_CHECK'

        logging.info('Going to make commit to %s with message: %s', revs_file, commit_msg)

        attempts = 5
        for i in range(attempts):
            try:
                svn_stdout = svn.Arcadia.commit(revs_file, commit_msg, 'zomb-sandbox-rw')
                logging.info('arcadia_tests_data revisions committed: {}\nMessage: {}'.format(svn_stdout, commit_msg))
            except svn.SvnError as e:
                if i < attempts:
                    continue
                else:
                    logging.info('Failed to commit arcadia_tests_data revisions in {} attempts ({}). Message: {}'.format(attempts, e, svn_stdout))
                    raise

    def _publish_revs(self, revs_file, revision):
        resource = ATDRevisionsResource(self, 'Output file', 'revisions.txt')
        resource.revision = revision
        data = sdk2.ResourceData(resource)
        resource.path.write_bytes(open(revs_file, 'r').read())
        data.ready()
        logging.info('arcadia_tests_data revisions for r{} written to resource'.format(revision))
