from pathlib import Path

import os

import logging

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import vcs
from yalibrary.resource_autoupdater import ProjectsAutoUpdater, RecursiveAutoUpdater

from library.python.retry import retry_intrusive, RetryConf


class ArcadiaResourceAutoupdater(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 60 * 60

        arcadia_url = sdk2.parameters.ArcadiaUrl("Arcadia Url")
        commiter = sdk2.parameters.String("Commiter", default="zomb-sandbox-rw")
        rewrite: bool = sdk2.parameters.Bool("Update files and create commit", default=True)
        start_dir: str = sdk2.parameters.String("Path to search projects", default="devtools/ya")
        exclude_folders: list[str] = sdk2.parameters.List("Excluded folders", default=['devtools/dummy_arcadia'])
        diff_resolver = sdk2.parameters.Staff("Diff resolver", description='by default diff_resolver is equal to task author')
        parallel = sdk2.parameters.Integer("Parallel worker count", default=5)

        ya_secret = sdk2.parameters.YavSecret('Yav secret to update resources', default='sec-01eq833we2adjttr0ab7h0v6h5')

        use_ya_grep = sdk2.parameters.Bool("Use ya grep to search autoupdate files", default=True)

        with sdk2.parameters.Output:
            count = sdk2.parameters.Integer("Autoupdated projects")
            changed = sdk2.parameters.Integer("Count of changed files")
            files = sdk2.parameters.List("List of changed files")
            errors = sdk2.parameters.List("List of errors")
            commit_info = sdk2.parameters.String('Commit info')

        binary_release = binary_task.binary_release_parameters(stable=True)

    def _branch_name(self, revision):
        return f"DEVTOOLS-6598-resource-autoupdate-{revision}"

    def on_execute(self):
        self.retry_conf = RetryConf(logger=logging.getLogger("retry")).waiting(5, backoff=1.2).upto(30)
        arc_token = self.Parameters.ya_secret.data()['ARC_TOKEN']

        arc = vcs.arc.Arc(arc_oauth_token=arc_token)
        del arc_token

        with arc.mount_path("", changeset="trunk", fetch_all=False) as source_root:
            self.source_root = source_root
            self.ya_bin = os.path.join(self.source_root, 'ya')

            logging.debug("Source root: %s", self.source_root)

            start_dir = self.source_root
            if self.Parameters.start_dir:
                start_dir = os.path.join(start_dir, self.Parameters.start_dir)
            logging.debug("Start directory: %s", start_dir)

            full_revision = "\n".join(arc.describe(self.source_root, svn=True))
            revision = full_revision.split("-")[0].replace("r", "")
            logging.debug("Describe: %s", full_revision)
            logging.debug("svn revision: %s", revision)

            self.set_info(f"Working from revision <a href=https://a.yandex-team.ru/arc/commit/{revision}>{revision}</a>", do_escape=False)

            if self.Parameters.use_ya_grep:
                logging.info("Using ya grep to search files")
                project_paths = tuple(self._search_project_paths(start_dir))

                autoupdater = ProjectsAutoUpdater(self.source_root, project_paths, self.Parameters.exclude_folders)
            else:
                logging.info("Using recursive os.walk to search files")

                autoupdater = RecursiveAutoUpdater(start_dir, self.Parameters.exclude_folders)

            self.Parameters.count = len(autoupdater.projects)

            changed_files = autoupdater.try_update(rewrite=self.Parameters.rewrite)

            self.Parameters.changed = len(changed_files)
            self.Parameters.files = [os.path.relpath(file_path, self.source_root) for file_path in changed_files]
            self.Parameters.errors = [name for name, info in autoupdater.errors.__iter__(self.source_root)]

            infos = []
            for name, info in autoupdater.errors.__iter__(self.source_root):
                infos.append(
                    "<a href=https://a.yandex-team.ru/arc/trunk/arcadia/{name}?rev={rev}>{name}</a>: {info}".format(
                        name=name,
                        rev=revision,
                        info=info
                    ))
            self.set_info("<br>".join(infos), do_escape=False)

            if self.Parameters.rewrite:
                if len(changed_files) > 0:
                    self.Parameters.commit_info = self.do_commit(arc, revision)
                else:
                    logging.info("Nothing to commit")
            else:
                logging.info("Skip commit")

    def _search_project_paths(self, start_dir):
        from yalibrary.resource_autoupdater import UPDATE_CONFIG_FILE_NAME

        cmd = ["find", ".", "-name", UPDATE_CONFIG_FILE_NAME]

        logging.info("Search %s files", UPDATE_CONFIG_FILE_NAME)
        with sdk2.helpers.ProcessLog(self, logger="ya_grep") as pl:
            out = sdk2.helpers.subprocess.check_output(
                cmd, stderr=pl.stderr, cwd=start_dir,
            )
        logging.debug("Output is %dKb", len(out) / 1024)

        logging.debug("Output will be dumped to files.json")
        with open(self.log_path("files.json"), "wb") as f:
            f.write(out)

        lines = out.decode().split('\n')

        logging.info("Search complete: %d files was found", len(lines))

        for path in lines:  # type: str
            path = path.removeprefix("./")

            yield os.path.join(start_dir, os.path.dirname(path.strip()))

        logging.debug("Search results processing complete")

    @retry_intrusive
    def do_commit(self, arc: vcs.arc.Arc, revision):
        logging.info("Do commit from `%s`", self.Parameters.commiter)

        branch_name = self._branch_name(revision)
        arc.checkout(self.source_root, branch=branch_name, create_branch=True, force=True)

        commit_msg = ("Update resource by ArcadiaResourceAutoupdater\n"
                      "https://sandbox.yandex-team.ru/task/{task_id}/view\n"
                      "issue:DEVTOOLS-6598\n"
                      "[diff-resolver:{diff_resolver}]")
        commit_message = commit_msg.format(
            task_id=self.id,
            diff_resolver=self.Parameters.diff_resolver or self.author
        )

        arc.add(self.source_root, all_changes=True)
        arc.commit(self.source_root, "Update ya.make files", all_changes=True)
        arc.push(
            self.source_root,
            upstream="users/{}/{}".format(self.Parameters.commiter, branch_name),
            force=True,
        )

        env = {
            **os.environ,
            'YA_USER': self.Parameters.commiter,
            'YA_TOKEN': self.Parameters.ya_secret.data()['YA_TOKEN'],
        }

        logging.debug("Create pr and wait")

        review_info = self._create_pr(env, commit_message)
        if 'exc' not in review_info:
            pr_info = "Commit: <a href={reviews_url}{pr_id}>PR#{pr_id}</a>".format(**review_info)
        else:
            pr_info = "Exception while parsing pr id:<br><code>{exc}</code><br><code>{exc_msg}</code>".format(
                **review_info
            )

        self.set_info(pr_info, do_escape=False)

        commit_out = self._merge_pr(env)

        return commit_out

    @retry_intrusive
    def _merge_pr(self, env):
        logging.debug("Merge pr without checks")
        with sdk2.helpers.ProcessLog(self, logger="ya_pr_merge") as pl:
            commit_out = sdk2.helpers.subprocess.check_output(
                [self.ya_bin, "pr", "merge", "--force", "--now"],
                stderr=pl.stderr, timeout=600, cwd=self.source_root,
                env=env,
            )
        commit_out = commit_out.decode()
        logging.debug('Commit response is %s', commit_out)
        return commit_out

    @retry_intrusive
    def _create_pr(self, env: dict, commit_message: str) -> dict:
        if not hasattr(self, '_create_pr_try'):
            self._create_pr_try = 0
        self._create_pr_try += 1

        with sdk2.helpers.ProcessLog(self, logger=f"ya_pr_create_{self._create_pr_try}") as pl:
            sdk2.helpers.subprocess.check_call(
                [self.ya_bin, "pr", "create", "--force", "--wait", "--publish", "-m", commit_message],
                stdout=pl.stdout, stderr=pl.stderr, timeout=600, cwd=self.source_root,
                env=env,
            )

            pr_info = {}
            try:
                reviews_url = 'https://a.yandex-team.ru/review/'
                parts = Path(pl.stderr.path).read_text().split(reviews_url)
                if parts:
                    last_part = parts[-1]
                    pr_id = int(last_part)
                    pr_info = dict(reviews_url=reviews_url, pr_id=pr_id)
            except Exception as e:
                from traceback import format_exc
                pr_info = dict(
                    exc=e,
                    exc_msg=format_exc()
                )

            return pr_info
