# -*- coding: utf-8 -*-

import logging
import os

from sandbox import sdk2
import sandbox.sdk2.parameters as sb_parameters
import sandbox.sandboxsdk.process as process
import sandbox.sandboxsdk.errors as errors
import sandbox.projects.common.arcadia.sdk as arcadia_sdk


from sandbox.projects.devtools.BalancingConfigGenerator.resources import AutocheckBalancingYaMakeParseResult, AutocheckBalancingConfig

YA_MAKE_PARSE_MODULE_PATH = "devtools/autocheck/partition_metod/ya_make_parser/ya_make_parser"
PARTITION_GENERATOR_MODULE_PATH = "devtools/autocheck/partition_metod/partition_generator/partition_generator"


class BalancingConfigGenerator(sdk2.Task):
    """
        Таск для генерации балансировочных конфигов автосборки
    """

    class Parameters(sdk2.Parameters):
        ArcadiaUrl = sb_parameters.ArcadiaUrl()
        YaMakeParseArgsParam = sb_parameters.String("Args to ya.make parser", default="")
        BalancingGeneratorArgsParam = sb_parameters.String("Args to balance generator", default="")
        ParseResourceId = sb_parameters.Integer("Id of resource which contains root:uids dict", default=None)
        MaxUidsDiff = sb_parameters.Integer("Set max diff by uids for partition", default=0)

    class Requirements(sdk2.Requirements):
        cores = 17

    def get_parse_resource_path(self):
        if self.Parameters.ParseResourceId is None:
            arcadia_dir = sdk2.vcs.svn.Arcadia.get_arcadia_src_dir(self.Parameters.ArcadiaUrl)

            resource = AutocheckBalancingYaMakeParseResult(self, "Output parse files", "output_parse_dir")
            data = sdk2.ResourceData(resource)
            data.path.mkdir(parents=True, exist_ok=True)

            build_target = os.path.dirname(YA_MAKE_PARSE_MODULE_PATH)

            logging.info("Target to build %s", build_target)
            build_return_code = arcadia_sdk.do_build(
                "ya", arcadia_dir, [build_target], clear_build=False, results_dir=arcadia_dir,
            )

            logging.info("Build returned %s", build_return_code)

            process.run_process(
                [os.path.join(arcadia_dir, YA_MAKE_PARSE_MODULE_PATH), ] + self.Parameters.YaMakeParseArgsParam.split(),
                log_prefix="program", outputs_to_one_file=False, shell=True, work_dir=str(data.path.absolute()),
            )

            data.ready()
        else:
            resource = sdk2.Resource.find(id=self.Parameters.ParseResourceId).first()
            data = sdk2.ResourceData(resource)

        return str(data.path.absolute())

    def on_execute(self):
        self.set_info("This task is DEPRECATED. Please use BALANCING_CONFIG_GENERATOR_3")

        arcadia_dir = sdk2.vcs.svn.Arcadia.get_arcadia_src_dir(self.Parameters.ArcadiaUrl)
        logging.debug("Arcadia src dir %s", arcadia_dir)
        os.environ["ARCADIA_ROOT"] = arcadia_dir

        parse_resource_path = self.get_parse_resource_path()

        resource = AutocheckBalancingConfig(self, "Output balance files", "output_balance_dir")
        data = sdk2.ResourceData(resource)
        data.path.mkdir(parents=True, exist_ok=True)

        build_target = os.path.dirname(PARTITION_GENERATOR_MODULE_PATH)

        logging.info("Target to build %s", build_target)
        build_return_code = arcadia_sdk.do_build(
            "ya", arcadia_dir, [build_target], clear_build=False, results_dir=arcadia_dir,
        )

        logging.info("Build returned %s", build_return_code)

        p = process.run_process(
            [os.path.join(arcadia_dir, PARTITION_GENERATOR_MODULE_PATH), ] +
            self.Parameters.BalancingGeneratorArgsParam.split() +
            ["--unique-uids-limit", self.Parameters.MaxUidsDiff, "--file-input", os.path.join(parse_resource_path, "data.json")],
            log_prefix="program", outputs_to_one_file=False, shell=True, work_dir=str(data.path.absolute()),
        )

        if p.returncode != 0:
            if p.returncode == 1:
                raise errors.SandboxTaskFailureError("Too many unique uids in partition")
            raise errors.SandboxTaskFailureError("partition_generator exited with code {}".format(p.returncode))

        data.ready()
